/*
**************************************************************************
This Source Code Form is subject to the terms of the Mozilla Public
License, v. 2.0. If a copy of the MPL was not distributed with this file,
You can obtain one at http://mozilla.org/MPL/2.0/.

Copyright (c) 2010-2016, Marvell International Ltd.

Alternatively, this software may be distributed under the terms of the GNU
General Public License Version 2, and any use shall comply with the terms and
conditions of the GPL.  A copy of the GPL is available at
http://www.gnu.org/licenses/old-licenses/gpl-2.0.html

THE FILE IS DISTRIBUTED AS-IS, WITHOUT WARRANTY OF ANY KIND, AND THE
IMPLIED WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE
ARE EXPRESSLY DISCLAIMED.  The GPL license provides additional details about
this warranty disclaimer.
******************************************************************************
*/


/**
 * \file picdma_descrip.c
 *
 * \brief PIC Write DMA driver for chained descriptor based DMA hardware
 *
 * davep 28-Mar-2010 ; Added new descriptor based code
 *
 */

#include <stdint.h>
#include <stdbool.h>
#include <string.h>

#include "scos.h"

#include "regAddrs.h"
#include "debug.h"
#include "lassert.h"
#include "cpu_api.h"
#include "memAPI.h"
#include "list.h"
#include "interrupt_api.h"

#include "scantypes.h"
#include "scancore.h"
#include "scandbg.h"
#include "safetylock.h"
#include "icedma.h"
#include "ddma.h"

#include "pic_handle_if.h"
#include "pic_if.h"
#include "pic_convenience_if.h"

#include "pic.h"
#include "picdma_descrip.h"
#include "scanalyzer.h"

#include "dros.h"

//#define PIC_WDMA_DEBUG
//#define PIC_WDMA_ISR_DEBUG

#ifdef PIC_WDMA_ISR_DEBUG
#define isr_dbg2 dbg2
#else
#define isr_dbg2(...)
#endif

//#define MAX_CWDMA_DESCRIPTORS 512
//#define MAX_CWDMA_DESCRIPTORS 128
//#define MAX_CWDMA_DESCRIPTORS 32
#define MAX_CWDMA_DESCRIPTORS 64
//#define MAX_CWDMA_DESCRIPTORS 16

struct pic_wdma_interrupt_stats {
    uint32_t count;
    uint32_t transfer_end; 
    uint32_t end_of_image;
    uint32_t reset; 

    /* increment counter when we find wdma block idle during channel ISR */
    uint32_t is_idle;
};

// semaphore
dros_sem_t icedma_softreset_sem;

static struct pic_wdma_interrupt_stats int_stats[PIC_WDMA_NUM_CHANNELS];

static struct ddma_channel desc_channel_list[PIC_WDMA_NUM_CHANNELS];

static struct ice_dma_driver pic_cwdma_driver = { 
    .msg_data  = SMSG_PIC_WDMA_DATA,

    /* set the queue depth to a number which, I hope, will keep our chained
     * descriptors far enough ahead of the running DMA
     */
    .max_queue_depth = MAX_CWDMA_DESCRIPTORS-1,
//    .max_queue_depth = 10,

    .reset =      pic_wdma_channel_reset,
    .enable =     pic_wdma_channel_enable,
    .disable =    pic_wdma_channel_disable,
    .load =       pic_wdma_channel_load, 
    .start =      pic_wdma_channel_start,   /* start; can't use because called for each load */
    .is_enabled = pic_wdma_channel_is_enabled,
    .icebuf_isr = NULL,
};

static struct ice_dma_mm pic_cwdma_mm;
static struct ice_dma_channel pic_cwdma_channels[PIC_WDMA_NUM_CHANNELS]; // FIXME - WHY HARDCODE PIC_WDMA_NUM_CHANNELS - should be set on a per pic basis, and query pic to calculate number of channels - then we can get rid of the silly convert function


void pic_cwdma_interrupt_stats_reset( void )
{
    /* Note: neither thread safe nor interrupt safe */
    memset( int_stats, 0, sizeof(int_stats) );
}

void pic_cwdma_interrupt_stats_dump( void )
{
    int channel_idx;

    /* Note: neither thread safe nor interrupt safe */

    for( channel_idx=0 ; channel_idx<PIC_WDMA_NUM_CHANNELS ; channel_idx++ ) {
        dbg2( "[%d] total=%d FIN=%d EOI=%d RST=%d idle=%d\n", channel_idx,
                    int_stats[channel_idx].count,
                    int_stats[channel_idx].transfer_end,
                    int_stats[channel_idx].end_of_image,
                    int_stats[channel_idx].reset,
                    int_stats[channel_idx].is_idle );
    }
}

// FIXME, PIC_WDMA_NUM_CHANNELS should be read from the ASIC
// and so should PIC_WDMA_NUM_CHAN_PER_SENSOR
int convert_pic_instance_channum_to_channel_idx(uint8_t pic_instance, int channum)
{
    switch(channum)
    {
    case 0:
        // mono or color 0
        if (pic_instance == 0)
            return PIC_DMA_CHANNEL_CEVEN_0; // front side
        else if (pic_instance == 1)
            return PIC_DMA_CHANNEL_CODD_0; // back side
        errprint("%s: ERROR!! invalid pic instance %c\n", __func__, pic_instance);
        XASSERT(0, channum);
        break;
    case 1:
        // color 1
        if (pic_instance == 0)
            return PIC_DMA_CHANNEL_CEVEN_1; // front side
        else if (pic_instance == 1)
            return PIC_DMA_CHANNEL_CODD_1; // back side
        errprint("%s: ERROR!! invalid pic instance %c\n", __func__, pic_instance);
        XASSERT(0, channum);
        break;
    case 2:
        // color 2
        if (pic_instance == 0)
            return PIC_DMA_CHANNEL_CEVEN_2; // front side
        else
            return PIC_DMA_CHANNEL_CODD_2; // back side
        errprint("%s: ERROR!! invalid pic instance %d\n", __func__, pic_instance);
        XASSERT(0, channum);
        break;
    default:
        errprint("%s: ERROR!! invalid channum %d\n", __func__, channum);
        XASSERT(0, channum);
    }
    return 0; // to keep compiler happy - should never reach here
}

/**
 * \brief Enable ODMA line_rev (line reverse) mode
 *
 * In line_rev mode, DMA is right-to-left instead of left-to-right.
 *
 * We use this to flip data from systems where paper is backwards. For example,
 * on certain ADFs, paper is pulled across the sensor backwards giving an image
 * mirrored about the Y axis. line_rev will flip image for us.
 *
 */

void pic_wdma_enable_line_reverse_by_channel(struct pic_handle_t *pic_handle, uint8_t channel, bool enable)
{
    uint8_t pic_instance, channum;

#ifdef PIC_WDMA_DEBUG
    dbg2("%s enable=%d\n", __FUNCTION__, enable);
#endif

    XASSERT(channel < PIC_WDMA_NUM_CHANNELS, channel) ;
    
    convert_channel_idx_to_pic_instance_channum(channel, &pic_instance, &channum);
    if (enable)
    {
        pic_output_dma_channel_enable_line_reverse(pic_handle, PIC_WDMA_CFG_LINE_REV, channum);
    }
    else
    {
        pic_output_dma_channel_enable_line_reverse(pic_handle, PIC_WDMA_CFG_LINE_NORM, channum);
    }
}

void pic_wdma_enable_line_reverse(uint8_t pic_instance, struct pic_handle_t *pic_handle, bool enable)
{
    int i;

#ifdef PIC_WDMA_DEBUG
    dbg2( "%s enable=%d\n", __FUNCTION__, enable );
#endif

    for (i=0 ; i<PIC_WDMA_NUM_CHANNELS ; i++)
    {
        uint8_t channel_pic_instance, channum;
        convert_channel_idx_to_pic_instance_channum(i, &channel_pic_instance, &channum);
        if (channel_pic_instance == pic_instance)
        {
            pic_wdma_enable_line_reverse_by_channel(pic_handle, i, enable);
        }
    }
}



void pic_wdma_channel_dump(uint8_t channel_idx)
{
    uint8_t channum, pic_instance;
    struct ddma_channel *dch;
    
    XASSERT(channel_idx < PIC_WDMA_NUM_CHANNELS, channel_idx);
    convert_channel_idx_to_pic_instance_channum(channel_idx, &pic_instance, &channum);
    pic_dump(0);

    dch = &desc_channel_list[channel_idx];

    if(dch->is_open)
    {
        ddma_channel_sanity(dch);
        // ddma_descriptor_list_dump( &dch->desc_list);
    }
}


void pic_wdma_channel_start( uint8_t channel )
{
    /* no op; see also pic_cwdma_late_launch() */
}

/**
 * \brief brute force jam a descriptor into a channel
 *
 * Created for some test code. Might be useful to this driver itself someday?
 *
 * \author David Poole
 * \date 29-Mar-2011
 */

void pic_ddma_desc_write(struct pic_handle_t *pic_handle, uint8_t channel, struct ddma_descriptor *desc, uint32_t line_width_bytes)
{
    uint8_t pic_instance, channum;
//SANDRA FIXME    bool enabled;

    XASSERT( channel < PIC_WDMA_NUM_CHANNELS, channel );

    convert_channel_idx_to_pic_instance_channum(channel, &pic_instance, &channum);

    /* fire in the hole! */
    pic_output_dma_channel_set_linesize(pic_handle, line_width_bytes, channum);

#if 0
    /* is this thing on? */
    enabled = pic_output_dma_channel_is_enabled(pic_handle, channum);
    ASSERT(enabled == true);
    // Who is calling this?  Do we have a current pic_handle so we need a convenience function?
#endif    

    // XASSERT( DESCRIPTOR_OWNED_BY_BLOCK(desc), desc->config_flags );

    pic_do_configure(pic_handle, pic_instance);

    ddma_desc_flush(desc);

    /* davep 25-Apr-2012 ; adding vma/pma destinction */
    XASSERT( desc->dma_ptr_self, (uint32_t)desc );
    ASSERT( desc->dma_ptr_self );
    ASSERT( desc->src_addr );

    pic_start_output_dma(pic_instance, channum, desc->dma_ptr_self);
}

static void channel_late_launch( struct ddma_channel *dch )
{
    uint8_t pic_instance, channum;
    //struct pic_wdma_status_info wdma_status;

#ifdef PIC_WDMA_DEBUG
    if( !IN_INTERRUPT() ) {
        dbg2("%s ch=%d running=%d\n", __FUNCTION__, dch->channel,
                dch->num_running );
    }
#endif
    convert_channel_idx_to_pic_instance_channum(dch->channel, &pic_instance, &channum);

    /* davep 13-Oct-2010 ; oy vey
     *
     * The new descriptor DMAs behave different than the old register DMAs. The
     * previous register DMAs (e.g., picdma2005.c) required a load-start (push
     * dest addr then push start) for each buffer.  
     *
     * The new descriptor DMAs use a chain of data structures in main memory.
     * A load pushes the start of the chain, a start starts the chain. However,
     * our icedma still wants a load-start for each buffer. (See
     * channel_launch()-icedma.c)
     * 
     * The descriptor DMAs will use the load to add the buffer to the end of
     * the descriptor chain. The 'start' will do nothing. This 'late launch'
     * will be called by scan_cmdq_launch(). The scan_cmdq_launch() function is
     * called once all buffers are loaded, cmds are queued into the scandmq.
     * scan_cmdq_launch() calling pic_cwdma_late_launch() is a kludge but a
     * brilliant kludge suggested by BurtP.
     */

#if 0
    //FIXME dzh 01-07-15 we need pic_handle_t to get config
    //pic_output_dma_channel_get_config(handle, &pic_wdma_cfg, channum);
    //XASSERT((pic_wdma_cfg.enable_dma != 0), pic_wdma_cfg.enable_dma);
#endif

    if (pic_output_dma_channel_busy(pic_instance, channum))
    {
//#ifdef PIC_WDMA_DEBUG
//        if( !IN_INTERRUPT() ) {
//            dbg2("%s ch=%d already running\n", __FUNCTION__, channel );
//        }
//#endif

        /* DMA already running, no need to start it again */
        return;
    }

    XASSERT( dch->num_running > 0, dch->num_running );
    XASSERT( dch->head_desc != NULL, dch->channel );

    XASSERT( DESCRIPTOR_OWNED_BY_BLOCK(dch->head_desc), dch->head_desc->config_flags );

    // fire in the hole! 
    // adding vma/pma destinction */
    XASSERT( dch->head_desc->dma_ptr_self, (uint32_t)dch->head_desc );
    pic_start_output_dma(pic_instance, channum, dch->head_desc->dma_ptr_self);
    
#ifdef PIC_WDMA_DEBUG
//    if( 1 ) {
    if(!IN_INTERRUPT())
    {
        dbg2( "%s %p\n", __FUNCTION__, dch->head_desc );
        pic_wdma_channel_dump(dch->channel); 
    }
#endif
}

void pic_cwdma_late_launch()
{
    int channel_idx;
    struct ddma_channel *dch;

    // FIXME, need to get pic_instance and number of channels dynamically
    // and should this be passed in to the function instead of done here?
    // Or, is the knowledge of this kept in the desc_channel_list, and the
    // is_open selects it for us?
    for (channel_idx=0 ; channel_idx<PIC_WDMA_NUM_CHANNELS ; channel_idx++)
    {
        dch = &desc_channel_list[channel_idx];
        if( !dch->is_open ) {
            continue;
        }
        channel_late_launch( dch );
    }
}

void pic_wdma_channel_load( uint8_t channel, uint8_t *data_ptr, dma_addr_t dma_dest,
                            uint32_t num_rows, uint32_t bytes_per_row )
{
    scan_err_t scerr;
    struct ddma_channel *dch;
    struct ddma_descriptor *next;
    dma_addr_t dma_handle;
    uint8_t pic_instance, channum;
//    struct pic_wdma_status_info wdma_status;
//    uint32_t line_size;
    bool first_entry;

    /* 
     * Warning! Can be called from interrupt context! 
     */

#ifdef PIC_WDMA_DEBUG
    if( !IN_INTERRUPT() ) {
        dbg2("%s ch=%d 0x%x %d %d\n", __FUNCTION__,
                    channel, dma_dest, num_rows, bytes_per_row );
    }
#endif
    dch = &desc_channel_list[channel];
    XASSERT( dch->channel==channel, dch->channel );

    XASSERT( ICE_DMA_IS_ALIGNED((uint32_t)dma_dest), (uint32_t)dma_dest );
    XASSERT( ICE_DMA_IS_ALIGNED(bytes_per_row), bytes_per_row );

    /* davep 05-Jun-2012 ; add channel data structure protection */
    scerr = ddma_channel_try_lock(dch);
    XASSERT( scerr==SCANERR_NONE, scerr );

    /* poke the bytes_per_row into the line_width (should be the same for all
     * load calls for this scan but I won't know the value until load is
     * called)
     */
    /* davep 22-Jun-2012 ; this scares me; troubleshooting this crazy PIC
     * overflow and reading code, trying to find anything hinky. Let's not
     * touch the DMA hardware while it's running.
     */
    convert_channel_idx_to_pic_instance_channum(channel, &pic_instance, &channum);

    // pic_output_dma_channel_set_linesize(pic_handle, bytes_per_row, channum);
#if 0
    // The reg values won't be set until we call pic_do_configure.
    pic_get_linesize_output_dma_channel(pic_instance, &line_size, channum);
    
    if (line_size==0)
    {
        pic_get_output_dma_channel_status(pic_instance, &wdma_status, channum);
        XASSERT((wdma_status.dma_busy == 0), wdma_status.debug_array);
        pic_set_linesize_output_dma_channel(pic_instance, bytes_per_row, channum);
    }
    else
    { 
        XASSERT(line_size==bytes_per_row, bytes_per_row);
    }
#endif

    /* davep 04-Oct-2010 ; can hit this assert if we're trying to dma without
     * allocating the descriptors via pic_wdma_open() (e.g., scantest.c)
     */
    XASSERT(dch->desc_list.num_descriptors>0, dch->channel);
    XASSERT(dch->desc_list.list, dch->channel);

    XASSERT(dch->num_running < dch->desc_list.num_descriptors, dch->num_running);

    /* make sure the block hasn't been here yet */
    XASSERT(DESCRIPTOR_OWNED_BY_BLOCK(dch->tail_desc), dch->tail_desc->config_flags);
    first_entry = false;
    /* set up tail->next transaction */
    if (dch->tail_desc->src_addr==0)
    {
        /* Empty list. Note head==tail in a list with one element AND an empty
         * list thus the check on src_addr 
         */
        next = dch->head_desc;
        first_entry = true;
    }
    else
    {
        next = (struct ddma_descriptor *)(dch->tail_desc->fw_next_descriptor_addr);
    }

    /* beware overrunning existing members of the list (src_addr set to zero in
     * ISR when the descriptor completes)
     */
    XASSERT(next->src_addr==0 || next->src_addr==0x42424242, next->src_addr);
    // XASSERT( next->src_addr==SRC_ADDR_EMPTY, next->src_addr );

    dma_handle = (dma_addr_t)dma_dest;

    next->src_addr = (uint32_t)dma_handle;
    next->fw_src_addr = (void *)dma_dest;
    next->transfer_len_bytes = num_rows * bytes_per_row;

    /* burtp 28-Sep-2010: FIXME! The scan block defines each scan command as being an 'image',
     *    problem is the new DMA hangs if EOI is not set when the scan image ends. The old
     *    DMA didn't care. This implies that our scan commands must always line up
     *    perfectly with the queued DMA buffers.
     */
    /* davep 22-Mar-2011 ; work around a discrepency between dma blocks: some
     * complain if EOI set (PIC overflow). Some stall (scan doesn't start) if EOI
     * isn't set. Once the "bound" versions fall out of use, we can get rid of
     * this EOI code.
     */
//#ifdef HAVE_SCMD_PICWDMA_BOUND
//    next->config_flags |= DDMA_DESCRIPTOR_CONFIG_EOI|DDMA_DESCRIPTOR_CONFIG_STOP
//                          |DDMA_DESCRIPTOR_CONFIG_OWNER_BLOCK;
//#else
    next->config_flags |= DDMA_DESCRIPTOR_CONFIG_STOP|DDMA_DESCRIPTOR_CONFIG_OWNER_BLOCK;
//#endif
    /* flush the descriptor from the cache to main mem where dma can find it */
    ddma_desc_flush(next);

    // are we the only entry?  If so, leave the stop there
    // otherwise, now that we are not the last one, set the previous one not to stop
    if (!first_entry)
    {
//    dch->tail_desc->config_flags &= ~(DDMA_DESCRIPTOR_CONFIG_STOP | DDMA_DESCRIPTOR_CONFIG_EOI);
        dch->tail_desc->config_flags &= ~(DDMA_DESCRIPTOR_CONFIG_STOP);
    }
    /* flush the descriptor from the cache to main mem where dma can find it */
    ddma_desc_flush(dch->tail_desc);

    /* move to next */
    dch->tail_desc = next;
    ddma_desc_flush(dch->tail_desc);

    dch->num_running += 1;

#ifdef PIC_WDMA_DEBUG
    if( !IN_INTERRUPT() ) {
        ddma_descriptor_list_dump( &dch->desc_list);
        pic_wdma_channel_dump(channel);
     }
#endif

    // ddma_channel_sanity(dch);
    ddma_channel_unlock(dch);
}

void pic_wdma_channel_enable(uint8_t channel)
{
#if 0
    // dzh - 01-08-15 no pic_handle. Can't enable.
    uint8_t pic_instance, channum;
    
    convert_channel_idx_to_pic_instance_channum(channel, &pic_instance, &channum);

    // NOTE, this is setting the pie handle, not the ASIC
    pic_output_dma_channel_set_enable_dma(pic_handle, PIC_WDMA_CFG_ENABLE, channum);
    // enable all interrupts for the channel (all == NULL)
    pic_enable_irq_output_dma_channel(pic_instance, NULL, channum);
#endif
}

bool pic_wdma_channel_is_enabled(uint8_t channel)
{
    uint8_t pic_instance, channum;

    convert_channel_idx_to_pic_instance_channum(channel, &pic_instance, &channum);

    return pic_output_dma_channel_is_enabled(pic_instance, channum);
}

void pic_wdma_channel_disable(uint8_t channel)
{
    uint8_t pic_instance, channum;

    convert_channel_idx_to_pic_instance_channum(channel, &pic_instance, &channum);

#if 0
    // dzh - 01-08-15 In the new simplified API, this call has no immediate effect.
    pic_output_dma_channel_set_enable_dma(pic_handle, PIC_WDMA_CFG_DISABLE, channum);
#endif

    // disable all interrupts for the channel (all == NULL)
    pic_disable_pic_output_dma_channel_irqs(pic_instance, NULL, channum);    
}

void pic_wdma_channel_status(struct pic_handle_t *pic_handle, uint8_t channel, uint32_t *cnt_status, uint32_t *addr_status)
{
    uint8_t pic_instance, channum;

    convert_channel_idx_to_pic_instance_channum(channel, &pic_instance, &channum);
    pic_output_dma_channel_get_trans_addr(pic_handle, addr_status, channum);
    pic_output_dma_channel_get_trans_len(pic_handle, cnt_status, channum);
}

// to make sure a semaphore is empty, do trywaits until
// we get a DROSERR
static void pic_clear_sem(dros_sem_t *pic_sem)
{
    dros_err_t dros_ret = DROSERR_NONE;
    
    while(DROSERR_NONE == dros_ret)
    {
        dros_ret = dros_sem_trywait(pic_sem);
    }
}

void pic_wdma_channel_reset(uint8_t channel)
{
    uint32_t sanity;
    uint8_t pic_instance, channum;
    dros_err_t dros_err;
    // FIXME, WHAT do we want this function to do?  or should we remove it
    // the the soft reset takes care of this????
    convert_channel_idx_to_pic_instance_channum(channel, &pic_instance, &channum);

    // dzh 01-21-15 do a reset only if the channel is open
    if (!desc_channel_list[channel].is_open) return;

    /* disable all output dma interrupts */
    pic_disable_pic_output_dma_channel_irqs(pic_instance, NULL, channum);    

    /* clear any pending output dma interrupts */
    pic_do_clear_all_irqs(pic_instance);

    /* davep 13-Jan-2011 ; channel must be enabled for the reset to lead to a
     * Soft Reset Complete interrupt 
     */

    //dzh - 01-08-15
    // pic_wdma_channel_enable(channel);

    pic_cwdma_interrupt_stats_reset(); // make sure soft reset interrupt count is 0
    // make sure semaphore has no spurious interrupt posts
    pic_clear_sem(&icedma_softreset_sem);
    
    // Enable the common pic interrupts and pic dma interrupts, we need
    // the reset complete IRQ to fire (or we will get stuck)!
    pic_enable_pic_common_irqs(pic_instance, NULL);
    pic_enable_pic_output_dma_channel_irqs(pic_instance, NULL, channum);

    pic_odma_soft_reset(pic_instance); // reset pic odma
    
    sanity = 0;
    // Wait for Soft Reset complete interrupt

    dros_err = dros_sem_wait(&icedma_softreset_sem); // FIXME - do we want a timed wait?
    if (DROSERR_NONE != dros_err)
    {
        errprint("ERROR! waiting on semaphore icedma_softreset_sem returned error=%d\n", dros_err);
        ASSERT(0);
    }
    if (int_stats[channel].reset == 0)
    {
        errprint("ERROR the callback didn't seem to run\n");
        ASSERT(0);
    }

    pic_disable_pic_common_irqs(pic_instance, NULL);    
    // pic_wdma_channel_dump(channel);

}

void pic_wdma_init_onetime( void ) 
{
    dros_err_t dros_err;
    
    ice_dma_init( &pic_cwdma_mm, "piccwdma", &pic_cwdma_driver, 
            pic_cwdma_channels, PIC_WDMA_NUM_CHANNELS );
    
    dros_err = dros_sem_init(&icedma_softreset_sem, "icedma_softreset", 0);
    if (dros_err != DROSERR_NONE)
    {
        errprint("uh oh, init semaphore icedma_softreset_sem returned error=%d\n", dros_err);
        ASSERT(0);
    }
    
}

scan_err_t pic_cwdma_channel_open( uint8_t channel )
{
    scan_err_t scerr;
    struct ddma_channel *dch;
    char name[DDMA_NAME_LEN+1];

    XASSERT( channel < PIC_WDMA_NUM_CHANNELS, channel );

    dch = &desc_channel_list[channel];

    strncpy( name, "picwdma ", DDMA_NAME_LEN );
    name[7] = '0' + channel;
    ddma_channel_open( dch, name, channel );

    scerr = ddma_channel_alloc_descriptors( dch, MAX_CWDMA_DESCRIPTORS );
    if( scerr != 0 ){
        /* failure! oh woe! free the channel memory we've already allocated */
        ddma_channel_free_descriptors( dch );
        ddma_channel_close( dch );

        return SCANERR_OUT_OF_MEMORY;
    }

    /* davep 21-Feb-2012 ; suggestion from PaulZ for my line_rev problem */
    dch->head_desc->config_flags |= DDMA_DESCRIPTOR_CONFIG_SOI;
    ddma_desc_flush( dch->head_desc );

    return SCANERR_NONE;
}

void pic_cwdma_channel_close( uint8_t channel )
{
    struct ddma_channel *dch;

    XASSERT( channel < PIC_WDMA_NUM_CHANNELS, channel );

    dch = &desc_channel_list[channel];

    XASSERT( dch->channel==channel, dch->channel );
    XASSERT( dch->num_running==0, dch->num_running );

    ddma_channel_free_descriptors( dch );
    ddma_channel_close( dch );
}

/**
 * \brief  
 *
 * Created to assign a scan_data_type to an ice_dma_channel during channel open
 *
 * \author David Poole
 * \date 09-Apr-2013
 */

static scan_data_type pic_channel_to_dtype( uint8_t channel )
{
    int i;
    static const struct {
        uint8_t channel;
        scan_data_type dtype;
    } channel_to_dtype[] = {
        { PIC_DMA_CHANNEL_CEVEN_0, SCAN_DATA_TYPE_RED },
        { PIC_DMA_CHANNEL_CODD_0,  SCAN_DATA_TYPE_RED },
        { PIC_DMA_CHANNEL_CEVEN_1, SCAN_DATA_TYPE_GREEN },
        { PIC_DMA_CHANNEL_CODD_1,  SCAN_DATA_TYPE_GREEN },
        { PIC_DMA_CHANNEL_CEVEN_2, SCAN_DATA_TYPE_BLUE },
        { PIC_DMA_CHANNEL_CODD_2,  SCAN_DATA_TYPE_BLUE },
        
        /* end of list marker */
        { 0, SCAN_DATA_NULL_TYPE },
    };

    /* linear search! hurray! */
    for( i=0 ; ; i++ ) {
        if( channel_to_dtype[i].dtype==SCAN_DATA_NULL_TYPE ) {
            XASSERT( 0, channel );
            return SCAN_DATA_NULL_TYPE;
        }

        if( channel_to_dtype[i].channel==channel ) {
            break;
        }
    }
    return channel_to_dtype[i].dtype;
}

/**
 * \brief  Assign a scan_data_type to all channels' ice_dma_channel.
 *
 * The ice_dma_channel's dtype is assigned to incoming ice_dma_buffers.
 *
 * \author David Poole
 * \date 09-Apr-2013
 */

static void pic_wdma_assign_channels_dtype( struct ice_dma_mm *mm, 
                                            uint8_t channels[],
                                            uint8_t num_channels )
{
    int i;
    struct ice_dma_channel *ch;

    for( i=0 ; i<num_channels ; i++ ) {
        ch = &mm->channels[channels[i]];
        XASSERT( ch->is_open, ch->channel );

        if( num_channels<=2 ) {
            ch->dtype = SCAN_DATA_TYPE_MONO;
        }
        else {
            ch->dtype = pic_channel_to_dtype( ch->channel );
        }

        /* make sure we found a match */
        XASSERT( ch->dtype!=SCAN_DATA_NULL_TYPE, ch->dtype );
    }
}

/**
 * \brief  Assign a sensor_num to all ice_dma_channels in the mm
 *
 * In order to support platforms with multiple image sensors (e.g., duplex
 * scanning), we need to track which DMA channel belongs to which sensor.
 *
 * Original reason for this code is to preserve the page side because, as of
 * this writing, PIC has six WDMA channels but PIE only has three channels. In
 * order to pass buffers back/forth, the PIE RDMA channels must be remapped
 * back to the correct PIC WDMA channel.
 *
 * \author David Poole
 * \date 23-May-2013
 */

static void pic_wdma_assign_channels_sensor_num( struct ice_dma_mm *mm, 
                                            uint8_t channels[],
                                            uint8_t num_channels )
{
    int i;
    struct ice_dma_channel *ch;

    /* davep 23-May-2013 ; as of this writing, the only active ASIC with six
     * PIC WDMA channels has assigned channels [024] to sensor #0 and [135] to
     * sensor #1. Older ASICs supported six WDMA channels for staggered sensors
     * but that support no longer exists in any chip.
     *
     * If newer ASICs arrive with weirder PIC WDMA support, we'll have to
     * revisit this code.
     */

    for( i=0 ; i<num_channels ; i++ ) {
        ch = &mm->channels[channels[i]];
        XASSERT( ch->is_open, ch->channel );

#ifdef HAVE_NSENSOR_SUPPORT 
        /* even channels are sensor #0, odd channels are sensor #1 */
        ch->sensor_num = ch->channel & 1;
#else
        ch->sensor_num = 0;
#endif
    }
}

int pic_wdma_open( struct pic_handle_t *pic_handle,
		   uint8_t channels[],
                   uint8_t num_channels,
                   int num_empties,
                   int total_rows,
                   int bytes_per_row )
{
    int retcode;
    scan_err_t scerr;
    int i;
    uint8_t pic_instance, channum;

    XASSERT( num_channels <= PIC_WDMA_NUM_CHANNELS, num_channels );
    XASSERT( pic_handle != NULL, (uint32_t)pic_handle );

    /* start our interrupt counters from zero (useful for debugging) */
    pic_cwdma_interrupt_stats_reset();
    for (i=0 ; i<num_channels; i++)
    {
        scerr = pic_cwdma_channel_open( channels[i] );
        if( scerr != 0 ){
            /* failure! oh woe! free the channel memory we've already allocated */
            i -= 1;
            while( i>=0 ) {
                pic_cwdma_channel_close( channels[i] );
                i -= 1;
            }

            /* this function returns the number of icebufs allocated */
            return 0;
        }

        /* davep 02-Apr-2013 ; use combined interrupts to reduce ISR calls */
        convert_channel_idx_to_pic_instance_channum(channels[i], &pic_instance, &channum);
        pic_output_dma_set_common_int(pic_handle, 1<<channum, true); // add (OR) in this bit
    }
    retcode = ice_dma_open( &pic_cwdma_mm, channels,
                    num_channels, num_empties,
                    total_rows, bytes_per_row, ICEBUF_TAG_PICWDMA );
    if( retcode <= 0 ) {
        /* error */
        return retcode;
    }

    /* davep 08-Apr-2013 ; assign scan_data_type to channels by
     * inferring from the number of channels
     */
    pic_wdma_assign_channels_dtype( &pic_cwdma_mm, channels, num_channels );

    /* davep 23-May-2013 ; adding sensor_num to ice_dma_channel to better
     * support multiple sensors
     */
    pic_wdma_assign_channels_sensor_num( &pic_cwdma_mm, channels, num_channels );

    return retcode;
}

scan_err_t pic_wdma_add_buffer( uint8_t channel, uint8_t *data, uint32_t datalen,
                     uint32_t rows, uint32_t bytes_per_row )
{
    return ice_dma_add_buffer( &pic_cwdma_mm, 
            channel, data, datalen, rows, bytes_per_row );
}

void pic_wdma_add_ready( uint8_t channel, struct ice_dma_buffer **addme )
{
    /* mark it empty and ready to be filled */
    (*addme)->num_rows = 0;

    ice_dma_add_ready( &pic_cwdma_mm, channel, addme );
}

void pic_wdma_free_empty( uint8_t channel, struct ice_dma_buffer **freeme )
{
    ice_dma_free_empty( &pic_cwdma_mm, channel, freeme );
}

void pic_wdma_channel_launch( uint8_t channel )
{
    ice_dma_channel_launch( &pic_cwdma_mm, channel );
}

void pic_wdma_cancel( void )
{
    int channel_idx;
    struct ddma_channel *dch;

    for( channel_idx=0 ; channel_idx<PIC_WDMA_NUM_CHANNELS ; channel_idx++ ) {
        dch = &desc_channel_list[channel_idx];
        if( !dch->is_open ) {
            continue;
        }

        /* Have to reset the channel BEFORE we disable it! Otherwise we'll get
         * stuck in channel_reset() because the channel never goes back to
         * ready.
         */
        pic_wdma_channel_reset(channel_idx);

        dch->num_running = 0;
    }

    /* now call the parent method */
    ice_dma_cancel( &pic_cwdma_mm );
}

void pic_wdma_sanity( void )
{
    int channel_idx;
    struct ddma_channel *dch;

    for( channel_idx=0 ; channel_idx<PIC_WDMA_NUM_CHANNELS ; channel_idx++ ) {
        dch = &desc_channel_list[channel_idx];

        if( !dch->is_open ) {
            continue;
        }

        ddma_channel_sanity( dch );
    }

    ice_dma_sanity( &pic_cwdma_mm );
}

void pic_wdma_debug_log( void )
{
    int channel_idx;
    struct ddma_channel *dch;

    for( channel_idx=0 ; channel_idx<PIC_WDMA_NUM_CHANNELS ; channel_idx++ ) {

        dch = &desc_channel_list[channel_idx];
        if( !dch->is_open ) {
            continue;
        }

        dbg2( "pic wdma idx=%d ch=%d num_running=%d\n", channel_idx, dch->channel, dch->num_running );
//        ddma_desc_chain_dump( dch->desc_list, dch->num_descriptors );

        /* davep 14-Jan-2011 ; XXX temp debug */
//        ddma_data_peek( &dch->desc_list );
    }
    ice_dma_debug_log( &pic_cwdma_mm );
}

struct ice_dma_mm *pic_wdma_get_mm( void )
{
    return &pic_cwdma_mm;
}

/**
 * \brief allocate, initialize, and load buffers for PIC Write DMA 
 *
 * \author David Poole
 * \date 01-Oct-2005
 *
 * Originally started life as pictest_setup_wdma_buffers(). Moved from test
 * code to mainline code on 1-Oct-05.
 *
 * davep 09-Dec-2010 ; moved to parent class with fct pointer for polymorphism
 */

int pic_wdma_setup_buffers( uint8_t channel, int num_buffers, 
                            int rows_per_buffer, int bytes_per_row )
{
    int ice_dma_return;

    ice_dma_return = ice_dma_setup_buffers( &pic_cwdma_mm, channel, num_buffers,
                                            rows_per_buffer, bytes_per_row, pic_wdma_add_buffer );
    return ice_dma_return;
}

void pic_wdma_refill( int rows_per_buffer )
{
    ice_dma_refill( &pic_cwdma_mm, rows_per_buffer,
            pic_wdma_add_buffer, pic_wdma_channel_launch );
}

void pic_wdma_run_forever( void )
{
    ice_dma_run_forever( &pic_cwdma_mm );
}

static void channel_isr( struct ddma_channel *dch )
{
    scan_err_t scerr;
    struct ddma_descriptor *curr;

    /* 
     * BIG FAT NOTE!  This is an interrupt handler! 
     *
     */

    /* use a convenience pointer; remember we still have to modify the list
     * itself
     */
    curr = dch->head_desc;
    cpu_dcache_invalidate_region( curr, sizeof(struct ddma_descriptor) );

    /* davep 11-Oct-2010 ; add safety check to make sure the hardware is far
     * enough behind us we won't collide ("listening for the bulldozers")
     */
    XASSERT( DESCRIPTOR_OWNED_BY_BLOCK((struct ddma_descriptor *)curr->fw_prev_descriptor_addr), 
            (uint32_t)curr->fw_prev_descriptor_addr );

    /* we could have more than one completed descriptor so we'll walk the list
     * until we find a descriptor with own bit set
     */
    while( !DESCRIPTOR_OWNED_BY_BLOCK( curr ) )
    {
//        dbg2( "%s %p %p %#x\n", __FUNCTION__, curr, 
//                    curr->fw_next_descriptor_addr, curr->config_flags );

        XASSERT( (curr->config_flags & DDMA_DESCRIPTOR_CONFIG_OWNER_BLOCK)==0, 
                    curr->config_flags );

        SCANALYZER_LOG( LOG_PIC_DDMA_WRITE_ISR, curr->transfer_len_bytes );

        /* davep 27-Apr-2010 ; XXX temp debug */
        if( ! (curr->config_flags & DDMA_DESCRIPTOR_CONFIG_INT) ) {
            scanlog_hex_dump( (unsigned char *)(curr)-32, 64 );
            ddma_descriptor_list_dump( &dch->desc_list);
        }
        XASSERT( curr->config_flags & DDMA_DESCRIPTOR_CONFIG_INT, 
                    curr->config_flags ); 

        /* pass the DMA done interrupt to its handler (change handler so can
         * run pictest.c and scantest.c)
         */
        /* call the default method to do the buffer dance */
        ice_dma_isr( &pic_cwdma_mm, dch->channel );

        /* davep 22-Jun-2012 ; add channel data structure protection */
        scerr = ddma_channel_try_lock(dch);
        XASSERT( scerr==SCANERR_NONE, scerr );

        /* poison it */
        curr->src_addr = 0x42424242;
//        curr->src_addr = 0;
        curr->transfer_len_bytes = 0;

        /* Set the "own" bit back on the descriptor so I know I've already
         * handled this buffer. We're in a circular list so need to know when
         * to stop.
         */
        curr->config_flags |= DDMA_DESCRIPTOR_CONFIG_OWNER_BLOCK;

        /* davep 21-Feb-2012 ; suggestion from PaulZ for my line_rev problems ;
         * set SOI on first descriptor (must clear after descriptor used)
         */
        curr->config_flags &= ~DDMA_DESCRIPTOR_CONFIG_SOI;

        ddma_desc_flush( curr );

        /* move head of list to next element */
        dch->head_desc = (struct ddma_descriptor *)curr->fw_next_descriptor_addr;

        dch->num_running -= 1;
        XASSERT( dch->num_running >=0, dch->channel );

        curr = dch->head_desc;
        cpu_dcache_invalidate_region( curr, sizeof(struct ddma_descriptor) );

        ddma_channel_unlock( dch );

//        ddma_channel_sanity( dch );
    }
}

// callback from the pic wdma ISR
void pic_wdma_interrupt(struct pic_output_dma_interrupt_info *int_info)
{
    dros_err_t dros_err;
    int channel_idx;
    struct ddma_channel *dch;
    bool fatal_error = false;

    /* enter */
    // SCANALYZER_LOG(LOG_PIC_WDMA_IPEND, 1); scanalyzer is not inited early on - initializer earlier, and then uncomment the scanalyzer lines

    channel_idx = convert_pic_instance_channum_to_channel_idx(int_info->pic_instance, int_info->chan_num);


#ifdef PIC_WDMA_ISR_DEBUG
    dbg2("%s callback: pic_instance:%d channel_number:%d, int=0x%X, channel_idx=%x\n", __FUNCTION__, 
         int_info->pic_instance, int_info->chan_num, int_info->int_array, channel_idx);
#endif
    
    if (pic_output_dma_channel_busy(int_info->pic_instance, int_info->chan_num) == false)
        int_stats[channel_idx].is_idle++;
    
//    SCANALYZER_LOG(LOG_PIC_WDMA_IPEND, int_info->int_array);
    
    int_stats[channel_idx].count++;
    
    /* "Transfer End" interrupt */
    if (int_info->xfer_end)
    {
       dch = &desc_channel_list[channel_idx];

       XASSERT(dch->channel==channel_idx, dch->channel);
       XASSERT(dch->is_open, dch->channel);

       channel_isr(dch);
       int_stats[channel_idx].transfer_end++;
       /* davep 13-Oct-2010 ; if we've drained our chain */
       if (dch->num_running==0)
       {
          /* make sure block is truly idle */
          if (pic_output_dma_channel_busy(int_info->pic_instance, int_info->chan_num) == true)
          {
             pic_dump_output_dma_regs(int_info->pic_instance);
             errprint("ERROR!! out of descriptors, but the channel is still busy\n");
             ASSERT(0);
          }
       }
    }

    // end of image interrupt
    if (int_info->eoi)
    {
       int_stats[channel_idx].end_of_image++;
    }

    // soft reset complete interrupt
    if (int_info->soft_rst_cmpl)
    {
       int_stats[channel_idx].reset++;
        /* davep o01-Feb-2013 ; add a log message so these interrupts don't
         * confuse me when chasing over/underflows
         */
        isr_dbg2( "%s ch=%d reset\n", __FUNCTION__, channel_idx );

        dros_err = dros_sem_post(&icedma_softreset_sem);
        if (DROSERR_NONE != dros_err)
        {
            errprint("ERROR, posting semaphore icedma_softreset_sem returned error=%d\n", dros_err);
            ASSERT(0);
        }
    }
            

    // we aren't expecting any other interrupts - if set, we need to crash

    if (int_info->bad_rresp)
    {
       errprint("%s:ERROR:bad_rresp interrupt, 0x%X\n", __func__, int_info->int_array);
       fatal_error = true;
    }
    if (int_info->bad_bresp)
    {
       errprint("%s:ERROR:bad_bresp interrupt, 0x%X\n", __func__, int_info->int_array);
       fatal_error = true;
    }
    if (int_info->dir_err)
    {
       errprint("%s:ERROR:dir_err interrupt, 0x%X\n", __func__, int_info->int_array);
       fatal_error = true;
    }
    if (int_info->chg_line_align_err)
    {
       errprint("%s:ERROR:chg_line_align_err interrupt, 0x%X\n", __func__, int_info->int_array);
       fatal_error = true;
    }
    if (int_info->eol_align_err)
    {
       errprint("%s:ERROR:eol_align_err interrupt, 0x%X\n", __func__, int_info->int_array);
       fatal_error = true;
    }
    if (int_info->eoi_align_err)
    {
       errprint("%s:ERROR:eoi_align_err interrupt, 0x%X\n", __func__, int_info->int_array);
       fatal_error = true;
    }
    if (int_info->eoi_err)
    {
       errprint("%s:ERROR:eoi_err interrupt, 0x%X\n", __func__, int_info->int_array);
       fatal_error = true;
    }
    // die hard and fast on owner errors (firmware has messed up) 
    if (int_info->own)
    {
       errprint("%s:ERROR:ownership interrupt, 0x%X\n", __func__, int_info->int_array);
       fatal_error = true;
    }

    if (fatal_error)
    {
        pic_dump_output_dma_regs(int_info->pic_instance);
        pic_wdma_debug_log();
        ASSERT(0);
    }
    /* exit */
//    SCANALYZER_LOG(LOG_PIC_WDMA_IPEND, 0);
}

void pic_wdma_close( uint8_t pic_instance )
{
    int channel_idx;
    struct ddma_channel *dch;
    bool all_closed = true;
    uint8_t channel_pic_instance, channum;

    for( channel_idx = 0 ; channel_idx < PIC_WDMA_NUM_CHANNELS ; channel_idx++ ) {
        dch = &desc_channel_list[channel_idx];

        if( !dch->is_open  ) {
            continue;
        }

        convert_channel_idx_to_pic_instance_channum(channel_idx, &channel_pic_instance, &channum);
        if ( pic_instance == channel_pic_instance ) {
            pic_cwdma_channel_close( dch->channel );
        }
        else {
            all_closed = false;
        }
    }

    if ( all_closed ) {
        ice_dma_close( &pic_cwdma_mm );
    }
}

void pic_wdma_set_bitpack_mode(struct pic_handle_t *pic_handle, pic_bitpack_mode mode)
{
    int i;
    uint8_t pic_instance, channum, in_width;

#ifdef PIC_WDMA_DEBUG
    dbg2( "%s mode=%d\n", __FUNCTION__, mode );
#endif

    in_width = 0;
    
    switch(mode)
    {
    case PIC_BITPACK_16BIT:
        in_width = PIC_WDMA_CFG_IN_WIDTH_16_BIT;
        break;
        
    case PIC_BITPACK_8BIT:
        in_width = PIC_WDMA_CFG_IN_WIDTH_8_BIT;
        break;

    default:
        XASSERT(0, (uint32_t)mode);
    } 

    for (i=0 ; i<PIC_WDMA_NUM_CHANNELS ; i++)
    {
        convert_channel_idx_to_pic_instance_channum(i, &pic_instance, &channum);
        pic_output_dma_channel_set_input_data_width(pic_handle, in_width, channum);
    }
}

pic_bitpack_mode pic_wdma_get_bitpack_mode(struct pic_handle_t *pic_handle)
{
    int mode_bits;
    pic_bitpack_mode mode;

    /* grab data from pic0, channel zero since the set_bitpack_mode() simply writes all
     * channels identically
     */
    mode_bits = pic_output_dma_channel_get_input_data_width(pic_handle, 0);

    switch(mode_bits)
    {
        case PIC_WDMA_CFG_IN_WIDTH_8_BIT :
            mode = PIC_BITPACK_8BIT;
            break;

        case PIC_WDMA_CFG_IN_WIDTH_16_BIT :
            mode = PIC_BITPACK_16BIT;
            break;
            
        default :
            mode = PIC_BITPACK_16BIT;
            XASSERT(0, mode_bits);
            break;
    }

    return mode;
}

/**
 * \brief  Convenience function to set PIC WDMA packing mode.
 *
 * \author David Poole
 * \date 28-Aug-2012
 */

scan_err_t pic_wdma_set_bpp(struct pic_handle_t *pic_handle, int pic_bits_per_pixel)
{
    switch(pic_bits_per_pixel)
    {
        case 8 :
            pic_wdma_set_bitpack_mode(pic_handle, PIC_BITPACK_8BIT);
            break;

        case 16 :
            pic_wdma_set_bitpack_mode(pic_handle, PIC_BITPACK_16BIT);
            break;

        default :
            XASSERT(0, pic_bits_per_pixel);
            return SCANERR_INVALID_PARAM;
    }
    return SCANERR_NONE;
}

void pic_wdma_init_routing(uint8_t pic_instance, struct pic_handle_t *pic_handle)
{
    const uint8_t mono_channel=0, color_channel2=2, color_channel1=1, color_channel0=0;
    // NOTE, the routing function wants a 0 for channel 0, 1 for channel
    // 1 and 2 for channel 2.  This is for all pics
    pic_output_dma_set_routing(pic_handle, mono_channel,
                               color_channel2,
                               color_channel1,
                               color_channel0);

    /* davep 02-Apr-2013 ; use combined interrupts to reduce ISR calls */
    pic_output_dma_set_common_int(pic_handle, 1, false); // set this, don't OR the bit in
}

void pic_wdma_set_burst_size( struct pic_handle_t *pic_handle, ice_dma_burst_size burst_size )
{
    int i;
    int bs_bits;
    uint8_t pic_instance, channum;

#ifdef PIC_WDMA_DEBUG
    dbg2( "%s burst_size=%d\n", __FUNCTION__, burst_size );
#endif

    bs_bits = 0;

    for (i=0 ; i<PIC_WDMA_NUM_CHANNELS ; i++)
    {
        /* set new setting */
        switch (burst_size)
        {
            case ICE_DMA_BURST_16 :
                /* set burst len to 4 words (16 bytes) */
                bs_bits = PIC_WDMA_CFG_BURST_LEN_4_WORD;
                break;

            case ICE_DMA_BURST_32 :
                /* set burst len to 8 words (32 bytes) */
                bs_bits = PIC_WDMA_CFG_BURST_LEN_8_WORD;
                break;

            default:
                XASSERT(0,(uint32_t) burst_size);
        }
        convert_channel_idx_to_pic_instance_channum(i, &pic_instance, &channum);
        pic_output_dma_channel_set_burst_len(pic_handle, bs_bits, channum);
    }
}

// resetting all pics, all wdma channels
void pic_wdma_reset( uint8_t pic_instance )
{
    int i;
    // struct pic_wdma_status_info info;
    uint8_t channel_pic_instance, channum;

#ifdef PIC_WDMA_DEBUG
    dbg2( "%s\n", __FUNCTION__ );
#endif

    /* our dma descriptors must be padded to a cache line size so we can flush
     * them from cache without breaking neighboring cache lines
     */
    XASSERT(sizeof(struct ddma_descriptor)==cpu_get_dcache_line_size(), 
            sizeof(struct ddma_descriptor));

    ASSERT(!IN_INTERRUPT());

    // FIXME, we want to dynamically figure out what channels to reset, not use
    // constant PIC_WDMA_NUM_CHANNELS
    for (i=0 ; i<PIC_WDMA_NUM_CHANNELS ; i++)
    {
        convert_channel_idx_to_pic_instance_channum(i, &channel_pic_instance, &channum);
        if ( channel_pic_instance == pic_instance ) {
            // pic_wdma_channel_dump(i);
            pic_wdma_channel_disable(i);

            pic_wdma_channel_reset(i);
            
#if 0
            // dzh 01-06-15 skip the check as we don't have pic_handle.
            // Besides, the convenient function returns only the cached value.
            pic_output_dma_channel_get_status(pic_instance, &info, channum);

            if ((info.packer_empty == 0) || (info.empty_dbuf == 0) ||
                (info.empty_cbuf == 0))
            {
                // error, one of the statusus is bad
                errprint("%s: BAD!!! status after reset p_e=%d, e_d=%d, e_c=%d, sr=%d\n",
                       __func__, info.packer_empty, info.empty_dbuf, info.empty_cbuf, info.softreset);
                ASSERT(0);
            }
            XASSERT((info.dma_busy == 0), info.dma_busy);
#endif
        }
    }

    // FIXME, figure out which pics are instantiated, and only reset those - hardcoded both here
    // dzh 01-06-15 skip it as we don't have pic_handle.
    // pic_output_dma_set_disable_mode(pic_instance, PIC_OUT_DMA_PROCESS_DATA); // normal mode, process data

    // useful for test/debug */
    // pic_set_disable_mode_output_dma(pic_instance, PIC_OUT_DMA_DISCARD_DATA); // debugging - throw out data
}

// FIXME, PIC_WDMA_NUM_CHANNELS should be read from the ASIC
void pic_wdma_dump( void )
{
    int i;
    
    for (i=0 ; i<PIC_WDMA_NUM_CHANNELS; i++)
    {
        pic_wdma_channel_dump(i);
    }
}

