/*
**************************************************************************
This Source Code Form is subject to the terms of the Mozilla Public
License, v. 2.0. If a copy of the MPL was not distributed with this file,
You can obtain one at http://mozilla.org/MPL/2.0/.

Copyright (c) 2013-2015, Marvell International Ltd.

Alternatively, this software may be distributed under the terms of the GNU
General Public License Version 2, and any use shall comply with the terms and
conditions of the GPL.  A copy of the GPL is available at
http://www.gnu.org/licenses/old-licenses/gpl-2.0.html

THE FILE IS DISTRIBUTED AS-IS, WITHOUT WARRANTY OF ANY KIND, AND THE
IMPLIED WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE
ARE EXPRESSLY DISCLAIMED.  The GPL license provides additional details about
this warranty disclaimer.
******************************************************************************
*/



/**
 * \file picchipgap.c 
 *
 * \brief Sensor chipgap correction. 
 *
 * The hardware block itself is dubbed "Newmen" (ostensibly from Alfred E.
 * Neuman). I'm using the name "chipgap" to better describe the functionality.
 *
 * davep 28-Feb-2013
 *
 */

#include <stdint.h>
#include <stdbool.h>
#include <string.h>

#include "lassert.h"

#include "scantypes.h"
#include "scancore.h"
#include "scandbg.h"
#include "pic_handle_if.h"
#include "pic_if.h"
#include "pic_convenience_if.h"
#include "pic.h"

#ifndef HAVE_PIC_CHIPGAP
#error This code requires the PIC CHIPGAP block.
#endif


void pic_chipgap_reset( uint8_t pic_instance, struct pic_handle_t *pic_handle )
{
    dbg2( "%s pic=%d\n", __FUNCTION__, pic_instance );

    pic_chipgap_set_bypass(pic_instance, pic_handle, true);
    pic_chipgap_set_location(pic_instance, pic_handle, 0, 0);

    // Use all the default gap size values
    pic_chipgap_set_gaps(pic_instance, pic_handle, NULL, 0);
}

void pic_chipgap_dump( uint8_t pic_instance )
{
    pic_chipgap_drv_dump(pic_instance);
}

void pic_chipgap_set_bypass( uint8_t pic_instance, struct pic_handle_t *pic_handle, bool bypass )
{
    pic_chipgap_set_CFG(pic_handle, bypass ? 1 : 0);
}

uint32_t pic_chipgap_sum_list( uint8_t gap_list[], int num_gap_list )
{
    int i;
    uint32_t sum = 0;

    /* convenience function to sum the total additional pixels in a gap_list */
    for( i=0 ; i<num_gap_list ; i++ ) {
        sum += gap_list[i];
    }
    dbg2( "%s sum=%d\n", __FUNCTION__, sum );
    return sum;
}

scan_err_t pic_chipgap_set_gaps( uint8_t pic_instance, struct pic_handle_t *pic_handle, uint8_t gap_list[], int num_gap_list )
{
    int     i;
    uint8_t full_gap_list[PIC_CHIPGAP_MAX_GAPS];  

    dbg2( "%s pic=%d num_gaps=%d\n", __FUNCTION__, pic_instance, num_gap_list );

    if (!gap_list) return SCANERR_NONE;

    // Verify the number of gap corrections is OK with the PIC hw
    XASSERT( num_gap_list >= 0 && num_gap_list <= PIC_CHIPGAP_MAX_GAPS, num_gap_list );

    // We may not have been given a fully populated gap list (from the PIC hardware
    // pointof view), make a full list and apply default values where needed
    dbg2("%s gap_list=", __FUNCTION__);
    for (i=0; i<PIC_CHIPGAP_MAX_GAPS; i++)
    {
        if (i < num_gap_list)
        {
            // Verify the gap pixel size is OK, then copy to the full list
            XASSERT(gap_list[i] <= PIC_CHIPGAP_MAX_GAP_PIXELS, gap_list[i]);
            full_gap_list[i] = gap_list[i];
        }
        else
        {
            // Value not provided, so default to 0
            full_gap_list[i] = 0;
        }
        dbg2("%x ", full_gap_list[i]);
    }
    dbg2("\n");

    // Fill out INSERT0 virtual register structure
    pic_chipgap_set_INSERT0(pic_handle, full_gap_list);

    // Fill out INSERT1 virtual register structure
    pic_chipgap_set_INSERT1(pic_handle, full_gap_list+4);

    // Fill out INSERT2 virtual register structure
    pic_chipgap_set_INSERT2(pic_handle, full_gap_list+8);

    // Fill out INSERT3 virtual register structure
    pic_chipgap_set_INSERT3(pic_handle, full_gap_list+12);

    // Verbose debug, dump out the registers
    //pic_chipgap_dump(pic_instance);

    return SCANERR_NONE;
}

scan_err_t pic_chipgap_set_location( uint8_t pic_instance, struct pic_handle_t *pic_handle,
		uint32_t first_gap_offset, uint32_t gap_to_gap_width )
{
    /* parameters are, at this writing (27-Feb-2013), 16-bit values */
    XASSERT( first_gap_offset < (1<<16), first_gap_offset ); 
    XASSERT( gap_to_gap_width < (1<<16), gap_to_gap_width ); 

    pic_chipgap_set_LOCAT(pic_handle, gap_to_gap_width, first_gap_offset);

    dbg2( "%s pic=%d first=%d width=%d\n", __FUNCTION__, pic_instance, first_gap_offset, gap_to_gap_width);
    return SCANERR_NONE;
}

