/*
**************************************************************************
This Source Code Form is subject to the terms of the Mozilla Public
License, v. 2.0. If a copy of the MPL was not distributed with this file,
You can obtain one at http://mozilla.org/MPL/2.0/.

Copyright (c) 2014-2016, Marvell International Ltd.

Alternatively, this software may be distributed under the terms of the GNU
General Public License Version 2, and any use shall comply with the terms and
conditions of the GPL.  A copy of the GPL is available at
http://www.gnu.org/licenses/old-licenses/gpl-2.0.html

THE FILE IS DISTRIBUTED AS-IS, WITHOUT WARRANTY OF ANY KIND, AND THE
IMPLIED WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE
ARE EXPRESSLY DISCLAIMED.  The GPL license provides additional details about
this warranty disclaimer.
******************************************************************************
*/


/**
 * \file pic.h
 *
 * \brief Header file for PIC interface functions (DMA, etc).
 */


#ifndef PIC_H
#define PIC_H

#define NSENSOR_NUM_PICS 2

#ifdef HAVE_STAGGERED_SENSOR_SUPPORT
  #define PIC_WDMA_NUM_CHANNELS 6
#elif defined HAVE_NSENSOR_SUPPORT
  /* dual scan (two sensors, one each side of paper) */
  #define PIC_WDMA_NUM_CHANNELS 6
  #define PIC_WDMA_NUM_CHAN_PER_SENSOR 3
#else
  #define PIC_WDMA_NUM_CHANNELS 3
  #define PIC_WDMA_NUM_CHAN_PER_SENSOR 3
#endif

struct pic_interrupt_stats {
    uint32_t count;           /* increment every time we enter pic_interrupt() */
    uint32_t combus_overrun;  /* common block overrun */
    uint32_t adc_overrun;     /* ADC overrun */
    uint32_t dma[PIC_WDMA_NUM_CHANNELS];
    /* more? */
};

/* PIC's Pixel Correction block's PDCR1 register is very complex. This structure
 * wraps around the complexity and gives a one stop shop for setting the
 * register.
 *
 * The fields of this structure are designed to make it easier for someone who
 * knows what they're doing to get done what they want to get done. 
 */
struct pic_prnudsnu_config {

    /* valid values are 12, 16, 20, 24 */
    uint8_t total_bits;

    /* valid values are -2, -1, 0, +1, +2 */
    int8_t extra_prnu_bits[3];

    /* Valid values are one of PIC_PRNUDSNU_CONFIG_SCALE_PRNU_xxx. These fields in
     * PDCR1 reflect a floating point operation and don't map exactly to a C
     * value very well.
     */
    uint8_t scale_prnu_color[3];

    /* valid values are 0..11 which are <<0 .. <<11  (shift left) */
    uint8_t scale_dsnu_color[3];

    /* davep 11-Mar-2011 ; quadratic (not supported in all ASICs) */
    uint32_t quad_SF1, quad_SF2, quad_shift;

    /* davep 03-Mar-2011 ; finer grained prnu/dsnu (not supported in all ASICs) */
    uint32_t pmult, pscale, poffset;
    uint32_t dmult, dscale, doffset;
};

#define PIC_PRNUDSNU_CONFIG_SCALE_PRNU_MUL4  0
#define PIC_PRNUDSNU_CONFIG_SCALE_PRNU_MUL2  1
#define PIC_PRNUDSNU_CONFIG_SCALE_PRNU_NONE  2
/* davep 04-Apr-2008 ; the reset asic introduced a PRNU/DSNU max gain of shift
 * left three by stealing the config bits from the shift right 1.
 */
#ifdef HAVE_ICE_PRNUDSNU_MUL8
#define PIC_PRNUDSNU_CONFIG_SCALE_PRNU_MUL8  3
#else
#define PIC_PRNUDSNU_CONFIG_SCALE_PRNU_DIV2  3
#endif

/* pic_marg_calc_fit()'s fitting_strategy parameter */
typedef enum {
    PIC_MARG_PIXELFIT_TRUNCATE_RIGHT = 1,
    PIC_MARG_PIXELFIT_PAD_RIGHT
} pic_marg_fit_t;

/* davep 05-Mar-2013 ; This structure represents pixels-per-row flowing through
 * PIC. Not all ASICs have all features. The scanpipe function
 * pic_calc_pixels_per_row() calculates the PPR based on scan settings, current
 * mech's sensor chipgaps, and ASIC capabilities.
 */
struct pic_pixels_per_row {
    bool use_lrmargins0;
    bool use_chipgap;
    bool use_lrmargins1;
    bool use_hscale;

    uint32_t pic_pixels_per_row_in;

    uint32_t lrmargins0_ppr_in;
    uint32_t lrmargins0_ppr_out;

    uint32_t chipgap_ppr_in;
    uint32_t chipgap_ppr_out;

    uint32_t lrmargins1_ppr_in;
    uint32_t lrmargins1_ppr_out;

    uint32_t hscale_ppr_in;
    uint32_t hscale_ppr_out;

    uint32_t pic_pixels_per_row_out;
};


/* function prototypes */
scan_err_t pic_onetime_init( void );
scan_err_t pic_cleanup_module( void );
scan_err_t pic_soft_setup( uint8_t pic_instance );
void pic_reset(uint8_t pic_instance);

void pic_channel_setup( uint8_t channel );

/* functions for 2+ sensors */
#if 0
uint32_t pic_prnudsnu_get_block( void );
uint32_t pic_prnudsnu_dma_get_block( void );
void pic_prnudsnu_set_block( uint32_t block );
void pic_prnudsnu_dma_set_block( uint32_t block );
scan_err_t pic_enable_dual_channel( bool enable );
bool pic_get_dual_channel_enabled( void );
#endif

void convert_channel_idx_to_pic_instance_channum(int channel_idx, uint8_t *pic_instance,
                                                 uint8_t *channum);

/* hardware level functions for bulb monitor */
void pic_bm_set_bypass( bool bypass );
bool pic_bm_get_bypass( void );

void pic_bm_set_color_counter( bool not_odd_even );
//bool pic_bm_get_color_counter( void );

void pic_bm_set_lines_to_avg( unsigned int lines );
//unsigned int pic_bm_get_lines_to_avg( void );

void pic_bm_set_window_size( unsigned int numpix );
//unsigned int pic_bm_get_window_size( void );

void pic_bm_set_startpix( unsigned int startpix );
//unsigned int pic_bm_get_startpix( void );

//unsigned int pic_bm_get_sum_col0_mono_even( void );
//unsigned int pic_bm_get_sum_col1_odd( void );
//unsigned int pic_bm_get_sum_col2( void );
//unsigned int pic_bm_get_exposure_data( void );

void pic_bm_dump( void );
void pic_bm_reset( void );

void pic_adcnorm_set_bypass( struct pic_handle_t *pic_handle, bool bypass );
void pic_adcnorm_set_abits( struct pic_handle_t *pic_handle, uint8_t abits );
void pic_adcnorm_set_left_justify( struct pic_handle_t *pic_handle, bool left_justify );

/* hardware level functions for left/right margins */
void pic_marg_set_right( struct pic_handle_t *pic_handle, uint32_t right_pixels, uint8_t lrmargin_block );
void pic_marg_set_left( struct pic_handle_t *pic_handle, uint32_t left, uint8_t lrmargin_block );
uint32_t pic_marg_get_right(struct pic_handle_t *pic_handle, uint8_t lrmargin_block );
uint32_t pic_marg_get_left( struct pic_handle_t *pic_handle, uint8_t lrmargin_block );

void pic_marg_reset( struct pic_handle_t *pic_handle );
void pic_marg_dump( uint8_t pic_instance, uint8_t lrmargin_block);

scan_err_t pic_marg_calc_fit( uint32_t left_margin_pixels, 
                       uint32_t *right_margin_pixels,
                       int bits_per_pixel, 
                       pic_marg_fit_t fitting_strategy );

void pic_marg_set_bypass( struct pic_handle_t *pic_handle, bool bypass, uint8_t lrmargin_block );
bool pic_marg_get_bypass( struct pic_handle_t *pic_handle, uint8_t lrmargin_block);

/* hardware level functions for PRNU/DSNU compensation */
void pic_prnudsnu_dma_onetime_init( uint8_t pic_instance );
void pic_prnudsnu_dma_dump( uint8_t pic_instance );
void pic_prnudsnu_dump( uint8_t pic_instance );
void pic_prnudsnu_set_num_correction_bits( unsigned int numbits );

typedef enum {
    /* note I'm using whacky numbers here to catch old code using the previous
     * float parameters
     */
    PIC_PRNU_MULTIPLIER_15=42, /* 1.5x */
    PIC_PRNU_MULTIPLIER_20=43, /* 2.0x */
    PIC_PRNU_MULTIPLIER_30=44, /* 3.0x */
    PIC_PRNU_MULTIPLIER_50=45, /* 5.0x */
} pic_prnu_multiplier_t;

void pic_prnudsnu_set_num_extra_prnu_bits_col0_mono_even( int numbits );
void pic_prnudsnu_set_max_prnu_mult_col0_mono_even( pic_prnu_multiplier_t multiplier );
void pic_prnudsnu_set_dsnu_mult_col0_mono_even( uint8_t multiplier );

void pic_prnudsnu_set_num_extra_prnu_bits_col1_odd( int numbits );
void pic_prnudsnu_set_max_prnu_mult_col1_odd( pic_prnu_multiplier_t multiplier );
void pic_prnudsnu_set_dsnu_mult_col1_odd( uint8_t multiplier );

void pic_prnudsnu_set_num_extra_prnu_bits_col2( int numbits );
void pic_prnudsnu_set_max_prnu_mult_col2( pic_prnu_multiplier_t multiplier );
void pic_prnudsnu_set_dsnu_mult_col2( uint8_t multiplier );

void pic_prnudsnu_set_config(struct pic_handle_t *pic_handle, struct pic_prnudsnu_config *p);
void pic_prnudsnu_encode_lut( int total_bits, int extra_prnu_bits,
                              int num_pixels,
                              uint16_t prnu[], uint16_t dsnu[], uint8_t quad[],
                              uint8_t prnudsnu_lut[],
                              int prnudsnu_lut_num_entries );

void pic_prnudsnu_set_color_counter( bool odd_even );
#ifdef ICEDMA_H
void pic_prnudsnu_set_lut_addr( uint8_t pic_instance, dma_addr_t lut_addr );
#endif
void pic_prnudsnu_set_lut_xfer_count( uint8_t pic_instance, struct pic_handle_t *pic_handle, unsigned long count );
unsigned long pic_prnudsnu_get_lut_xfer_count( uint8_t pic_instance );
void pic_prnudsnu_start_lut_dma(uint8_t pic_instance);
bool pic_prnudsnu_lut_dma_active( void );
bool pic_prnudsnu_lut_dma_queue_avail( void );
bool pic_prnudsnu_lut_dma_complete( void );

/* hardware level functions for hscale */
uint32_t pic_hs_calc_factor( uint32_t pixels_in, uint32_t pixels_out );
void pic_hs_calc_row( int pixels_per_row_in, int x_numer, int x_denom, 
                      int bits_per_pixel, int *pixels_per_row_out );

/* 
 * 7-Apr-05; new functions compatible with generic ICE DMA driver 
 */

/* forward declaration of the buffer management structure */
struct ice_dma_buffer;

typedef enum { PIC_BITPACK_16BIT, PIC_BITPACK_8BIT } pic_bitpack_mode;

#ifdef ICEDMA_H
void pic_wdma_set_burst_size( struct pic_handle_t *pic_hdl, ice_dma_burst_size burst_size );
#endif
void pic_wdma_set_bitpack_mode( struct pic_handle_t *pic_hdl, pic_bitpack_mode mode );
pic_bitpack_mode pic_wdma_get_bitpack_mode( struct pic_handle_t *pic_hdl );
void pic_wdma_init_routing(uint8_t pic_instance, struct pic_handle_t *pic_hdl);
void pic_wdma_enable_line_reverse( uint8_t pic_instance, struct pic_handle_t *pic_handle, bool enable );
void pic_wdma_enable_line_reverse_by_channel( struct pic_handle_t *pic_handle, uint8_t channel, bool enable );
scan_err_t pic_wdma_set_bpp( struct pic_handle_t *pic_hdl, int pic_bits_per_pixel );

/* hardware level functions for PIC write DMA */
void pic_wdma_reset( uint8_t pic_instance );
void pic_wdma_channel_reset( uint8_t channel );
void pic_wdma_channel_enable( uint8_t channel );
bool pic_wdma_channel_is_enabled( uint8_t channel );
void pic_wdma_channel_disable( uint8_t channel );
void pic_wdma_channel_start( uint8_t channel );
void pic_wdma_channel_load( uint8_t channel, uint8_t *data_ptr, dma_addr_t dma_dest,
                            uint32_t rows, uint32_t bytes_per_row );
void pic_wdma_channel_status( struct pic_handle_t *pic_handle,
		              uint8_t channel,
                              uint32_t *cnt_status, 
                              uint32_t *addr_status);
void pic_wdma_channel_dump( uint8_t channel );
void pic_wdma_dump( void );

void pic_wdma_channel_launch( uint8_t channel );

/* software driver level functions for PIC write dma (icedma.c methods) */
void pic_wdma_init_onetime(void);

int pic_wdma_open( struct pic_handle_t *pic_handle,
		   uint8_t channels[],
                   uint8_t num_channels,
                   int num_empties,
                   int total_rows,
                   int bytes_per_row );

scan_err_t pic_wdma_add_buffer( uint8_t channel, 
                                uint8_t *data, 
                                uint32_t datalen,
                                uint32_t rows, 
                                uint32_t bytes_per_row );

void pic_wdma_add_ready( uint8_t channel, 
                                struct ice_dma_buffer **addme );

//int pic_wdma_channel_num_readies(uint8_t channel);

void pic_wdma_free_empty( uint8_t channel, struct ice_dma_buffer **freeme );

void pic_wdma_isr( uint8_t channel );

void pic_wdma_sanity( void );

void pic_wdma_debug_log( void );

void pic_wdma_close( uint8_t pic_instance );
void pic_wdma_refill( int rows_per_buffer );
void pic_wdma_cancel( void );
void pic_wdma_run_forever( void );

/* testing code needs to be able to see the insides of picdma.c so we don't
 * have a ridiculous proliferation of small utility functions
 */
#ifdef PICDMA_FRIEND
struct ice_dma_mm *pic_wdma_get_mm( void );
#endif

int pic_wdma_setup_buffers( uint8_t channel, 
                            int num_buffers, 
                            int rows_per_buffer, 
                            int bytes_per_row );

/* called from pic_interrupt() to handle WDMA interrupts */
void pic_wdma_interrupt(struct pic_output_dma_interrupt_info *int_info);

void pic_interrupt_disable( uint8_t pic_instance );
void pic_interrupt_disable_all_instances( void );
void pic_interrupt_enable( uint8_t pic_instance );

void pd_lut_swap32( uint8_t *lut, int lutsize_bytes );

struct scanvars; /* forward decl */
scan_err_t pic_calc_pixels_per_row( const struct scanvars *sv, 
                                    struct pic_pixels_per_row *p,
                                    uint32_t scan_pixels_per_row_in  );
#ifdef HAVE_PIC_PSESD
void pic_psesd_reset( void );
void pic_psesd_dump( void );
void pic_pgesd_interrupt();
void pic_esd_setup(int pix_per_row);
void pic_esd_color_reset(uint8_t color);
void pic_esd_enable( bool enable );
void pic_ps_go(bool tof_nbof);
void pic_ps_setup(bool start_bit, bool is_color, uint8_t chan_en, uint32_t res);
#endif

#ifdef HAVE_PIC_CHIPGAP
void pic_chipgap_reset( uint8_t pic_instance, struct pic_handle_t *pic_handle );
void pic_chipgap_dump( uint8_t pic_instance );
void pic_chipgap_set_bypass( uint8_t pic_instance, struct pic_handle_t *pic_handle, bool bypass );
uint32_t pic_chipgap_sum_list( uint8_t gap_list[], int num_gap_list );
scan_err_t pic_chipgap_set_location( uint8_t pic_instance, struct pic_handle_t *pic_handle, uint32_t first_gap_offset, uint32_t gap_to_gap_width );
scan_err_t pic_chipgap_set_gaps( uint8_t pic_instance, struct pic_handle_t *pic_handle, uint8_t gap_list[], int num_gap_list ) ;
#else
static inline void pic_chipgap_reset( uint8_t pic_instance ) {}
static inline void pic_chipgap_dump( uint8_t pic_instance ) {}
static inline void pic_chipgap_set_bypass( uint8_t pic_instance, bool bypass ) {}
static inline uint32_t pic_chipgap_sum_list( uint8_t gap_list[], int num_gap_list ) { return 0; }
static inline scan_err_t pic_chipgap_set_location( uint8_t pic_instance, uint32_t first_gap_offset, uint32_t gap_to_gap_width )
                                                    { return SCANERR_NOT_IMPLEMENTED; }
static inline scan_err_t pic_chipgap_set_gaps( uint8_t pic_instance, uint8_t gap_list[], int num_gap_list ) 
                                                    { return SCANERR_NOT_IMPLEMENTED; }
#endif

#endif

