/*
**************************************************************************
This Source Code Form is subject to the terms of the Mozilla Public
License, v. 2.0. If a copy of the MPL was not distributed with this file,
You can obtain one at http://mozilla.org/MPL/2.0/.

Copyright (c) 2014-2015, Marvell International Ltd.

Alternatively, this software may be distributed under the terms of the GNU
General Public License Version 2, and any use shall comply with the terms and
conditions of the GPL.  A copy of the GPL is available at
http://www.gnu.org/licenses/old-licenses/gpl-2.0.html

THE FILE IS DISTRIBUTED AS-IS, WITHOUT WARRANTY OF ANY KIND, AND THE
IMPLIED WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE
ARE EXPRESSLY DISCLAIMED.  The GPL license provides additional details about
this warranty disclaimer.
******************************************************************************
*/


/**
 * \file pic.c
 *
 * \brief Interface functions to PIC
 *
 * PIC - Pipelined Image Correction.  PIC's goal in life is to make an image as
 * accurate as possible.
 */

#include <stdint.h>
#include <stdbool.h>
#include <string.h>

#include "scos.h"

#include "list.h"
#include "interrupt_api.h"
#include "lassert.h"
#include "hwconfig_api.h"
#include "cpu_api.h"
#include "regAddrs.h"
#include "io.h"

#include "scantypes.h"
#include "scancore.h"
#include "scandbg.h"
#include "safetylock.h"
#include "scantask.h"
#include "icedma.h"

#include "pic_handle_if.h"
#include "pic_if.h"  // pic driver
#include "pic_convenience_if.h"

#include "pic.h"
#include "picreset.h"
#include "scanalyzer.h"
#include "safeint.h"
#include "scansen.h"
#include "scanvars.h"

#include "afe_if.h"


/* Define to turn on more verbose debugging */
//#define PIC_DEBUG  

#ifdef PIC_DEBUG
  #define pic_dbg2 dbg2 
#else
  #define pic_dbg2(...) 
#endif

/* Fields for PDCR2 COEFFWIDTH. Keep private to this file. Used inside
 * pic_prnudsnu_set_coeffwidth()  
 */
#define PIC_PC_PDCR2_COEFFWIDTH_12BITS 0
#define PIC_PC_PDCR2_COEFFWIDTH_16BITS 1
#define PIC_PC_PDCR2_COEFFWIDTH_20BITS 2
#define PIC_PC_PDCR2_COEFFWIDTH_24BITS 3
#define PIC_PC_PDCR2_COEFFWIDTH_28BITS 4
#define PIC_PC_PDCR2_COEFFWIDTH_32BITS 5

void pic_interrupt_disable( uint8_t pic_instance )
{
    pic_disable_pic_common_irqs(pic_instance, NULL);
}

void pic_interrupt_disable_all_instances( void )
{
    int pic_instance;
    for( pic_instance = 0; pic_instance < scansen_get_num_sensors(); pic_instance++ )
    {
        pic_interrupt_disable(pic_instance);
    }
}

void pic_interrupt_enable( uint8_t pic_instance )
{
    pic_enable_pic_common_irqs(pic_instance, NULL); 
}

void pic_adcnorm_set_bypass( struct pic_handle_t *pic_handle, bool bypass )
{
	bool invert, left_justify, bypass_old;
	uint8_t pad_fill, numadcbits;

	pic_adc_get_norm(pic_handle, &bypass_old, &invert, &left_justify, &pad_fill, &numadcbits);
	if (bypass_old != bypass) {
		pic_adc_set_norm(pic_handle, bypass, invert, left_justify, pad_fill, numadcbits);
	}
}

void pic_adcnorm_set_abits( struct pic_handle_t *pic_handle, uint8_t abits )
{
	bool invert, left_justify, bypass;
	uint8_t pad_fill, bits_old;

	pic_adc_get_norm(pic_handle, &bypass, &invert, &left_justify, &pad_fill, &bits_old);
	if (bits_old != abits) {
		pic_adc_set_norm(pic_handle, bypass, invert, left_justify, pad_fill, abits);
	}
}

void pic_adcnorm_set_left_justify( struct pic_handle_t *pic_handle, bool left_justify )
{
	bool invert, lj_old, bypass;
	uint8_t pad_fill, bits;

	pic_adc_get_norm(pic_handle, &bypass, &invert, &lj_old, &pad_fill, &bits);
	if (lj_old != left_justify) {
		pic_adc_set_norm(pic_handle, bypass, invert, left_justify, pad_fill, bits);
	}
}

#if 0
//FIXME!!!! THESE 2 functions need to be removed once we mod all 
//functions to pass down the pic_instance/sensor_num, whatever
uint32_t pic_prnudsnu_get_block(void)
{
    return current_block_num;
}

/**
 * \brief swap global PRNU/DSNU registers between blocks
 *
 * Original idea is similar to Bank Switching.
 * http://en.wikipedia.org/wiki/Bank_switching
 *
 * Adding dual-sensor support without breaking all existing code. We will
 * swap the global pointer back/forth depending on which sensor we're
 * configuring.
 *
 *  pic_prnudsnu_set_block(0);
 *  calibrate sensor 0
 *  pic_prnudsnu_set_block(1);
 *  calibrate sensor 1
 *
 * \author David Poole
 * \date 15-Jan-2013
 */

void pic_prnudsnu_set_block( uint32_t block ) 
{
    uint32_t sensor_num;

    sensor_num = scansen_get_num_sensors();

    XASSERT(block < sensor_num, block );

//    prnudsnu_regs = prnudsnu_regs_array[block];
//    dbg2("%s prnudsnu_regs is %p\n",__FUNCTION__,prnudsnu_regs);
    dbg2("current block is %d\n",current_block_num);
    current_block_num = block;
}

/**
 * \brief  Enable/disable dual scan mode
 *
 * \author David Poole
 * \date 15-Jan-2013
 */

scan_err_t pic_enable_dual_channel( bool enable )
{
#if 0    
#ifdef HAVE_NSENSOR_SUPPORT
    if( enable ) {
        /* turn on dual channel */
        pic_ci_regs->PCR |= PIC_INTCTRL_PCR_DUALCH_MASK; 
    }
    else {
        /* turn off dual channel */
        pic_ci_regs->PCR &= ~PIC_INTCTRL_PCR_DUALCH_MASK; 
    }
    return SCANERR_NONE;
#else
    return SCANERR_NOT_IMPLEMENTED;
#endif
#endif
    dbg1("%s:STUB - need to implement\n", __func__);
    return SCANERR_NONE;
}

/**
 * \brief  Get PIC Dual Channel enable/disable state
 *
 * Created for sanity checking. All of Scan block, CISX, and PIC must be in
 * agreement for dual channel mode.
 *
 * \author David Poole
 * \date 03-Apr-2013
 */

bool pic_get_dual_channel_enabled( void )
{
#if 0
#ifdef HAVE_NSENSOR_SUPPORT
    return (pic_ci_regs->PCR & PIC_INTCTRL_PCR_DUALCH_MASK)==PIC_INTCTRL_PCR_DUALCH_MASK;
#else
    return false;
#endif
#endif
    dbg1("%s:STUB - need to implement\n", __func__);
    return false;
}
#endif

void convert_channel_idx_to_pic_instance_channum(int channel_idx, uint8_t *pic_instance,
                                                 uint8_t *channum)
{
    switch(channel_idx)
    {
    case PIC_DMA_CHANNEL_CEVEN_0:
        // front side color 0 or mono
        // same as case PIC_DMA_CHANNEL_MEVEN
        *pic_instance = 0;
        *channum = 0;
        break;
    case PIC_DMA_CHANNEL_CEVEN_1:
        // front side color 1
        *pic_instance = 0;
        *channum = 1;
        break;
    case PIC_DMA_CHANNEL_CEVEN_2:
        // front side color 2
        *pic_instance = 0;
        *channum = 2;
        break;
    case PIC_DMA_CHANNEL_CODD_0:
        // back side color 0 or mono
        // same as case PIC_DMA_CHANNEL_MODD        
        *pic_instance = 1;
        *channum = 0;
        break;
    case PIC_DMA_CHANNEL_CODD_1:
        // back side color 1
        *pic_instance = 1;
        *channum = 1;
        break;
    case PIC_DMA_CHANNEL_CODD_2:
        // back side color 2
        *pic_instance = 1;
        *channum = 2;
        break;
    default:
        errprint("%s: Bad channel id %d\n",__func__, channel_idx);
        XASSERT(0, channel_idx);
    }
}

/*
 * Start of PIC hardware/software setup code.
 *
 */
//--------------------------------------------------------------------------
// Function     : pic_reset
//   returns    : 
//   arg1       : 
//   arg2       : 
// Created by   : Brad Smith
// Date created : 5/13/05
// Description  : Resets the PIC block to its power-on default values.
//
// 
// Notes        : 
//
//--------------------------------------------------------------------------
// FIXME, should these values move into pic_if.h?
#define DEF_PIC_ADC_BYPASS  true
#define DEF_PIC_ADC_INVERT false
#define DEF_PIC_ADC_LEFT_JUST false
#define DEF_PIC_ADC_PAD_FILL 0
#define DEF_PIC_ADC_NUM_BITS 0xD    
#define DEF_PIC_HS_SCALE_FACTOR 0x00010000
void pic_reset(uint8_t pic_instance)
{
    //bool bypass, invert, left_justify;
    //char pad_fill, num_adc_bits;

    /* restore PIC to power-on default settings */
    pic_dbg2( "%s\n", __FUNCTION__) ;

#ifndef PD_USE_EXTERNAL_VALS
    /* 
     * Reset PIC Common Block
     */
    /* Clear PIC common block */
    pic_do_reset(pic_instance); // start a soft reset

#if 0
    SANDRA FIXME, what do we want to do in  a pic_reset????
    pic_disable_irq(pic_instance, NULL); // disable all pic ints

    /* 
     * Reset ADC
     */
    // pic_adc_set_norm and pic_adc_get_norm have not actual implemenation
    //pic_adc_set_norm(pic_instance, DEF_PIC_ADC_BYPASS, DEF_PIC_ADC_INVERT,
    //                 DEF_PIC_ADC_LEFT_JUST, DEF_PIC_ADC_PAD_FILL,
    //                 DEF_PIC_ADC_NUM_BITS);
    /* verify PIC is at least alive */
    //pic_adc_get_norm(pic_instance, &bypass, &invert, &left_justify, &pad_fill, &num_adc_bits);
#endif    
#if 0
    // FIXME - once pic_adc_get/set_norm is implemented, uncomment these asserts
    XASSERT(DEF_PIC_ADC_BYPASS == bypass, bypass);
    XASSERT(DEF_PIC_ADC_INVERT == invert, invert);
    XASSERT(DEF_PIC_ADC_LEFT_JUST == left_justify, left_justify);
    XASSERT(DEF_PIC_ADC_PAD_FILL == pad_fill, pad_fill);
    XASSERT(DEF_PIC_ADC_NUM_BITS == num_adc_bits, num_adc_bits);
#endif    
    /* 
     * Reset Bulb Monitor block
     */
 //   pic_bm_reset();  //  FIXME fix the picbm2005.c file to use driver

    /* 
     * Reset Pixel Correction block
     */
    // we may have multiple pixel correction blocks nowadays.  Reset all of them
#if 0 //Eric H. Let user do the setup
    for (i=0;i<scansen_get_num_sensors();i++) // FIXME CHANGE THIS TO get_num_pic_blocks?
    {
        pic_prnudsnu_set_block(i);   // switch to the block to reset
        pic_prnudsnu_dma_set_block(i);
        pic_pd_idma_enable_lut_dma(i, false);
        pic_prnudsnu_reset_lut_dma(); // reset each lut dma

        // configure the LUT for total of 12 bits, equal number of bits for prnu and dsnu
        pic_pd_set_config1(i, 12, 0, 0, 0);
        // and 12 bit coefficients, bypass all
        pic_pd_set_coeffwidth(i, 12);
        pic_pd_set_bypass_all(i, true);
        // set defaults for Percentage of normal
        pic_pd_set_PON(i, PD_PON_N_DEFAULT, PD_PON_NORM_DEFAULT);
    }
#endif

#endif // PD_USE_EXTERNAL_VALS

    /* 
     * Reset H Scale block
     */
    // pic_hs_set_bypass and pic_hs_set_hscale_factor have no actual implementation
    //pic_hs_set_bypass(pic_instance, true);
    //pic_hs_set_hscale_factor(pic_instance, DEF_PIC_HS_SCALE_FACTOR);

    /*
     * Margin block(s)
     *
     * Older ICE blocks lumped the margins in with Bulb Monitor.
     * New in 2011 is an ICE block with standalong margin blocks. 
     */
    // dzh - 01-07-15, with the new api, maybe it is unnecessary to do marg_reset as we will
    //                 call pic_create_new_default_handle and then pic_do_configure to
    //                 reset everything.
    //pic_marg_reset(pic_instance);

    /* 
     * Reset Bit-Depth Reduction block
     */
    // pic_bdr_reset(pic_instance);

    // Reset Bulb Shadow registers
    // FIXME - there really are n sensor bulb shadow registers - but since not currently using....
#if 0
    FIXME - remove all these direct writes and replace with bulb shadow block calls
    *PIC_PC_BSRLSGCR = PIC_BS_RLSGC_R;
    *PIC_PC_BSLSVCR  = PIC_BS_LSVC_R;
    *PIC_PC_BSLMPCR0 = PIC_BS_LMPCR0_R;
    *PIC_PC_BSLMPCR1 = PIC_BS_LMPCR1_R;
    *PIC_PC_BSLMPCR2 = PIC_BS_LMPCR2_R;
    *PIC_PC_BSRSVCR  = PIC_BS_RSVC_R;
    *PIC_PC_BSRSPCR0 = PIC_BS_RSPCR0_R;
    *PIC_PC_BSRSPCR1 = PIC_BS_RSPCR1_R;
    *PIC_PC_BSRSPCR2 = PIC_BS_RSPCR2_R;
#endif
    dbg1("%s:STUB - need to implement\n", __func__);
    /* 
     * Reset PIC WDMA
     */
    pic_wdma_reset( pic_instance );   

//    pic_chipgap_reset( pic_instance, pic_handle );
}

/**
 * \brief PIC hardware setup.
 *
 *  Restores PIC to as close to power-on values as possible. Doesn't interfere
 *  with operating system interface (interrupts, extra debugging, etc).
 *
 *  Compare to pic_onetime_init().
 *
 *  pic_onetime_init() does first-time intialization of PIC (turn on the
 *  hardware, hook up operating system interrupts, add debug commands, one-time
 *  data structure set-up).
 *
 *  This function does things that need to be done to reset PIC completely to
 *  a predictable, clean, runnable state.
 *
 *  Originally this code was all part of pic_onetime_init() but we now want to
 *  be able to restore PIC to power-on defaults after each cal (Bug 10870).
 *
 * \retval 0 success
 * \retval !0 detectable hardware failure
 *
 * \author David Poole
 * \date 21-Oct-2008
 *
 */
scan_err_t pic_soft_setup( uint8_t pic_instance )
{
    //int     afe_bpp;

    pic_dbg2("%s\n", __FUNCTION__);
        
    /* don't enable until everything is set up and quiescent */
//    pic_disable_irq(pic_instance, NULL);  // SANDRA FIXME - DO WE WANT THIS?

    // Reset all PIC registers to their power-on values.
//    pic_reset(pic_instance);
    pic_do_reset(pic_instance);

   // FIXME, need to add callbacks for pic common, (and other interrupts?)
    
    /* reset ADC by setting / clearing bypass bit */
//    pic_adcnorm_set_bypass( true );
//    pic_adcnorm_set_bypass( false );

#if 0
    // dzh - 01-07-15 moved to pic_start in scanpipe.c
    /* davep 21-Jan-2010; set up the interface between PIC-ADC and AFE based
     * on the new AFE hardware descriptor structure. (Was hardwired to 16-bpp,
     * right justify)
     */
    dbg1("STUB - need to call afe_get_hardware_config\n");

    afe_bpp = afe_get_bits_per_pixel( pic_instance );
    if ( afe_bpp < 0 ) 
    {
        afe_bpp = 16;
    }

//    pic_adcnorm_set_abits( afe_bpp );
//    pic_adcnorm_set_left_justify( true );

    /* set BWM to bypass, rest defaults */
    dbg1("Stub - need to call pic_bm_set_bypass\n");
    //pic_bm_set_bypass( true );

    /* by default, set bypass to disable hscale block */
//    pic_hs_set_bypass(pic_instance, true);

    pic_wdma_init_routing(pic_instance, pic_handle);

    /* davep 04-Oct-2010 ; default to 16-bpp. Adding this because PICDMA2005
     * defaults to 16-bpp and a lot of code is assuming 16-bpp on reset. The
     * descriptor PIC DMA defaults to 8-bpp.
     */
    pic_wdma_set_bitpack_mode( pic_handle, PIC_BITPACK_16BIT );

    /* davep 21-Feb-2012 ; set the burst size */
#if ICE_DMA_BYTE_ALIGN==16
    pic_wdma_set_burst_size( ICE_DMA_BURST_16 );
#elif ICE_DMA_BYTE_ALIGN==32
    pic_wdma_set_burst_size( pic_handle, ICE_DMA_BURST_32 );
#else
    #error Unknown ICE_DMA_BYTE_ALIGN
#endif

//     pic_enable_irq(pic_instance, NULL); // enable all ints  // SANDRA FIXME, do we want this?
#endif

    return SCANERR_NONE;
}

//--------------------------------------------------------------------------
// Function     : pic_onetime_init
//   returns    : integer status
//   arg1       :
//   arg2       :
// Created by   : David Poole
// Date created : ????
// Description  : Sets up the PIC block with some default values.
//
// 
//--------------------------------------------------------------------------

scan_err_t pic_onetime_init( void )
{
//    scan_err_t scerr;
    int i;

    pic_dbg2("%s\n", __FUNCTION__);

    /* add PIC interrupt into OS but don't enable until everything is set up */
    for (i=0;i<scansen_get_num_sensors();i++)
    {
//        pic_disable_irq(i, NULL); SANDRA, FIXME

#if 0
        // dzh 01-06-15 pic_soft_setup moved to pic_start
        // FIXME, when passing parms down, pass the sensor down to pic_soft_setup
        scerr = pic_soft_setup( i );
        if( scerr != 0 ) {
            /* pic_soft_setup() hardwired to return 0 at this time but just in
             * case... 
             */
            return scerr;
        }
#endif

        /* note: unconditionally enabling the interrupts */
//        pic_enable_irq(i, NULL); SANDRA FIXME
        pic_prnudsnu_dma_onetime_init( i );
    }
    SCANALYZER_ENABLE_LOG( LOG_PIC_INTERRUPT );
    SCANALYZER_ENABLE_LOG( LOG_PIC_WDMA_IPEND );

    return SCANERR_NONE;
}

/**
 * \brief  Release all PIC resources. 
 *
 * Originally created for Linux kernel module __exit
 *
 * \author David Poole
 * \date 23-Apr-2012
 */

scan_err_t pic_cleanup_module( void )
{
    //pic_release_interrupt(); FIXME

    return SCANERR_NONE;
}

/**
 * \brief  htonl the pixel correction LUT
 *
 *
 * \author David Poole
 * \date 06-Apr-2010
 *
 */

void pd_lut_swap32( uint8_t *lut, int lutsize_bytes )
{
    int lutsize_uint32;
    uint32_t *ptr32;

    /* note I'm not trying to handle a lut that doesn't align exactly on a
     * 4 byte boundary. Given our DMA alignment restrictions, we should be
     * getting a 16- or 32-byte aligned buffer anyway.
     */
    lutsize_uint32 = lutsize_bytes / 4;

    if( lutsize_bytes%4 != 0 ) {
        dbg1( "%s not 4-byte aligned!\n", __FUNCTION__ );
    }

    ptr32 = (uint32_t *)lut;
    while( lutsize_uint32 ) {

        *ptr32 = asp_htonl( *ptr32 );
        ptr32++;
        lutsize_uint32--;
    }

//    scanlog_hex_dump( lut, MIN(lutsize_bytes,64) );
}

/**
 * \brief  Pack the PRNU/DSNU correction factors into a LUT suitable for input
 * into the PIC PC RDMA.
 *
 * \param total_bits ; total bits from PDCR1 ; valid values are 12, 16, 20, 24
 *
 * \param extra_prnu_bits ; extra prnu bits from PDCR1 ; valid values are -2,-1,0,1,2
 *      
 * \param num_pixels ; number of pixels in the tables
 *
 * \param prnu ; prnu correction factors to encode into the hardware LUT
 *
 * \param dsnu ; dsnu correction factors to encode into the hardware LUT
 *
 * \param prnudsnu_lut ; prnu[] and dsnu[] will be encoded into this byte array;
 *
 * It is the caller's responsibility to make sure prnudsnu_lut[] is large enough
 * for all the pixels in the input.
 *
 * \author David Poole
 * \date 04-Mar-2008
 *
 */

void
pic_prnudsnu_encode_lut( int total_bits, int extra_prnu_bits,
                         int num_pixels,
                         uint16_t prnu[], uint16_t dsnu[], uint8_t quad[],
                         uint8_t prnudsnu_lut[], int prnudsnu_lut_num_entries )
{
    int i;
    uint32_t num32;
    uint8_t bit;
    int bitpos, curr_bit, curr_byte, curr_bit_of_curr_byte;
    int prnu_bits, dsnu_bits;

    /* davep 10-Aug-2010 ; asic changed. I have no clue how to pack the
     * bits. Restrict all LUTs to 12+12.  
     */
#ifdef HAVE_PIC_PC_QUAD
    /* davep 11-Aug-2010 ; hardwire to 8+12+12 (8 is for quad) */
    XASSERT( total_bits==32, total_bits );
#else
    XASSERT( total_bits==24, total_bits );
#endif
    XASSERT( extra_prnu_bits==0, extra_prnu_bits );


#ifdef HAVE_PIC_PC_QUAD
    prnu_bits = (total_bits-8)/2 + extra_prnu_bits;
    dsnu_bits = (total_bits-8)/2 - extra_prnu_bits;
#else
    prnu_bits = total_bits/2 + extra_prnu_bits;
    dsnu_bits = total_bits/2 - extra_prnu_bits;
#endif

    pic_dbg2( "%s prnu_bits=%d dsnu_bits=%d num_pixels=%d\n", __FUNCTION__,
            prnu_bits, dsnu_bits, num_pixels );

    /* quiet a warning */
    prnu_bits = prnu_bits;

    curr_bit = 0;
    for( i=0 ; i<num_pixels ; i++ ) {

//        num32 = encode_prnudsnu( prnu[i], dsnu[i], prnu_bits, dsnu_bits );
        /* typecast to uint32_t to avoid losing bits if we shift up past 16 */
        num32 = (((uint32_t)prnu[i])<<dsnu_bits) | dsnu[i];
//        num32 = prnu[i];
//        num32 <<= dsnu_bits;
//        num32 |= dsnu[i];

        /* davep 10-Aug-2010 ; stupid endian crap */
        num32 = (num32&0x0000ff)<<16
              | (num32&0x00ff00)
              | (num32&0xff0000)>>16
              ;
#ifdef HAVE_PIC_PC_QUAD
        num32 <<= 8;  /* start with quad of zero */
//        num32 |= 127;  /* add a quad value */
//        uint8_t quad = 0;
//        uint8_t quad = 0x33;
//        quad |= 0x80; /* negative */
        num32 |= quad[i];  /* add a quad value */
#endif

        for( bitpos = total_bits-1 ; bitpos >=0 ; bitpos-- ) {
            bit = (num32 & (1<<bitpos)) && 1;

            curr_byte = curr_bit / 8;
            curr_bit_of_curr_byte = 7 - (curr_bit % 8);

            XASSERT( curr_byte < prnudsnu_lut_num_entries, curr_byte );

            prnudsnu_lut[curr_byte] |= (bit << curr_bit_of_curr_byte);

//            dbg2( "%#lx bitpos=%d bit=%d cb=%d cB=%d cbcB=%d %#x\n", num32,
//                    bitpos, bit, curr_bit, curr_byte, curr_bit_of_curr_byte,
//                    prnudsnu_lut[curr_byte] );

            curr_bit++;
        }

    }

    /* davep 20-Jan-2011 ; XXX temp debug */
    scanlog_hex_dump( (unsigned char *)prnudsnu_lut, 128 );
}

//--------------------------------------------------------------------------
// Function     : pic_prnudsnu_set_color_counter
//   returns    : 
//   arg1       :
//   arg2       :
// Created by   : Brad Smith
// Date created : 5/13/05
// Description  : This routine is part of the bulb shadow correction logic.
//                From the HLD document: "When color_counter=1, then color0,
//                color1 and color2 register settings are used for color0/mono,
//                color1 and color2, respectively. When color_counter=0, color0
//                register settings are used for all even pixels and color 1 register
//                settings are used for all odd pixels."
//
// 
// Notes        : 
//
//--------------------------------------------------------------------------
void pic_prnudsnu_set_color_counter( bool odd_even )
{
#if 0    
    if (odd_even)
        prnudsnu_regs->PDCR2 &= ~PIC_PC_PDCR2_CC;
    else
        prnudsnu_regs->PDCR2 |= PIC_PC_PDCR2_CC;
#endif
    dbg1("%s:STUB - need to implement\n", __func__);
}


/**
 * \brief  
 *
 *
 * \author David Poole
 * \date 11-Aug-2010
 *
 */

void pic_prnudsnu_set_quadratic_config( uint32_t scale_factor, uint32_t limit_factor, uint32_t shift_factor )
{
#if 0    
#ifdef HAVE_PIC_PC_QUAD
    pic_dbg2( "%s SF1=%d SF2=%d shift=%d\n", __FUNCTION__, scale_factor, limit_factor, shift_factor );

    /* set the scale factor, hardwire the rest */
    prnudsnu_regs->QUAD = 0; /* default value */

    prnudsnu_regs->QUAD = PRNUDSNU_CORR_QUAD_SF1_REPLACE_VAL( prnudsnu_regs->QUAD, scale_factor );
    prnudsnu_regs->QUAD = PRNUDSNU_CORR_QUAD_SF2_REPLACE_VAL( prnudsnu_regs->QUAD, limit_factor);
    prnudsnu_regs->QUAD = PRNUDSNU_CORR_QUAD_Q_SHIFT_REPLACE_VAL( prnudsnu_regs->QUAD, (shift_factor&0x07));
#endif
#endif
    dbg1("%s:STUB - need to implement\n", __func__);
}

/**
 * \brief dump just the prnu/dsnu stuff
 *
 * \author David Poole
 * \date 25-Jul-2011
 *
 */

void pic_prnudsnu_dump( uint8_t pic_instance )
{
    // NOTE: dumps all the PRNU/DSNU registers
    pic_pd_dump(pic_instance);
}

/**
 * \brief  calculate and return a PIC Hscale factor 
 *
 * Given pixels_in and how many pixels we want out, calculate the HScale factor
 * necessary.
 *
 * \author David Poole
 * \date 12-Nov-2007
 *
 */

uint32_t pic_hs_calc_factor( uint32_t pixels_in, uint32_t pixels_out )
{
    uint32_t hscale_factor;
    int extra_pixels;
    int numer, denom;

    /* from the PIC Hscale MA:
     * "P = number of pixels to add or subtrace (positive to add, negative to
     * subtract"
     *
     * I'm using extra_pixels as "P"
     *
     * Note extra_pixels can be negative!!!!
     */
    extra_pixels = pixels_out - pixels_in;

    XASSERT( pixels_in + extra_pixels > 0, extra_pixels );

//    hscale_factor = (int)( 65536 * (f_line_length + extra_pixels)/f_line_length );

    /* can't use safeint because we want need an accurate intermediate result
     * in order to get the fractional 65536
     */
    numer = (pixels_in + extra_pixels) << 8;
    denom = pixels_in << 8;
    hscale_factor = ((numer*(65536<<8))/denom)>>8;

//    error = abs( (f_line_length + extra_pixels)/f_line_length -
//                  hscale_factor/65536 )

    return hscale_factor;
}

/**
 * \brief  Calculate a DMA aligned output pixel size based on scale and input
 * pixel size
 *
 * Originally created to handle hacking a 75/150 DPI into ICE Lite based
 * products. ICE Lite can't XY scale color.
 *
 *
 * \author David Poole
 * \date 01-Apr-2009
 *
 */

void pic_hs_calc_row( int pixels_per_row_in, int x_numer, int x_denom, 
                      int bits_per_pixel, 
                      int *pixels_per_row_out )
{
    int ppr, bytes_per_row;

    XASSERT( bits_per_pixel==8||bits_per_pixel==16, bits_per_pixel );

//    /* add 0.5 to round up */
//    ppr = (uint32_t)(((float)(pixels_per_row_in * x_numer) / (float)x_denom)+0.5);
    ppr = safeint_scale_int( pixels_per_row_in, x_numer, x_denom );

    /* convert to bytes per row, then round up for DMA alignment */
    bytes_per_row = ICE_DMA_ALIGN_ME( ppr * (bits_per_pixel/8) );

    /* convert back to pixels */
    ppr = bytes_per_row / (bits_per_pixel/8);

    *pixels_per_row_out = ppr;

    dbg2( "%s in=%d scale=%d/%d out=%d\n", __FUNCTION__, 
                pixels_per_row_in, x_numer, x_denom, *pixels_per_row_out );
}

/**
 * \brief  Reset the BDR to power-on defaults
 *
 *
 * \author David Poole
 * \date 26-Mar-2010
 *
 */

#if 0    
void pic_bdr_reset( void )
{
    uint32_t i;
    volatile uint32_t *bdr_lut;

    *PIC_BDR_BDRCR = PIC_BDR_BDRC_R;

    // Reset the BDR LUT
    *PIC_BDR_BDRCR |= PIC_BDR_BDRCR_CPU;
    bdr_lut = PIC_BDR_LUT;
    for (i=0 ; i<PIC_BDR_LUT_SIZE; i++ ) {
        *bdr_lut++=0;
    }

#ifdef HAVE_THREE_BDR_LUT
    /* davep 24-Mar-2010 ; Adding support for new three LUT BDR;
     * initialize the other two luts 
     */
    pic_bdr_select_lut( PIC_BDR_BDRCR_LUTSEL_COLOR1 );
    bdr_lut = PIC_BDR_LUT;
    for (i=0 ; i<PIC_BDR_LUT_SIZE; i++ ) {
        *bdr_lut++=0;
    }
    pic_bdr_select_lut( PIC_BDR_BDRCR_LUTSEL_COLOR2 );
    bdr_lut = PIC_BDR_LUT;
    for (i=0 ; i<PIC_BDR_LUT_SIZE; i++ ) {
        *bdr_lut++=0;
    }
#endif

    // Reset BDR block again
    *PIC_BDR_BDRCR = PIC_BDR_BDRC_R;
    dbg1("%s:STUB - need to implement\n", __func__);    
}
#endif

/**
 * \brief  Calculate the pixels-per-row flowing through PIC.
 *
 * \author David Poole
 * \date 01-Mar-2013
 */

scan_err_t pic_calc_pixels_per_row( const struct scanvars *sv, 
                                    struct pic_pixels_per_row *p,
                                    uint32_t scan_pixels_per_row_in  )
{
    scan_err_t scerr;
    int num32;
    struct scan_sensor_chipgap sensor_gaps;

    /* WARNING: as of this writing, this function is a bit of a work in progress.
     *   While the numbers calculated here look right, most of them are not
     *   actually used.  For example:
     *     - margins tend to always be set to the scan area
     *     - disturbing the DMA alignment of width will end badly
     *   So the goal is not to change the width, LR margin 0 sets it
     *   as needed and LR margin 1 will enforce it at the end.
     */

    dbg2( "%s scan_ppr_in=%d\n", __FUNCTION__, scan_pixels_per_row_in );

#if !defined HAVE_PIC_CHIPGAP
    if( p->use_chipgap ) {
        /* no chipgap support in this ASIC */
        ASSERT(0);
        /* use scerr here to quiet a warning */
        scerr = SCANERR_NOT_IMPLEMENTED;
        return scerr;
    }
#else
    memset( &sensor_gaps, 0, sizeof(struct scan_sensor_chipgap) );
    scerr = scansen_get_chipgap_conf( sv->hw_dpi_horiz, &sensor_gaps );
    if( scerr != SCANERR_NONE ) {
        dbg2( "%s no chipgap entry for dpi=%d\n", __FUNCTION__, sv->hw_dpi_horiz );
        p->use_chipgap = false;
    }
#endif

    p->pic_pixels_per_row_in = scan_pixels_per_row_in;

    p->lrmargins0_ppr_in = scan_pixels_per_row_in;
    if( p->use_lrmargins0 ) {
        /* margin's output is the scan area */
        p->lrmargins0_ppr_out = sv->scan_area_pixels.width;
    }
    else {
        p->lrmargins0_ppr_out = p->lrmargins0_ppr_in;
    }

    p->chipgap_ppr_in = p->lrmargins0_ppr_out;
    if( p->use_chipgap ) {
        /* pixels out will be pixels in plus all the gap repair pixels we'll add 
         * FIXME: this should probably be the chipgap pixels to be added WITHIN
         *        the LR margin 0 area (not the entire sensor area). If using LR 
         *        margin 1 it won't matter ... the extras will get chopped off.
         */
        p->chipgap_ppr_out = p->chipgap_ppr_in + 
                    pic_chipgap_sum_list( sensor_gaps.chipgap_list,
                                          sensor_gaps.num_chipgap_list );
    }
    else {
        p->chipgap_ppr_out = p->chipgap_ppr_in;
    }

    p->hscale_ppr_in = p->chipgap_ppr_out;
    if( p->use_hscale ) {
        /* FIXME: We primarily used hscale for magnification error correction on
         *   CMOS sensors, don't do that much anymore. pic_hs_calc_row does re-align
         *   things for DMA, which could change things such that our pipe will stall.
         */
        ASSERT(0);
        pic_hs_calc_row( p->chipgap_ppr_out, 
                        sv->pic_scale.x_numer, sv->pic_scale.x_denom, 
                        sv->pic_bpp, &num32 );
        p->hscale_ppr_out = (uint32_t)num32;
    }
    else {
        p->hscale_ppr_out = p->hscale_ppr_in;
    }

    p->lrmargins1_ppr_in = p->hscale_ppr_out;
    if( p->use_lrmargins1 ) {
        /* margin's output is the scan area */
        p->lrmargins1_ppr_out = sv->scan_area_pixels.width;
    }
    else {
        p->lrmargins1_ppr_out = p->lrmargins1_ppr_in;
    }

    p->pic_pixels_per_row_out = p->lrmargins1_ppr_out;

    dbg2( "%s pic_ppr_out=%d\n", __FUNCTION__, p->pic_pixels_per_row_out );

    /* stupid human check */
    XASSERT( p->pic_pixels_per_row_out < 65536, p->pic_pixels_per_row_out );

    return SCANERR_NONE;
}

int pic_debug_msg( uint32_t param2, void *param3 )
{
    /* parse the rest of the message fields */

    switch( param2 ) {
        case SCAN_DBGMSG_PIC_DUMP :
            pic_dump_common_regs(0); // FIXME, pass the pic_instance here
            break;

        default :
            XASSERT( 0, param2 );
            /* ignore */
    }

    return 0;
}

