/*
**************************************************************************
This Source Code Form is subject to the terms of the Mozilla Public
License, v. 2.0. If a copy of the MPL was not distributed with this file,
You can obtain one at http://mozilla.org/MPL/2.0/.

Copyright (c) 2014-2016, Marvell International Ltd.

Alternatively, this software may be distributed under the terms of the GNU
General Public License Version 2, and any use shall comply with the terms and
conditions of the GPL.  A copy of the GPL is available at
http://www.gnu.org/licenses/old-licenses/gpl-2.0.html

THE FILE IS DISTRIBUTED AS-IS, WITHOUT WARRANTY OF ANY KIND, AND THE
IMPLIED WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE
ARE EXPRESSLY DISCLAIMED.  The GPL license provides additional details about
this warranty disclaimer.
******************************************************************************
*/


/**
 * \file icetest.c
 *
 * \brief Functions to interface to the ICE Test hardware block.
 *
 * ICE Test is a DMA interface that will feed data into the top of the Scan
 * block or CISX.
 *
 */

#include <stdint.h>
#include <stdbool.h>
#include <string.h>

#include "scos.h"

#include "lassert.h"
#include "regAddrs.h"
#include "interrupt_api.h"
#include "cpu_api.h"
#include "memAPI.h"
#include "list.h"

#include "scancore.h"
#include "scantypes.h"
#include "scandbg.h"
#include "safetylock.h"
#include "icedma.h"
#include "safetylock.h"
#include "ddma.h"

#include "icetest.h"
#include "icetest_if.h"

#include "ICE_test_dma_regmasks.h"
#include "ICE_test_dma_regstructs.h"
#include "scanalyzer.h"

#include "dros.h"

//#define MAX_ICETEST_IDMA_DESCRIPTORS 48
#define MAX_ICETEST_IDMA_DESCRIPTORS 16


//#define ICETEST_DMA_ISR_DEBUG

#ifdef ICETEST_DMA_ISR_DEBUG
#define isr_dbg2 dbg2
#else
#define isr_dbg2(...)
#endif

/* power-on value for some registers as of 28-Mar-2011 */

#define ICE_TEST_IDMA_CFG_R TESTIDMA_CFG_REPLICATE_MASK 

#define ICE_TEST_IDMA_STATUS_R ( TESTIDMA_STATUS_PACKER_EMPTY_MASK \
                               | TESTIDMA_STATUS_EMPTY_DBUF_MASK \
                               | TESTIDMA_STATUS_EMPTY_CBUF_MASK \
                               | TESTIDMA_STATUS_SOFTRESET_MASK \
                              )

struct {
    int num_interrupts;
    
    int channel_isr[ICETEST_IDMA_NUM_CHANNELS];

} icetest_interrupt_stats;

dros_sem_t icetest_fin_sem;

static struct ddma_channel icetest_desc_channel;

static struct ice_dma_mm icetest_idma_mm;
static struct ice_dma_channel icetest_idma_channel;

static struct ice_dma_driver icetest_idma_driver = { 
    .msg_data = SMSG_ICETEST_IDMA_DATA,

    /* set the queue depth to a number which, I hope, will keep our chained
     * descriptors far enough a head of the running DMA
     */
    .max_queue_depth = 10,

    .reset =      icetest_idma_channel_reset,
    .enable =     icetest_idma_channel_enable,
    .disable =    icetest_idma_channel_disable,
    .load =       icetest_idma_channel_load, 
    .start =      icetest_idma_channel_start,   /* start; can't use because called for each load */
    .is_enabled = icetest_idma_channel_is_enabled,
    .icebuf_isr = NULL,  /* dma_isr */
};

static void channel_isr( struct ddma_channel *dch )
{
    struct ddma_descriptor *curr;

    /* 
     * 
     *
     * davep 04-Apr-2011 ; TODO move to ddma.c 
     *
     *
     */

    isr_dbg2( "%s\n", __FUNCTION__ );

    /* drop a log on entry */
//    SCANALYZER_LOG(LOG_ICETEST_CHANNEL_ISR,(1<<16)|dch->channel);

    icetest_interrupt_stats.channel_isr[dch->channel]++;

    /* use a convenience pointer; remember we still have to modify the list
     * itself
     */
    curr = dch->head_desc;
    cpu_dcache_invalidate_region( curr, sizeof(struct ddma_descriptor) );

    /* davep 11-Oct-2010 ; add safety check to make sure the hardware is far
     * enough behind us we won't collide ("listening for the bulldozers")
     */
    XASSERT( DESCRIPTOR_OWNED_BY_BLOCK((struct ddma_descriptor *)curr->fw_prev_descriptor_addr), 
            (uint32_t)curr->fw_prev_descriptor_addr );

    /* we could have more than one completed descriptor so we'll walk the list
     * until we find a descriptor with own bit set
     */
    while( !DESCRIPTOR_OWNED_BY_BLOCK( curr ) )
    {
//        dbg2( "%s %d %p %p %#x %#x\n", __FUNCTION__, dch->channel, curr, 
//                    curr->fw_next_descriptor_addr, curr->config_flags );

        XASSERT( (curr->config_flags & DDMA_DESCRIPTOR_CONFIG_OWNER_BLOCK)==0, 
                    curr->config_flags );

        /* davep 27-Apr-2010 ; XXX temp debug */
        if( ! (curr->config_flags & DDMA_DESCRIPTOR_CONFIG_INT) ) {
            scanlog_hex_dump( (unsigned char *)(curr)-32, 64 );
            ddma_descriptor_list_dump( &dch->desc_list );
        }
        XASSERT( curr->config_flags & DDMA_DESCRIPTOR_CONFIG_INT, 
                    curr->config_flags ); 

        /* call the default method to do the buffer dance */
        ice_dma_isr( &icetest_idma_mm, dch->channel );

        curr->src_addr = 0;

        /* Set the "own" bit back on the descriptor so I know I've already
         * handled this buffer. We're in a circular list so need to know when
         * to stop.
         */
        curr->config_flags |= DDMA_DESCRIPTOR_CONFIG_OWNER_BLOCK;

        ddma_desc_flush( curr );

        /* move head of list to next element */
        dch->head_desc = (struct ddma_descriptor *)curr->fw_next_descriptor_addr;

        dch->num_running -= 1;
        XASSERT( dch->num_running >=0, dch->channel );

        curr = dch->head_desc;
        cpu_dcache_invalidate_region( curr, sizeof(struct ddma_descriptor) );

        ddma_channel_sanity( dch );
    }

    // 28-Jul-2014: It is possible that the DMA completed before getting to this
    //   point.  Call launch here to get things running again (it does nothing if
    //   the DMA is still running).
    if (dch->num_running > 0)
    {
        icetest_idma_late_launch();
    }

    /* drop another at exit */
//    SCANALYZER_LOG(LOG_ICETEST_CHANNEL_ISR,dch->channel);
}

// callback from the icetest idma driver interrupt
static void icetest_dma_interrupt(struct icetest_idma_interrupt_info *intstruct)
{
    struct ddma_channel *dch;
    struct icetest_idma_status_info status;
    dros_err_t dros_err;
    

    /* drop a log on entry */
//    SCANALYZER_LOG(LOG_ICETEST_ISR,(ipend<<16)|dma_ipend);

    icetest_interrupt_stats.num_interrupts++;

    isr_dbg2( "%s %#x \n", __func__, intstruct->debug_intarray);

    /* fail on bad stuff we don't ever want to see */
    XASSERT((intstruct->who_int == false), intstruct->debug_intarray);

    /* if we get this far, we should have something to handle */
    XASSERT(intstruct->debug_intarray, intstruct->debug_intarray);

    icetest_get_idma_status(&status);

    SCANALYZER_LOG(LOG_ICETEST_INTERRUPT, intstruct->debug_intarray); 
    SCANALYZER_LOG(LOG_ICETEST_STATUS, status.debug_array); 

    // check for a pending "Transfer End" interrupt

    if (intstruct->fin_int)
    {
        dch = &icetest_desc_channel;
        if (dch->is_open)
        {
            channel_isr(dch);

            /* davep 13-Oct-2010 ; if we've drained our chain */
            if (dch->num_running==0)
            {
                /* make sure block is truly idle */
                XASSERT(!(status.dma_busy), status.debug_array);
            }
        } 
        else
        {
            /* assume we're doing a one-shot (probably via icetest_idma_desc_write())
             * so do some extra sanity checks here
             */
            XASSERT(!(status.dma_busy), status.debug_array);

            // post the semaphore for a transfer end
            dros_err = dros_sem_post(&icetest_fin_sem);
            if (DROSERR_NONE != dros_err)
            {
                errprint("ERROR! posting semaphore icetest_fin_sem returned error=%d\n", dros_err);
                ASSERT(0);
            }
        }
    }

    /* drop another log on exit log on entry */
//    SCANALYZER_LOG(LOG_ICETEST_ISR,0);
}

// callback from the icetest common driver interrupt
static void icetest_interrupt(struct icetest_interrupt_info *intdata)
{
#if 0
    icetest_dump_regs();    
    icetest_dump_idma_regs();
#endif	
    /* obviously temp debug */
    isr_dbg2("%s %#x\n", __FUNCTION__, intdata->debug_intarray);

    /* fail on unexpected interrupts (stuff we don't ever want to see) */
    if ((intdata->underflow_int) || (intdata->overflow_int))
    {
        //XASSERT(0, intdata->debug_intarray); FIXME!!!

        // ballen 02/20/15 REVISIT TODO
        // Temporarily removed, pending confirmation that the ASIC logic is wonky and this fires at times that it shouldn't.
        // The pixels are getting through, but this comes on periodically.
        //errprint("PLEASE NOTE!!! %s got overflow or underflow int!! 0x%X\n",__func__, intdata->debug_intarray);
    }
}

void icetest_interrupt_disable(void)
{
    isr_dbg2("%s\n", __FUNCTION__);

    /* disable interrupt(s) */
    icetest_disable_irq(NULL); // disable all ints
}
void icetest_interrupt_enable( void )
{
    isr_dbg2("%s\n", __FUNCTION__);

    /* turn on everything */
    icetest_enable_irq(NULL);  // enable all icetest common ints

    icetest_enable_idma_irqs(NULL);  //enable all icetest idma ints
}

/*
 * ICE Test IDMA
 */

void icetest_idma_reset( void )
{
    struct icetest_idma_config_info info;

    memset(&info, 0, sizeof(struct icetest_idma_config_info)); // mark all bits invalid

    icetest_idma_soft_reset(1);

    /* continuing in my long tradition of 1 usec sleeps on hardware reset */
    cpu_spin_delay(1);

    icetest_idma_soft_reset(0);

    icetest_set_idma_linewidth(0);

    /* ack any possible interrupts */
    icetest_clear_idma_irqs(NULL); //clear all interrupts

    icetest_enable_idma_irqs(NULL);

    // set default value of config - replicate only set
    info.replicate_valid = true;
    info.replicate = 1;
    icetest_set_idma_config(&info);

    /* our dma descriptors must be padded to a cache line size so we can flush
     * them from cache without breaking neighboring cache lines
     */
    XASSERT(sizeof(struct ddma_descriptor)==cpu_get_dcache_line_size(), 
            sizeof(struct ddma_descriptor));
}

uint32_t icetest_idma_interrupt_disable(void)
{
    isr_dbg2("%s\n", __FUNCTION__);

    icetest_disable_idma_irqs(NULL);  //disable all icetest idma ints
    return 0;  // FIXME, what to do
}

void icetest_idma_interrupt_enable(void)
{
    struct icetest_idma_interrupt_info intstruct;

    memset(&intstruct, 0, sizeof(struct icetest_idma_interrupt_info));

    /* enable selected error interrupts */
    intstruct.eoi_int = true; // end of image
    intstruct.fin_int = true; // transfer end
    intstruct.who_int = true; // incorrect owner (this is bad)
    
    icetest_enable_idma_irqs(&intstruct);
}

void icetest_idma_disable(void)
{
    isr_dbg2("%s\n", __FUNCTION__);

    icetest_set_idma_enable(0);  // clear config enable bit
    icetest_idma_interrupt_disable();
}

void icetest_idma_enable(void)
{
    isr_dbg2("%s\n", __FUNCTION__);

    icetest_set_idma_enable(1);  // set config enable bit
    icetest_idma_interrupt_enable();
}

void icetest_idma_soft_setup(void)
{
    icetest_set_idma_dma_width(5);
    icetest_set_idma_burstlen(2);

    /* davep 25-May-2011 ; copying unit test settings */
    icetest_set_idma_replicate(0);
}

void icetest_idma_desc_write( struct ddma_descriptor *desc, uint32_t line_width_bytes )
{
    struct icetest_idma_config_info config;
    struct icetest_idma_status_info status;

    /* davep 10-Feb-2012 ; I /think/ transfer_len_bytes is limited to 24-bits */
    XASSERT((desc->transfer_len_bytes & 0xff000000)==0, desc->transfer_len_bytes);

    /* is this thing on? */
    icetest_get_idma_config(&config);
    XASSERT(config.enable, config.debug_array);

    XASSERT(DESCRIPTOR_OWNED_BY_BLOCK(desc), desc->config_flags);

    /* assume we're doing a one-shot so make sure we're idle */
    icetest_get_idma_status(&status);
    XASSERT(!(status.dma_busy), status.debug_array);

    ddma_desc_flush(desc);

    /* davep 09-Feb-2012 ; sanity check we're not overflowing the max width */
    XASSERT((line_width_bytes & ~TESTIDMA_LINE_WIDTH_LINE_WIDTH_MASK )==0, 
            line_width_bytes);
    
    /* davep 27-May-2011 ; line_width is actually a don't care */ //  TRUE?  FIXME
    icetest_set_idma_linewidth(line_width_bytes/4); // must be in samples per icetest html doc

    XASSERT(desc->dma_ptr_self, (uint32_t)desc);
    ASSERT(desc->src_addr);
    ASSERT(desc->fw_src_addr);
    
    // Shotgun debug
    // ddma_desc_dump(desc);
    // icetest_dump();

    icetest_start_idma(desc->dma_ptr_self);
}


// to make sure a semaphore is empty, do trywaits until
// we get a DROSERR
static void pic_clear_sem(dros_sem_t *pic_sem)
{
    dros_err_t dros_ret = DROSERR_NONE;
    
    while(DROSERR_NONE == dros_ret)
    {
        dros_ret = dros_sem_trywait(pic_sem);
    }
}

void icetest_wait_for_fin( void )
{
    int i;
    dros_err_t dros_err;
    struct icetest_idma_status_info info;
    
    // Blocking wait for icetest FIN. Used by test/debug code. 

    dros_err = dros_sem_wait(&icetest_fin_sem);  // FIXME - do we want a timed wait?
    if (DROSERR_NONE != dros_err)
    {
        errprint("uh oh, waiting on semaphore icetest_find_sem returned error=%d\n", dros_err);
        ASSERT(0);
    }

    icetest_get_idma_status(&info);
    SCANALYZER_LOG(LOG_ICETEST_STATUS, info.debug_array); 

    /* davep 20-May-2013 ; looks like icetest not idle after the transfer
     * complete interrupt. Poll waiting for the empty flags to be set.
     */
    for (i=0 ; i<10 ; i++)
    {
        icetest_get_idma_status(&info);
        SCANALYZER_LOG(LOG_ICETEST_STATUS, info.debug_array);
        if (info.packer_empty && info.empty_dbuf && info.empty_cbuf)
        {
            break;
        }
        scos_sleep_milliseconds(100);
    }
    /* if we hit this assert, we timed out waiting for ICETest to go idle */
    XASSERT(info.packer_empty && info.empty_dbuf && info.empty_cbuf, info.debug_array);
    SCANALYZER_LOG(LOG_ICETEST_STATUS, info.debug_array); 
}

void icetest_idma_desc_write_and_wait(struct ddma_descriptor *desc, uint32_t line_width_bytes)
{
    // make sure the fin semaphore is empty
    pic_clear_sem(&icetest_fin_sem);

    /* ack all interrupts before we start poking the block; assumes ICETest
     * interrupt disabled!
     */
    icetest_clear_idma_irqs(NULL);

    icetest_idma_desc_write(desc, line_width_bytes);

    icetest_wait_for_fin();
}


scan_err_t icetest_idma_channel_open( uint8_t channel )
{
    scan_err_t scerr;
    struct ddma_channel *dch;
    char name[DDMA_NAME_LEN+1];

    XASSERT( channel==0, channel );

    dch = &icetest_desc_channel;
    
    strncpy( name, "icetestidma ", DDMA_NAME_LEN );
    name[11] = '0' + channel;
    ddma_channel_open( dch, name, channel );

    scerr = ddma_channel_alloc_descriptors( dch, MAX_ICETEST_IDMA_DESCRIPTORS );
    if( scerr != 0 ){
        /* failure! oh woe! free the channel memory we've already allocated */
        ddma_channel_free_descriptors( dch );
        ddma_channel_close( dch );

        return SCANERR_OUT_OF_MEMORY;
    }
    return SCANERR_NONE;
}

void icetest_idma_channel_close( uint8_t channel )
{
    struct ddma_channel *dch;

    XASSERT( channel==0, channel );
    dch = &icetest_desc_channel;

    XASSERT( dch->channel==channel, dch->channel );
    XASSERT( dch->num_running==0, dch->num_running );

    ddma_channel_free_descriptors( dch );
    ddma_channel_close( dch );
}

int icetest_idma_open( uint8_t channels[],
                   uint8_t num_channels,
                   int num_empties,
                   int total_rows,
                   int bytes_per_row )
{
    scan_err_t scerr;
    int i;

    XASSERT( num_channels==1, num_channels );
    for( i=0 ; i<num_channels; i++ ) {

        scerr = icetest_idma_channel_open( channels[i] );
        if( scerr != 0 ){
            /* failure! oh woe! free the channel memory we've already allocated */
            i -= 1;
            while( i>=0 ) {
                icetest_idma_channel_close( channels[i] );
                i -= 1;
            }

            /* this function returns the number of icebufs allocated */
            return 0;
        }
    }

    return ice_dma_open( &icetest_idma_mm, channels,
                    num_channels, num_empties,
                    total_rows, bytes_per_row, ICEBUF_TAG_ICETEST_IDMA );
}

scan_err_t icetest_idma_add_buffer( uint8_t channel, uint8_t *data, uint32_t datalen,
                     uint32_t rows, uint32_t bytes_per_row )
{
    return ice_dma_add_buffer( &icetest_idma_mm, 
            channel, data, datalen, rows, bytes_per_row );
}

void icetest_idma_add_ready( uint8_t channel, struct ice_dma_buffer **addme )
{
    /* Don't touch num_rows here (like we do in other _add_ready() methods.
     * Because icetest is a input DMA, I'm assuming the buffer we're adding is
     * full of data to be pushed into ICEtest.
     */

    ice_dma_add_ready( &icetest_idma_mm, channel, addme );
}

void icetest_idma_channel_launch( uint8_t channel )
{
    dbg2( "%s ch=%d\n", __FUNCTION__, channel );
    ice_dma_channel_launch( &icetest_idma_mm, channel );
}

void icetest_idma_close( void )
{
    struct ddma_channel *dch;

    dbg2( "%s\n", __FUNCTION__ );

    dch = &icetest_desc_channel;

    if( dch->is_open  ) {
        icetest_idma_channel_close( dch->channel );
    }

    ice_dma_close( &icetest_idma_mm );
}

void icetest_idma_channel_reset( uint8_t channel )
{
    XASSERT( channel==0, channel );

    icetest_idma_reset();
}

void icetest_idma_channel_enable( uint8_t channel ) 
{
    isr_dbg2("%s channel=%d\n", __FUNCTION__, channel);

    XASSERT( channel==0, channel );

    icetest_idma_enable();
}

void icetest_idma_channel_disable( uint8_t channel ) 
{
    isr_dbg2("%s channel=%d\n", __FUNCTION__, channel);

    XASSERT( channel==0, channel );

    icetest_idma_disable();
}

void icetest_idma_channel_load( uint8_t channel, uint8_t *data_ptr, dma_addr_t dma_dest, 
                                uint32_t rows, uint32_t bytes_per_row )
{
    /* 
     * BIG FAT NOTE! 
     *
     * This can be called from interrupt context.
     *
     */

//    dbg2( "%s %#x %d %d\n", __FUNCTION__, dma_dest, rows, bytes_per_row );

    /* poke the bytes_per_row into the line_width (should be the same for all
     * load calls for this scan but I won't know the value until load is
     * called)
     */
    icetest_set_idma_linewidth(bytes_per_row);

    /* davep 02-Jul-2013 ; ddma needs the CPU pointer (ddma does the Linux
     * kernel dma mapping)
     */
    ddma_channel_load( &icetest_desc_channel, data_ptr, rows, bytes_per_row );
}

void icetest_idma_channel_start( uint8_t channel )
{
    /* no op; see also icetest_idma_late_launch() */
}

void icetest_idma_late_launch(void)
{
    struct ddma_channel *dch;
    struct icetest_idma_status_info info;    

    dch = &icetest_desc_channel;

    XASSERT(dch->is_open, dch->channel);
    XASSERT(icetest_idma_channel_is_enabled(dch->channel), dch->channel);

    icetest_get_idma_status(&info);
    if (info.dma_busy)
    {
        /* DMA already running, no need to start it again */
        dbg2( "%s DMA already running, no need to start it again\n", __FUNCTION__ );
        return;
    }

    XASSERT(dch->num_running > 0, dch->num_running);
    XASSERT(dch->head_desc != NULL, dch->channel);
    XASSERT(DESCRIPTOR_OWNED_BY_BLOCK(dch->head_desc), dch->head_desc->config_flags);

    /* davep 27-May-2011 ; shotgun debug ; fiddle the flags to match ASIC
     * team's unit test
     */
//    dch->head_desc->config_flags |= DDMA_DESCRIPTOR_CONFIG_SOI; 
//    ddma_desc_flush( dch->head_desc );
//    dch->tail_desc->config_flags |= DDMA_DESCRIPTOR_CONFIG_EOI;  
//    ddma_desc_flush( dch->tail_desc );

    /* load head of descriptor list into the idma */
    ddma_desc_flush(dch->head_desc);
    /* davep 25-Apr-2012 ; adding vma/pma destinction */
    XASSERT( dch->head_desc->dma_ptr_self, (uint32_t)dch->head_desc );

    ASSERT(dch->head_desc->src_addr);

    icetest_start_idma(dch->head_desc->dma_ptr_self);
}

bool icetest_idma_channel_is_enabled(uint8_t channel)
{
    struct icetest_idma_config_info info;

    XASSERT(channel==0, channel);
        
    icetest_get_idma_config(&info);
    return (info.enable) && true;
}

void icetest_idma_cancel( void )
{
    struct ddma_channel *dch;

    dch = &icetest_desc_channel;

    if( dch->is_open ) {
        icetest_idma_channel_reset( dch->channel );
        dch->num_running = 0;
    }

    /* now call the parent method */
    ice_dma_cancel( &icetest_idma_mm );
}

void icetest_idma_sanity( void )
{
    struct ddma_channel *dch;

    dch = &icetest_desc_channel;

    if( dch->is_open ) {
        ddma_channel_sanity( dch );
    }

    ice_dma_sanity( &icetest_idma_mm );
}

int icetest_idma_setup_buffers( uint8_t channel, int num_buffers, 
                            int rows_per_buffer, int bytes_per_row )
{
    return ice_dma_setup_buffers( &icetest_idma_mm, channel, num_buffers,
                    rows_per_buffer, bytes_per_row, icetest_idma_add_buffer );
}

void icetest_idma_scanpipe_init( void ) 
{
    dros_err_t dros_err;
        
    ice_dma_init( &icetest_idma_mm, "icetestidma", &icetest_idma_driver, 
            &icetest_idma_channel, 1 );

    dros_err = dros_sem_init(&icetest_fin_sem, "icetest_fin", 0);
    if (dros_err != DROSERR_NONE)
    {
        errprint("uh oh, init semaphore icetest_fin_sem returned error=%d\n", dros_err);
        ASSERT(0);
    }
}

/*
 * ICE Test Config
 */

scan_err_t icetest_onetime_init( void )
{
    icetest_reset();

    /* sanity check to make sure the firmware pointing to correct address(s) */
//    XASSERT( icetest_idma_regs->status==ICE_TEST_IDMA_STATUS_R, icetest_idma_regs->status );
//    XASSERT( icetest_idma_regs->cfg==ICE_TEST_IDMA_CFG_R, icetest_idma_regs->cfg );

    /* hook interrupt */
    icetest_interrupt_disable();

    icetest_register_callback(icetest_interrupt);
    icetest_register_callback_idma(icetest_dma_interrupt);    

    //icetest_rtos_init();

    SCANALYZER_ENABLE_LOG(LOG_ICETEST_INTERRUPT);
    SCANALYZER_ENABLE_LOG(LOG_ICETEST_STATUS);

    return SCANERR_NONE;
}

scan_err_t icetest_cleanup_module( void )
{
//    icetest_release_interrupt();

    return SCANERR_NONE;
}

scan_err_t icetest_soft_setup(void)
{
    icetest_reset(); // reset icetest and disable icetest interrupts
    /* turn on all icetest common interrupts */
    icetest_enable_irq(NULL);
	// needed?
   //icetest_regs->TestConfig = ICE_TEST_TESTCONFIG_PICRATE_REPLACE_VAL(icetest_regs->TestConfig, 128 )
    icetest_idma_soft_setup();
    return SCANERR_NONE;
}

scan_err_t icetest_set_test_mode( uint8_t pic_instance, icetest_mode_t mode )
{
    struct icetest_testconfig_info info;

    /* Selects the destination for ICE_test DMA data. Only valid when TestEn is
     * also set. 
     *
     * 0 : ICE_test DMA data is driven into the top of the SCAN block
     * 1 : ICE_test DMA data is driven into the top of the CISX block (note
     * that by putting CISX in bypass, the data can also effectively be driven
     * into the top of PIC)
     */
    memset(&info, 0, sizeof(struct icetest_testconfig_info));  // mark all bits invalid

    info.icetest_mode_select_valid = true;
    switch( mode ) {
        case ICETEST_CONFIG_DATA_TO_SCAN :
            info.icetest_mode_select = ICETEST_SEL_SCAN;
            break;

        case ICETEST_CONFIG_DATA_TO_CISX :
            if (pic_instance == 0)
                info.icetest_mode_select = ICETEST_SEL_CISX0;
            if (pic_instance == 1)
                info.icetest_mode_select = ICETEST_SEL_CISX1;
            break;

        default :   
            XASSERT( 0, mode );
            return SCANERR_INVALID_PARAM;
    }
    icetest_set_config(&info);
    return SCANERR_NONE;
}

void icetest_enable(bool enable)
{
    struct icetest_testconfig_info info;

    isr_dbg2("%s enable=%d\n", __FUNCTION__, enable);

    if (enable)
    {
        icetest_interrupt_enable();
    } 
    else
    {
        icetest_interrupt_disable();
    }

    memset(&info, 0, sizeof(struct icetest_testconfig_info));  // set all fields invalid

    info.icetest_enable_valid = true;
    info.softreset_active_valid = true;

    /* 0==normal mode  1==test mode */
    info.icetest_enable = (enable?1:0);
    /* 0==reset inactive  1==reset active */
    info.softreset_active = (enable?0:1);

    icetest_set_config(&info);
}

void icetest_reset( void )
{
    struct icetest_testconfig_info info;

    memset(&info, 0, sizeof(struct icetest_testconfig_info));  // set all fields invalid
    
    icetest_idma_reset();

    /* disable interrupt(s) */
    icetest_disable_irq(NULL); // disable all ints

    /* ack any possible pending interrupts */
    icetest_clear_irq(NULL);

    /* set to power-on default config */
    info.pic_rate_valid = true;
    info.disable_pacing_valid = true;
    info.blank_en_valid = true;
    info.softreset_active_valid = true;
    info.icetest_enable_valid = true;
    info.icetest_mode_select_valid = true;
    
    info.pic_rate = 0;
    info.disable_pacing = 1;
    info.blank_en = 0;
    info.softreset_active = 0;
    info.icetest_enable = 0;
    info.icetest_mode_select = 0;
    icetest_set_config(&info);
}

void icetest_dump( void )
{
    dbg2( "%s\n", __FUNCTION__ );

    icetest_dump_regs();

    icetest_dump_idma_regs();

    if (icetest_desc_channel.head_desc)
    {
        ddma_desc_chain_dump(icetest_desc_channel.head_desc,
                              icetest_desc_channel.desc_list.num_descriptors);
    }
//    ddma_data_peek( &icetest_desc_channel );
//    ice_dma_debug_log( &icetest_idma_mm );
}

void icetest_ddma_dump( void )
{
    int linewidth;

    icetest_get_idma_linewidth(&linewidth);
    
    dbg2( "%s\n", __FUNCTION__ );
    ddma_descriptor_list_dump( &icetest_desc_channel.desc_list );
    ddma_data_peek(&icetest_desc_channel.desc_list,
                   linewidth);
}

void icetest_interrupt_stats_dump( void )
{
    int i;

    dbg2( "icetest_isr=%d\n", icetest_interrupt_stats.num_interrupts );
    for( i=0 ; i<ICETEST_IDMA_NUM_CHANNELS ; i++ ) {
        dbg2( "icetest_isr dma ch=%d count=%d\n", i, icetest_interrupt_stats.channel_isr[i] );
    }
}

