/*
**************************************************************************
This Source Code Form is subject to the terms of the Mozilla Public
License, v. 2.0. If a copy of the MPL was not distributed with this file,
You can obtain one at http://mozilla.org/MPL/2.0/.

Copyright (c) 2012-2016, Marvell International Ltd.

Alternatively, this software may be distributed under the terms of the GNU
General Public License Version 2, and any use shall comply with the terms and
conditions of the GPL.  A copy of the GPL is available at
http://www.gnu.org/licenses/old-licenses/gpl-2.0.html

THE FILE IS DISTRIBUTED AS-IS, WITHOUT WARRANTY OF ANY KIND, AND THE
IMPLIED WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE
ARE EXPRESSLY DISCLAIMED.  The GPL license provides additional details about
this warranty disclaimer.
******************************************************************************
*/



/**
 * \file icefile.c
 *
 * \brief Utility functions for managing icefile
 *
 * davep 03-Feb-2012
 */

#include <stdint.h>
#include <stdbool.h>

#include "scos.h"

#include "lassert.h"
#include "memAPI.h"
#include "cpu_api.h"

#include "scantypes.h"
#include "scancore.h"
#include "scandbg.h"
#include "icefile.h"
#include "scanvars.h"
#include "scanmem.h"
#include "cal.h"
#include "cal_common.h"
#include "pichw.h"
#include "pic.h"

struct icefile_members {
    uint8_t *lut;
    uint32_t lut_num_bytes;

    bool is_open;
};

struct icefile_members current;

static void current_save_lut_ptr( uint8_t *lut, uint32_t datalen )
{
    /* beware of memory leaks! */
    XASSERT( current.lut==NULL, (uint32_t)current.lut );

    current.lut = lut;
    current.lut_num_bytes = datalen;
}

static void icefile_load_cal( const CAL_CONFIG *cal_config, const struct scanvars *sv, 
                              int32_t quad_SF1, int32_t quad_SF2, uint32_t quad_shift, 
                              uint8_t *lut, int lut_size_bytes ) 
{
    pic_pd_set_bypass_exp_comp(true);
    pic_pd_set_bypass_bad_pix_replace(cal_config->BYPASS_REPLACE_BAD_PIXEL);

    pic_pd_set_bypass_prnu(false);
    pic_pd_set_bypass_dsnu(false);
    pic_pd_set_bypass_all(false);

#ifdef HAVE_PIC_PC_QUAD
     if(cal_config->USE_QUAD && quad_SF1!= 0) {
         pic_pd_set_quadratic_config( quad_SF1, quad_SF2, quad_shift);
         pic_pd_set_bypass_quadratic(false);
     }
     else {
         pic_pd_set_bypass_quadratic(true);
     }
#else
     pic_pd_set_bypass_quadratic(true);
#endif

     pic_pd_idma_enable_lut_dma(false);
    pic_prnudsnu_reset_lut_dma();

    pic_prnudsnu_set_lut_xfer_count( lut_size_bytes );

#ifdef HAVE_PIC_PRNUDSNU_BIG_ENDIAN
    pd_lut_swap32( lut, lut_size_bytes );
#endif

    /* flush our data from cache into main memory */
    cpu_dcache_writeback_region( lut, lut_size_bytes );

    pic_prnudsnu_set_lut_addr((uint32_t)lut);

    prnudsnu_sanity_check( sv->cmode );

    /* and away we go */
    pic_pd_idma_enable_lut_dma(true);
    // since the pic_do_configure won't happen until pic_start, that's where we start the pd lut dma
}

scan_err_t icefile_cal_setup( const struct scanvars *sv )
{
//    scan_err_t scerr;
    const CAL_CONFIG *cal_config;
    int32_t quad_SF1, quad_SF2;
    uint32_t quad_shift;
    int lut_size_bytes;
    uint8_t *lut;

    ASSERT( !current.is_open );

    /* totally bogus numbers */
    quad_SF1 = 111;
    quad_SF2 = 55;
    quad_shift = 0;

    if( scan_get_config()&SCAN_CONFIG_NOCAL ) {
        dbg1(("%s nocal is set so not loading cal\n", __FUNCTION__ ));
        return SCANERR_NONE;
    }

    cal_config = cal_get_config();

    lut_size_bytes = sv->scan_area_pixels.width * cal_config->PRNUDSNU_LUT_ENTRY_SIZE;

    dbg2(( "%s width_pixels=%d lut_entry_bytes=%d total_bytes=%d\n", __FUNCTION__, 
                sv->scan_area_pixels.width, cal_config->PRNUDSNU_LUT_ENTRY_SIZE, lut_size_bytes ));

    lut = scanmem_alloc_aligned( lut_size_bytes );
    if( lut==NULL ) {
        return SCANERR_OUT_OF_MEMORY;
    }
    memset( lut, 0, lut_size_bytes );

    /* save the pointer so we can free it later */
    current_save_lut_ptr( lut, lut_size_bytes );
    
    /* load the lut with something crazy damaging to the pixels */
    int i;
    for( i=0 ; i<lut_size_bytes ; i++ ) {
        lut[i] = i;
    }

    dbg2(( "%s lut=%p numbytes=%d\n", __FUNCTION__, lut, lut_size_bytes ));

    pic_prnudsnu_set_bypass_all(true);

    /* comment out to turn off cal for this test */
    if( 1 ) {
        icefile_load_cal( cal_config, sv, quad_SF1, quad_SF2, quad_shift, lut, lut_size_bytes );
    }

    current.is_open = true;

    return SCANERR_NONE;
}

int icefile_load_pd_lut(int pixel_width){
	size_t lut_size_bytes;
	dma_addr_t lut;
	int tmp;

	//If bypassed, do nothing
	if(pic_pd_get_bypass_all(0)){
		dbg1(( "%s:PIC PD bypassed. Do not load LUT\n", __FUNCTION__));
		return 0;
	}

	/* ToDo: What PIC instance to use? Fix to use 0 for now*/
	lut_size_bytes = ((pixel_width * pic_pd_get_coeffwidth(0)) +7)/8;

	tmp = pic_pd_get_lut(0, &lut) ;
	if(tmp != lut_size_bytes){
		dbg1(( "%s:lut size mismatch! actual %d, expect %d\n", __FUNCTION__,
				tmp, lut_size_bytes));
		return 0;
	}


    pic_pd_idma_enable_lut_dma(false);
    pic_prnudsnu_reset_lut_dma();

    pic_prnudsnu_set_lut_xfer_count( lut_size_bytes );

#ifdef HAVE_PIC_PRNUDSNU_BIG_ENDIAN
    pd_lut_swap32( lut, lut_size_bytes );
#endif

    /* flush our data from cache into main memory */
    cpu_dcache_writeback_region( lut, lut_size_bytes );

    pic_prnudsnu_set_lut_addr((uint32_t)lut);

    /* and away we go */
    pic_pd_idma_enable_lut_dma(true);
    // since the pic_do_configure won't happen until pic_start, that's where we start the pd lut dma

    dbg1(( "%s:lut loaded, siz %d\n", __FUNCTION__,
           lut_size_bytes));
    return lut_size_bytes;
}
EXPORT_SYMBOL(icefile_load_pd_lut);

void icefile_cal_done( void )
{
    ASSERT( current.is_open );

    /* davep 14-May-2013 ; memory allocated via scanmem_alloc_aligned() must be
     * released by scanmem_free_aligned()
     */
    if( current.lut ) {
        scanmem_free_aligned( &current.lut, current.lut_size_bytes );
    }
    current.is_open = false;
}

