/*
**************************************************************************
This Source Code Form is subject to the terms of the Mozilla Public
License, v. 2.0. If a copy of the MPL was not distributed with this file,
You can obtain one at http://mozilla.org/MPL/2.0/.

Copyright (c) 2009-2015, Marvell International Ltd.

Alternatively, this software may be distributed under the terms of the GNU
General Public License Version 2, and any use shall comply with the terms and
conditions of the GPL.  A copy of the GPL is available at
http://www.gnu.org/licenses/old-licenses/gpl-2.0.html

THE FILE IS DISTRIBUTED AS-IS, WITHOUT WARRANTY OF ANY KIND, AND THE
IMPLIED WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE
ARE EXPRESSLY DISCLAIMED.  The GPL license provides additional details about
this warranty disclaimer.
******************************************************************************
*/



#include <stdint.h>

#include "lassert.h"

#include "fraction.h"
#include "safeint.h"

/**
 * \brief Algorithm for finding greatest common divisor
 *
 */
static int gcd(int x, int y)
{
    int a, b;
    if (x<y)
    {
        a = y;
        b = x;
    }
    else if (x>y)
    {
        a = x;
        b = y;
    }
    else 
    {
        return x;
    }

    do
    {
        int r = a % b;
        a = b;
        b = r;
    } while (b != 0);

    return ((a < 0) ? -a : a);
}
 
/**
 * \brief  reduce a fraction
 *
 * Make the scale factors smaller to reduce (but not eliminate!) the
 * possibility of overflow.
 *
 * The PIE XY Scaler firmware could still overflow if given a couple of huge
 * prime numbers.
 *
 */

void fraction_reduce_copy( const struct fraction *f_in, struct fraction *f_out )
{
    int divisor;

    divisor = gcd( f_in->numer, f_in->denom );

    XASSERT( f_in->numer%divisor==0, divisor );
    XASSERT( f_in->denom%divisor==0, divisor );

    f_out->numer /= divisor;
    f_out->denom /= divisor;
}

void fraction_reduce( struct fraction *f_inout )
{
    struct fraction f;

    f.numer = f_inout->numer;
    f.denom = f_inout->denom;

    fraction_reduce_copy( &f, f_inout );
}

void fraction_multiply( struct fraction *a, struct fraction *b, struct fraction *result )
{
    result->numer = safeint_scale_int( a->numer, b->numer, 1 );
    result->denom = safeint_scale_int( a->denom, b->denom, 1 );
    fraction_reduce( result );
}

