/*
**************************************************************************
This Source Code Form is subject to the terms of the Mozilla Public
License, v. 2.0. If a copy of the MPL was not distributed with this file,
You can obtain one at http://mozilla.org/MPL/2.0/.

Copyright (c) 2004-2015, Marvell International Ltd.

Alternatively, this software may be distributed under the terms of the GNU
General Public License Version 2, and any use shall comply with the terms and
conditions of the GPL.  A copy of the GPL is available at
http://www.gnu.org/licenses/old-licenses/gpl-2.0.html

THE FILE IS DISTRIBUTED AS-IS, WITHOUT WARRANTY OF ANY KIND, AND THE
IMPLIED WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE
ARE EXPRESSLY DISCLAIMED.  The GPL license provides additional details about
this warranty disclaimer.
******************************************************************************
*/


/****************************************************************************** 
 *
 * Description: Routines for figure objects used in calibration graph printing.
 *
 *****************************************************************************/

/* davep 23-Jul-2008 ; added -D__AGMESSAGE_H_ to makefile to prevent agMessage.h
 * from being included in the scan code.  But we need agMessage.h in this file.
 */
#ifdef __AGMESSAGE_H__
#undef __AGMESSAGE_H__
#endif

#include <string.h>

#include "tx_api.h"

#include "ATypes.h"
#include "memAPI.h"
//#include "renderer_4in1.h"
#include "renderer.h"
#include "figure.h"
#include "lassert.h"
#include "dprintf.h"

#include "scantypes.h"
#include "scancore.h"
#include "scandbg.h"

#define CALDBG(x) do{ DPRINTF((DBG_LOUD | DBG_OUTPUT), ("CAL.%s@%d:", __FILE__, __LINE__));\
               DPRINTF((DBG_LOUD | DBG_OUTPUT), x);\
               DPRINTF((DBG_LOUD | DBG_OUTPUT),("\n"))}while(0)

/**\brief Utility routine used by a figure to obtain a memory buffer.
 * 
 *  A figure uses this routine to obtain a registerd buffer that can be released
 *  in a figure's destroy routine. Each memry buffer allocated by this routine 
 *  can not be freed explicitly by a figure's user.
 * 
 * \param[in] pFig  The figure that requests the buffer
 * \param[in] size  The size of the buffer
 * 
 */
char *getFigureBuffer( Figure *pFig, int size )
{
    char * pMem = MEM_MALLOC( size );
    XASSERT( pMem!=NULL, size );
//    memset( pMem, 0, size );

    FigBuffer *pBuf = MEM_MALLOC( sizeof( FigBuffer ) );
    XASSERT( pBuf!=NULL, sizeof(FigBuffer) );
//    memset( pBuf, 0, sizeof(FigBuffer) );

    pBuf->buf = pMem;
    pBuf->next = pFig->buffer;
    pFig->buffer = pBuf; 
    
    return pMem;
}

/** \brief Routine to free all the memory resources allocated during a figure's
 *   life-time
 * 
 * \param[in] pFig   The figure to be destroyed
 */
void freeFigResources( Figure *pFig )
{

    FigBuffer * pBuf;
    FigBuffer *pBuf2;

    pBuf = pFig->buffer;

    while ( pBuf )
    {
        pBuf2 = pBuf;
        pBuf = pBuf2->next;
        PTR_FREE( pBuf2->buf );
        PTR_FREE( pBuf2 );
    }
   
}

/**\brief A utility routine to add a new DISP_LIST item to a figure
 * 
 *  Each individually allocated DISP_LIST object must be added to a figure via this
 *  routine in order to be released properly later when the figure is destroyed. 
 *  Those DISP_LIST objects that are allocated collectively by dividing a large 
 *  single memory, should only be added once by using the element with 
 *  its address matching the start address of the memory chunk.
 * 
 * \param[in] pFig    The figure to which the new item to be added to.
 * \param[in] newItem The item to be added 
 */

static void addItemToFigure(Figure *pFig, DISP_LIST *newItem)
{
  newItem->NextObj = pFig->contentHead;
  pFig->contentHead = newItem;
} 

/*\brief Get the dimensions of the graph box in a figure.
 * 
 * In each figure, there are two areas, one is for graph, the other is
 * for text. This routine returns the location/dimensions of the rectangular 
 * graph area in a figure.
 *  
 * \param[in] x1  The x of left bottom
 * \param[in] y1  The y of left bottom
 * \param[in] x2  The x of right top
 * \param[in] y2  The y of right top

 */
static void getGraphBoxDim(int *x1, int *y1, int *x2, int *y2)
{
    *x1 = 50;
    *y1 = 100;
    *x2 = 5090;
    *y2 = 4700;
}

/*\bref Get the dimensions of the text box in a figure
 * 
 * In each figure, there are two areas, one is for graph, the other is
 * for text. This routine returns the location/dimensions of the rectangular 
 * text area in a figure.
 * 
 * \param[in] x1  The x of left bottom
 * \param[in] y1  The y of left bottom
 * \param[in] x2  The x of right top
 * \param[in] y2  The y of right top
 */

static void getTextBoxDim(int *x1, int *y1, int *x2, int *y2)
{
    *x1 = 5100;
    *y1 = 100;
    *x2 = 6150;
    *y2 = 4700;
}

/**\brief Routine to draw a text on a given fiure.
 * 
 * \param[in] pFig  The figure to draw the text on
 * \param[in] font  The font to be used
 * \param[in] text  The text to be drawn
 * \param[in] x     The x coordinate of the text location
 * \param[in] y     The y coordinate of the text location
 */
static void drawText(Figure *pFig, int  font, const char *text, int x, int y)
{

    //Build a text object for diaply
    DISP_LIST * strDISP;
    TEXT *strTEXT;
    char *pTxt;
   
    //CPDBG( ( "Build text: %s", text ) );

    strDISP = ( DISP_LIST * ) getFigureBuffer( pFig, sizeof( DISP_LIST ));
    strTEXT = ( TEXT * ) getFigureBuffer(pFig, sizeof( TEXT ));
    pTxt = (char *)getFigureBuffer(pFig, strlen(text) + 4);
    strcpy(pTxt, text);
    
    //CALDBG(("pTxt=%s", pTxt));

    strTEXT->Data = pTxt;
    strTEXT->Color = 1;  
    strTEXT->FONT = font;
    strTEXT->justify = JUSTIFY_LEFT;

    strDISP->NextObj = 0;
    strDISP->OBJ_X_LEN = 0;
    strDISP->OBJ_Y_LEN = 0;
    strDISP->Object = strTEXT;
    strDISP->OBJ_X = x; 
    strDISP->OBJ_Y = y;
    strDISP->Operation = e_or;
    strDISP->Type = e_Text; 

    addItemToFigure( pFig, strDISP );
    
}
/**\brief Routine to draw the title of a figure.
 * 
 * \param[in] pFig The figure to draw the title on
 * \param[in] title The text for the title
 * 
 */
static void drawTitle(Figure *pFig, const char *title)
{

    drawText(pFig, 0, title, 2000, 4750);
}

/**\brief Routine to draw a line between two given points in a figure.
 * 
 * \param[in] pFig  The figure to draw the line
 * \param[in] x1    The x coordinate for the starting point
 * \param[in] y1    The y coordinate for the starting point
 * \param[in] x2    The x coordinate for the ending point
 * \param[in] y2    The y coordinate for the ending point
 * \param[in] lineWidth The width of the line to be drawn
 * \param[in] lineStyle The style of the line to be drawn
 * 
 * 
 */
static void drawLine(Figure *pFig,  int x1, int y1, int x2, int y2, int lineWidth, int lineStyle)
{
     DISP_LIST * lineDISP;
     LINE  *pLineDef; 
     LINE_OBJ *pLineObj;
     UINT8 *pLineTemp;
     int i;

//     dbg2(( "%s x1=%d y1=%d x2=%d y2=%d\n", __FUNCTION__, x1, y1, x2, y2 ));
     
     //setup the thickness for the curve
     pLineDef = (LINE *)getFigureBuffer(pFig, sizeof(LINE));
     pLineDef->LineWidth = lineWidth;
    
     pLineObj = (LINE_OBJ *)getFigureBuffer(pFig, sizeof(LINE_OBJ));
     pLineTemp = (UINT8 *)getFigureBuffer(pFig, 32);
     for(i = 0; i < 32;i++)
     {    if(i <= lineStyle) 
               pLineTemp[i] = 0xff;
        else
             pLineTemp[i] = 0;
     }
     pLineObj->Template = pLineTemp;
     pLineObj->TemplateLength = 32;
     pLineDef->LineObj = pLineObj;
     
     lineDISP = ( DISP_LIST * ) getFigureBuffer( pFig, sizeof( DISP_LIST ));
     lineDISP->OBJ_X = x1;
     lineDISP->OBJ_Y = y1;
     lineDISP->OBJ_X_LEN = x2;
     lineDISP->OBJ_Y_LEN = y2;

     lineDISP->Operation = e_or;
     lineDISP->Type = e_DrawLine;
     lineDISP->Object = pLineDef;

     addItemToFigure( pFig, lineDISP );
     

}
/** \brief Routine to draw a curve that passes through all the given points.
 * 
 * The routine draws a curve from the leftmost to the rightmost of the graph box of
 * the given figure. 
 * 
 * Note: it will fail if no graph box of the figure is defined
 * 
 * 
 * \param[in] pFig   The figure to draw the curve on
 * \param[in] length The number of the points 
 * \param[in] data   The buffer of the data values
 * \param[in] x0     The starting x position
 * \param[in] y0     The starting y position 
 * \param[in] lineWidth  The witdth of the curve
 * \param[in] lineStyle  The style of the curve
 * \param[in] x_inc   The x step distance
 * \param[in] y_inc   The y step distance
 * 
 */
#if 1
#define SEGNUM 600    //Large curve is draw as multiple sections with this length
static void drawCurve(Figure *pFig, int length, int x0, int y0, UINT16 *data, 
                      int lineWidth, int lineStyle, float x_inc, float y_inc)
{
    /* davep 25-Oct-2007 ; XXX have to remove this function until I can bring
     * EricH's new renderer into the trunk code. He added a new CURVE structure
     * to reduce the amount of memory.
     */
#if 0
    ASSERT(0);
#else
    DISP_LIST *DSPPixels;
    CURVE *pCurve;
    int i,j;
    UINT16 y_max;

//    dbg2(( "%s x0=%d y0=%d length=%d x_inc=%d y_inc=%d\n", __FUNCTION__, x0, y0,
//                length, x_inc, y_inc ));

//    scanlog_hex_dump( data, 1024 );

    for(j=0; j< (length/SEGNUM +1) ;j++)
    {

        //setup the thickness for the curve
        pCurve = (CURVE *)getFigureBuffer(pFig, sizeof(CURVE));
        memset( pCurve, 0, sizeof(CURVE) );
        pCurve->LineWidth = lineWidth;
        pCurve->x_inc = x_inc;
        pCurve->y_inc = y_inc;
        pCurve->data = &(data[j*SEGNUM]);
       
        DSPPixels = (DISP_LIST *)getFigureBuffer(pFig, sizeof(DISP_LIST));
        ASSERT(DSPPixels);
        memset( DSPPixels, 0, sizeof(DISP_LIST) );

        DSPPixels->OBJ_X = x0 + x_inc*j*SEGNUM;  //start x    
        DSPPixels->OBJ_Y = y0; //start y

        y_max = data[j*SEGNUM];
        for(i=j*SEGNUM; i<((j+1)*SEGNUM); i++)
        {
            if(i>=length)
                break;

            if(data[i] > y_max)
                y_max = data[i];

        }
        if( i >= length )
            pCurve->length = length - j*SEGNUM;
        else
            pCurve->length = SEGNUM;

        DSPPixels->OBJ_X_LEN =  pCurve->length * x_inc; //X length
        DSPPixels->OBJ_Y_LEN = y_max * y_inc;  //Y length  <<<<FIX this!

        DSPPixels->Operation = e_or;
        DSPPixels->Type = e_DrawCurve;
        DSPPixels->Object = pCurve;
           
        addItemToFigure(pFig, DSPPixels);

        if(i>=length)
            break; //done
    }
#endif
}

#endif

#if 0
static void drawCurve(Figure *pFig, int length, int x0, int y0, UINT16 *data, 
                      int lineWidth, int lineStyle, float x_inc, float y_inc)
{
    int i;
    LINE  *pLineDef;
    DISP_LIST *DSPPixels;
    LINE_OBJ *pLineObj;
    UINT8 *pLineTemp;
    int gx1, gx2, gy1, gy2;
    
    pFig->getGraphBoxDim(&gx1, &gy1,&gx2, &gy2);
    
    //setup the thickness for the curve
    pLineDef = (LINE *)getFigureBuffer(pFig, sizeof(LINE));
    pLineDef->LineWidth = lineWidth;
    
    pLineObj = (LINE_OBJ *)getFigureBuffer(pFig, sizeof(LINE_OBJ));
    pLineTemp = (UINT8 *)getFigureBuffer(pFig, 64);
    for(i = 0; i < 64;i++)
    {    if(i <= lineStyle*4) 
               pLineTemp[i] = 0xff;
        else
             pLineTemp[i] = 0;
    }
    pLineObj->Template = pLineTemp;
    pLineObj->TemplateLength = 64;
    pLineDef->LineObj = pLineObj;
    
    DSPPixels = (DISP_LIST *)getFigureBuffer(pFig, sizeof(DISP_LIST) * length);
    
    ASSERT(DSPPixels);
    
    DSPPixels[0].OBJ_Y = ((float)data[0]) * y_inc + y0;
    DSPPixels[0].OBJ_Y_LEN = DSPPixels[0].OBJ_Y;
    DSPPixels[0].OBJ_X = x0;
    DSPPixels[0].OBJ_X_LEN = x0;
    DSPPixels[0].Operation = e_or;
    DSPPixels[0].Type = e_DrawLine;
    DSPPixels[0].Object = pLineDef;
        
    addItemToFigure(pFig, &DSPPixels[0]);

    
    // Iterate through the remaining pixels, drawing lines between them.
    for (i=1; i<length;i++)
    {
        // Draw a line from the previous pixel....
        DSPPixels[i].OBJ_X = DSPPixels[i-1].OBJ_X_LEN;
        DSPPixels[i].OBJ_Y = DSPPixels[i-1].OBJ_Y_LEN;

        // to this pixel.
        DSPPixels[i].OBJ_X_LEN = i* x_inc + x0;
        DSPPixels[i].OBJ_Y_LEN = ((float)data[i]) * y_inc + y0;
        if (DSPPixels[i].OBJ_Y_LEN >gy2)
            DSPPixels[i].OBJ_Y_LEN = gy2;

        DSPPixels[i].Operation = e_or;
        DSPPixels[i].Type = e_DrawLine;
        DSPPixels[i].Object = pLineDef; 
        
        //CALDBG(("Draw line data[%d]=%d\n", i, data[i])); 
        addItemToFigure(pFig, &DSPPixels[i]);
    } 
}
#endif

/** \brief Rroutine to draw a collection of pixel points on a figure.
 * 
 * \param[in] pFig  The figure to draw the points
 * \param[in] length The size of the points array 
 * \param[in] x      The array of the x coordinates for the points
 * \param[in] y      The array of the y coordinates for the points
 * \param[in] mode   The mode, which is actually the radius of each point
 *                   to be drawn on the figure. 
 * 
 */
static void drawPixels(Figure *pFig, int length, UINT16 *x, UINT16 *y, int mode)
{
    int i;
    int r = mode*2;

//    dbg2(( "%s length=%d mode=%d x=%d y=%d\n", __FUNCTION__, 
//                length, mode, *x, *y ));

    if(r<2)r=2;
    
    for(i=0;i<length; i++)
    {
        pFig->drawLine(pFig, x[i] - r,y[i], x[i] + r, y[i], 1, 8);
        pFig->drawLine(pFig, x[i],y[i]- r, x[i], y[i] + r, 1, 8);
    }
}

/**\brief Routine to create a titled figure with graph/text boxes drawn
 * 
 * \param[in] title The text of the title
 */
Figure *createTitledFigure(const char *title)
{
    Figure *pFig = createBlankFigure();
    
    if( pFig==NULL ) {
        return NULL;
    }

    drawTitle(pFig, title);
    
    //draw graph box
    pFig->drawLine(pFig, 50, 100, 50, 4700, 1,8); //left
    pFig->drawLine(pFig, 50, 100, 5090,100, 1,8); //bottom
    pFig->drawLine(pFig, 50, 4700, 5090, 4700, 1,8); //top 
    pFig->drawLine(pFig, 5090, 100, 5090, 4700, 1,8); //right 
    
    //draw text box
    pFig->drawLine(pFig, 5100, 100, 5100, 4700, 1, 8); //left 
    pFig->drawLine(pFig, 5100, 100, 6150, 100, 1, 8); //bottom
    pFig->drawLine(pFig, 5100, 4700, 6150, 4700, 1, 8); //top 
    pFig->drawLine(pFig, 6150, 100, 6150, 4700, 1, 8); //right
    
    return pFig;
    
}
/**\brief Routine to create a blank figure.
 * 
 * This routine creates a blank figure with no title and no text/graph
 * boxes drawn.
 */
Figure *createBlankFigure()
{
    Figure *pFig = MEM_MALLOC(sizeof(Figure) );

    if( pFig==NULL ) {
        return NULL;
    }
    memset( pFig, 0, sizeof(Figure) );

    pFig->contentHead = 0;
    pFig->drawText = drawText;
    pFig->drawLine = drawLine;
    pFig->drawTitle = drawTitle;
    pFig->drawPixels = drawPixels;
    pFig->drawCurve = drawCurve;
    pFig->getTextBoxDim = getTextBoxDim;
    pFig->getGraphBoxDim = getGraphBoxDim;
    pFig->buffer = 0;
    
    return pFig;
} 

/** \brief Utility routine to destroy a figure object.
 * 
 * \param[in] pFig The figure to be destroyed
 * 
 */
void destroyFigure(Figure *pFig)
{
    freeFigResources(pFig);
    PTR_FREE(pFig);
}

