/*
**************************************************************************
This Source Code Form is subject to the terms of the Mozilla Public
License, v. 2.0. If a copy of the MPL was not distributed with this file,
You can obtain one at http://mozilla.org/MPL/2.0/.

Copyright (c) 2010-2016, Marvell International Ltd.

Alternatively, this software may be distributed under the terms of the GNU
General Public License Version 2, and any use shall comply with the terms and
conditions of the GPL.  A copy of the GPL is available at
http://www.gnu.org/licenses/old-licenses/gpl-2.0.html

THE FILE IS DISTRIBUTED AS-IS, WITHOUT WARRANTY OF ANY KIND, AND THE
IMPLIED WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE
ARE EXPRESSLY DISCLAIMED.  The GPL license provides additional details about
this warranty disclaimer.
******************************************************************************
*/


/*
 *
 *  CISX management code.
 *  19-Nov-2010
 *
 *  Adding dual scanner support (two CISX blocks)
 *  30-Jan-2013
 */

#include <stdint.h>
#include <stdbool.h>
#include <string.h>

#include "scos.h"

#include "lassert.h"
#include "regAddrs.h"
#include "interrupt_api.h"
#include "list.h"
#include "memAPI.h"
#include "cpu_api.h"

#include "scancore.h"
#include "scantypes.h"
#include "scandbg.h"
#include "safetylock.h"
#include "icedma.h"
#include "ddma.h"
#include "cisx.h"
#include "scantools.h"
#include "scanblk_if.h"
#include "scanif.h"
#include "scancmdq.h"
#include "scantask.h"

#include "scands.h"
#include "scanmem.h"
#include "scanimg.h"

#include "cisx_if.h"   // CISX driver

/* Define to turn on more verbose debugging */
#define CISX_DEBUG  

#ifdef CISX_DEBUG
  #define cisx_dbg2 dbg2 
#else
  #define cisx_dbg2(...)
#endif

// Currently scanconfig defines HAVE_NSENSOR_SUPPORT to configure the max number
// of sensors in the system (one or two).  Going to map this to the number of
// CISX instances that we will support.  Note that other parts of the scan system
// may also limit the number of sensors used at any particular time, but NSENSOR
// is a hard limit at compile time.
#ifdef HAVE_NSENSOR_SUPPORT
    #define NUM_CISX_INSTANCES 2
#else
    #define NUM_CISX_INSTANCES 1
#endif

#define MAX_CISX_DMA_DESCRIPTORS 2
//#define MAX_CISX_DMA_DESCRIPTORS 8
//#define MAX_CISX_DMA_DESCRIPTORS 32

struct cisx_mm {
    /** icebufs ready and waiting to be loaded into DMA controller */
    ATLISTENTRY readies;

    /** icebufs currently loaded into DMA controller */
    ATLISTENTRY running;

    int num_readies;
    int num_running;

    /* 13-Mar-2013 ; check for memory leaks, double pointers, etc. */
    int num_descriptors_allocated;

    /* 03-Apr-2013 ; adding run-time enabling of dual scan */
    uint32_t sensor_bitmask;

    /* 22-Apr-2013 ; attempt to allocate from fast memory (e.g. SRAM). If
     * the allocation fails, we'll alloc from DRAM instead
     */
    void *cisx_buf;
    uint32_t cisx_buf_num_bytes;
    bool cisx_buf_is_fast_memory;
};

#define MAX_CISX_CHANNEL_NAME 31

struct cisx_channel {
    /* the channel id this data structure assigned to */
    uint8_t channel;

    char name[MAX_CISX_CHANNEL_NAME];

    /* 27-Apr-05 ; needed a way to unambiguously tell if a channel is in-use */
    bool is_open;

    uint32_t bytes_per_row;

    struct ddma_descriptor_list desc_list;
};

static struct cisx_channel cisx_odma_channel[CISX_ODMA_NUM_CHANNELS];
static struct cisx_channel cisx_idma_channel[CISX_IDMA_NUM_CHANNELS];

static struct cisx_mm cisx_mm;

/* 29-Jan-2013 ; adding dual scan */
static int num_odma_channels;
static cisx_odma_chan_t odma_channels[CISX_ODMA_NUM_CHANNELS];
static int num_idma_channels;
static cisx_idma_chan_t idma_channels[CISX_IDMA_NUM_CHANNELS];

struct cisx_block {
    char name[8];
    cisx_odma_chan_t odma_channels[3];
    cisx_idma_chan_t idma_channel;
};


//*****************************************************************************
// Porting helper functions
//*****************************************************************************
static void get_instance_from_idma_channel(cisx_idma_chan_t channel, 
                                    uint8_t *cisx_instance,
                                    uint8_t *idma_instance)
{
    switch (channel)
    {
        case CISX0_IDMA0:
            *cisx_instance = 0;
            *idma_instance = 0;
            break;
        case CISX1_IDMA0:
            *cisx_instance = 1;
            *idma_instance = 0;
            break;
        default:
            XASSERT( 0, channel );
            break;
    }

    //cisx_dbg2( "%s: chan=%d cisx=%d idma=%d\n", 
    //           __FUNCTION__,  channel, *cisx_instance, *idma_instance);
}

static void get_instance_from_odma_channel(cisx_odma_chan_t channel, 
                                    uint8_t *cisx_instance,
                                    uint8_t *odma_instance)
{
    switch (channel)
    {
        case CISX0_ODMA0:
            *cisx_instance = 0;
            *odma_instance = 0;
            break;
        case CISX0_ODMA1:
            *cisx_instance = 0;
            *odma_instance = 1;
            break;
        case CISX0_ODMA2:
            *cisx_instance = 0;
            *odma_instance = 2;
            break;
       case CISX1_ODMA0:
            *cisx_instance = 1;
            *odma_instance = 0;
            break;
        case CISX1_ODMA1:
            *cisx_instance = 1;
            *odma_instance = 1;
            break;
        case CISX1_ODMA2:
            *cisx_instance = 1;
            *odma_instance = 2;
            break;
        default:
            XASSERT( 0, channel );
            break;
    }

    //cisx_dbg2( "%s: chan=%d cisx=%d odma=%d\n", 
    //           __FUNCTION__,  channel, *cisx_instance, *odma_instance);
}


//*****************************************************************************
// Interrupt callbacks
//*****************************************************************************

static void cisx_idma_interrupt( struct cisx_idma_irqs *irq_struct )
{
    /* 
     * BIT FAT NOTE: THIS IS AN INTERRUPT HANDLER.
     *
     */

    /* The current code only uses interrupts in error conditions.  The DMA
     * callbacks will happen first, then the INT callback.  Simply output
     * the pending IRQ here for debug purposes, then let the INT callback do
     * the heavy lifting.
     */
    dbg1( "ERROR! %s: cisx=%d dma=%d ipend=%#x\n", __FUNCTION__, 
              irq_struct->cisx_instance, irq_struct->dma_instance, irq_struct->irq_array );
}

static void cisx_odma_interrupt( struct cisx_odma_irqs *irq_struct )
{
    /* 
     * BIT FAT NOTE: THIS IS AN INTERRUPT HANDLER.
     *
     */

    /* The current code only uses interrupts in error conditions.  The DMA
     * callbacks will happen first, then the INT callback.  Simply output
     * the pending IRQ here for debug purposes, then let the INT callback do
     * the heavy lifting.
     */
    dbg1( "ERROR! %s: cisx=%d dma=%d ipend=%#x\n", __FUNCTION__, 
              irq_struct->cisx_instance, irq_struct->dma_instance, irq_struct->irq_array );
}

static void cisx_int_interrupt( struct cisx_int_irqs *irq_struct )
{
    uint8_t  cisx_instance;

    /* 
     * BIT FAT NOTE: THIS IS AN INTERRUPT HANDLER.
     *
     */

    dbg1( "ERROR! %s: cisx=%d ipend=%#x\n", __FUNCTION__, 
              irq_struct->cisx_instance, irq_struct->irq_array );

    cisx_dump();

    /* As of this writing (05-Apr-2011) the only interrupts I'm using are those
     * indicating an error. CISX runs autonomously once started so no
     * interrupts are necessary for normal behavior.
     *
     * If you are here, something has gone wrong. Horribly, horribly wrong.
     */

    /* 11-Feb-2013 ; ack everything so we don't get stuck in here */
    for (cisx_instance = 0; cisx_instance < NUM_CISX_INSTANCES; cisx_instance++)
    { 
        cisx_int_disable_irqs(cisx_instance, NULL);
        cisx_int_clear_irqs(cisx_instance, NULL);
    }

    /* 08-Jul-2011 ; stomp on the scan block and cisx immediately so we
     * can get an accurate snapshot of where we went pear shaped
     */

    // Note: indicate we are calling from interrupt context
    scif_clock(0, true);
    scif_control(0, 0, true);

    cisx_odma_channel_disable(CISX0_ODMA0);
    cisx_odma_channel_disable(CISX0_ODMA1);
    cisx_odma_channel_disable(CISX0_ODMA2);
    cisx_idma_channel_disable(CISX0_IDMA0);
    scan_cmdq_emergency_halt();

//    scif_dump();
//    icetest_dump();
//    icetest_ddma_dump();
//    pic_dump();
//    cisx_ddma_data_peek();

    XASSERT( 0, irq_struct->irq_array );
}

scan_err_t cisx_duplex_setup( void )
{
    scan_err_t scerr = SCANERR_NOT_IMPLEMENTED;
#ifdef HAVE_NSENSOR_SUPPORT
    /* 29-Jan-2013 ; tinkering with dual scan
     *  CISX0 cisx_tag = tag value from Scan for CISX0 (0)
     *  CISX1 cisx_tag = tag value from Scan for CISX1 (1)
     *     (CISX_INT_CISXCFG_CISX_TAG_REPLACE_VAL)
     *
     *  numsensors==1 in CISX0 and CISX1
     *     (CISX_INT_CISXCFG_NUMSENSORS_REPLACE_VAL)
     *
     *  CISX0 force_odd=0
     *  CISX1 force_odd=1
     *     (CISX_INT_CISXCFG_FORCE_ODD_REPLACE_VAL)
     *
     */

    // FIXME: We no longer have force_odd, numsensors, or tag settings. Not sure
    //        what duplex specific setup will be required here, if any.

    scerr = SCANERR_NONE;
#endif
    return scerr;
}

void cisx_interrupt_disable( void )
{
    uint8_t cisx_instance;

    cisx_dbg2( "%s\n", __FUNCTION__ );

    for (cisx_instance = 0; cisx_instance < NUM_CISX_INSTANCES; cisx_instance++)
    { 
        // Calling with a NULL cisx_int_irqs struct will change all the interrupts,
        // in this case we are turning them all off.
        cisx_int_disable_irqs(cisx_instance, NULL);
    }
}

void cisx_interrupt_enable( void )
{
    uint8_t cisx_instance;

    cisx_dbg2( "%s\n", __FUNCTION__ );

    /* 08-Jul-2011 ; turn 'em all on; gotta catch 'em all!
     * The non-error DMA interrupts (e.g., fin_int_en) are disabled at each DMA
     * channel. Firmware doesn't need to handle CISX DMA interrupts for CISX to
     * function.
     */
    for (cisx_instance = 0; cisx_instance < NUM_CISX_INSTANCES; cisx_instance++)
    { 
        // Calling with a NULL cisx_int_irqs struct will change all the interrupts,
        // in this case we are turning them all on.
        cisx_int_enable_irqs(cisx_instance, NULL);
    }
}

static void cisx_desc_list_sanity( struct ddma_descriptor_list *desc_list )
{
    int sanity_count; 
    struct ddma_descriptor *head;
    struct ddma_descriptor *curr;

//    dbg2( "%s head=%p num=%d\n", __FUNCTION__, head, num_descriptors ));

    head = desc_list->list;

    ASSERT( head );
    ASSERT( desc_list->num_descriptors );

    curr = head;
    sanity_count = 0;
    do {
        /* CISX never adds buffers to the list so every descriptor must have a
         * pointer attached
         */
        XASSERT( curr->src_addr, (uint32_t )curr );
        XASSERT( curr->transfer_len_bytes, (uint32_t )curr );

        /* all buffers must be identical in size */
        XASSERT( curr->transfer_len_bytes==head->transfer_len_bytes, curr->transfer_len_bytes );

        curr = (struct ddma_descriptor *)curr->fw_next_descriptor_addr;

        /* 07-Jul-2012 ; make sure we have our convenience pointers
         * correctly set
         */
        XASSERT( curr->fw_src_addr, (uint32_t)curr );

        /* avoid infinite loops on corrupted lists */
        sanity_count += 1;
        XASSERT( sanity_count<=desc_list->num_descriptors, sanity_count );

    } while( curr != head );

    XASSERT( sanity_count==desc_list->num_descriptors, sanity_count );
}


/*
 *  CISX ODMA Functions
 *
 */

static void odma_channel_dump( cisx_odma_chan_t channel )
{
    uint8_t cisx_instance, dma_instance;

    get_instance_from_odma_channel(channel, &cisx_instance, &dma_instance);

    cisx_odma_dump_regs(cisx_instance, dma_instance);
}

void cisx_odma_channel_disable( cisx_odma_chan_t channel )
{
    struct cisx_odma_cfg_reg reg_struct;
    uint8_t cisx_instance, dma_instance;

    get_instance_from_odma_channel(channel, &cisx_instance, &dma_instance);

    memset(&reg_struct, 0, sizeof(struct cisx_odma_cfg_reg));
    reg_struct.enable       = 0;
    reg_struct.enable_valid = true;
    cisx_odma_set_cfg(cisx_instance, dma_instance, &reg_struct);

    // Disable all ODMA interrupts (for this channel)
    cisx_odma_disable_irqs(cisx_instance, dma_instance, NULL);
}

void cisx_odma_channel_enable( cisx_odma_chan_t channel )
{
    struct cisx_odma_cfg_reg reg_struct;
    struct cisx_odma_irqs    irq_struct;
    uint8_t cisx_instance, dma_instance;

    get_instance_from_odma_channel(channel, &cisx_instance, &dma_instance);

    memset(&reg_struct, 0, sizeof(struct cisx_odma_cfg_reg));

    // Enable the channel
    reg_struct.enable          = 1;
    reg_struct.enable_valid    = true;

    // Set input width to 16-bits
    reg_struct.in_width        = CISX_ODMA_CFG_IN_WDTH_16;
    reg_struct.in_width_valid  = true;

    // Set dma burst length to 4 words
    reg_struct.burst_len       = CISX_ODMA_CFG_BURST_LEN_4;
    reg_struct.burst_len_valid = true;

    reg_struct.serpentine       = 0;     // Normal order
    reg_struct.serpentine_valid = true;
    reg_struct.line_rev         = 0;     // Normal order
    reg_struct.line_rev_valid   = true;
    reg_struct.tran_rev         = 0;     // Normal order
    reg_struct.tran_rev_valid   = true;
    reg_struct.upper_half       = 1;
    reg_struct.upper_half_valid = true;
    reg_struct.msb_in           = 1;
    reg_struct.msb_in_valid     = true;

    cisx_odma_set_cfg(cisx_instance, dma_instance, &reg_struct);

    // Enable all the error interrupts
    memset(&irq_struct, 0, sizeof(struct cisx_odma_irqs));
    irq_struct.rresp         = true;
    irq_struct.bresp         = true;
    irq_struct.dir           = true;
    irq_struct.cl_ali        = true;
    irq_struct.eol_ali       = true;
    irq_struct.eoi_ali       = true;
    irq_struct.eoi_err       = true;
    irq_struct.who           = true;

    cisx_odma_enable_irqs(cisx_instance, dma_instance, &irq_struct);
}

void cisx_odma_channel_reset( cisx_odma_chan_t channel )
{
    struct cisx_odma_reset_reg     rst_struct;
    struct cisx_odma_line_size_reg ls_struct;
    uint8_t cisx_instance, dma_instance;

    get_instance_from_odma_channel(channel, &cisx_instance, &dma_instance);

    // Reset ODMA
    memset(&rst_struct, 0, sizeof(struct cisx_odma_reset_reg));
    rst_struct.soft_reset       = 1;
    rst_struct.soft_reset_valid = true;
    cisx_odma_set_reset(cisx_instance, dma_instance, &rst_struct);

    cpu_spin_delay( 10 );  /* wait a little while to allow hw to reset */

    //rst_struct.soft_reset       = 0;
    //rst_struct.soft_reset_valid = true;
    //cisx_odma_set_reset(cisx_instance, dma_instance, &rst_struct);

    // Disable and ack all interrupts
    cisx_odma_disable_irqs(cisx_instance, dma_instance, NULL);
    cisx_odma_clear_irqs(cisx_instance, dma_instance, NULL);

    // Write back select power-on defaults
    // TODO: anything else needed?
    memset(&ls_struct, 0, sizeof(struct cisx_odma_line_size_reg));
    ls_struct.line_size       = 0;
    ls_struct.line_size_valid = true;
    cisx_odma_set_line_size(cisx_instance, dma_instance, &ls_struct);
}

static void cisx_odma_channel_launch( struct cisx_channel *cch )
{
    int i;
    struct ddma_descriptor_list *desc_list;
    struct cisx_odma_line_size_reg  ls_struct;
    struct cisx_odma_desc_write_reg dw_struct;
    uint8_t cisx_instance, dma_instance;

    XASSERT( cch->is_open, cch->channel );

    cisx_dbg2( "%s: %s %d\n", __FUNCTION__, cch->name, cch->channel );

    get_instance_from_odma_channel(cch->channel, &cisx_instance, &dma_instance);

    /* use a shortcut to keep expression size down */
    desc_list = &cch->desc_list;
     
    /* once we start CISX, this list never changes. So flush *everything* */
    for( i=0 ; i<desc_list->num_descriptors ; i++ ) {
        ddma_desc_flush( &desc_list->list[i] );
    }

    /* fire in the hole! */
    memset(&ls_struct, 0, sizeof(struct cisx_odma_line_size_reg));
    ls_struct.line_size       = cch->bytes_per_row;
    ls_struct.line_size_valid = true;
    cisx_odma_set_line_size(cisx_instance, dma_instance, &ls_struct);

    memset(&dw_struct, 0, sizeof(struct cisx_odma_desc_write_reg));
    dw_struct.desc       = (uint32_t)desc_list->list[0].dma_ptr_self;
    dw_struct.desc_valid = true;
    cisx_odma_set_desc_write(cisx_instance, dma_instance, &dw_struct);
}

/*
 * CISX IDMA Functions
 *
 */

static void idma_channel_dump( cisx_idma_chan_t channel )
{
    uint8_t cisx_instance, dma_instance;

    get_instance_from_idma_channel(channel, &cisx_instance, &dma_instance);

    cisx_idma_dump_regs(cisx_instance, dma_instance);
}

void cisx_idma_channel_disable( cisx_idma_chan_t channel )
{
    struct cisx_idma_cfg_reg reg_struct;
    uint8_t cisx_instance, dma_instance;

    get_instance_from_idma_channel(channel, &cisx_instance, &dma_instance);

    memset(&reg_struct, 0, sizeof(struct cisx_idma_cfg_reg));
    reg_struct.enable       = 0;
    reg_struct.enable_valid = true;
    cisx_idma_set_cfg(cisx_instance, dma_instance, &reg_struct);

    // Disable all IDMA interrupts (for this channel)
    cisx_idma_disable_irqs(cisx_instance, dma_instance, NULL);
}

void cisx_idma_channel_enable( cisx_idma_chan_t channel )
{
    struct cisx_idma_cfg_reg reg_struct;
    struct cisx_idma_irqs    irq_struct;
    uint8_t cisx_instance, dma_instance;

    get_instance_from_idma_channel(channel, &cisx_instance, &dma_instance);

    memset(&reg_struct, 0, sizeof(struct cisx_idma_cfg_reg));

    // Enable the channel
    reg_struct.enable          = 1;
    reg_struct.enable_valid    = true;

    // Set output width to 16-bits
    reg_struct.out_width        = CISX_IDMA_CFG_OUT_WDTH_16;
    reg_struct.out_width_valid  = true;

    // Set dma burst length to 4 words
    reg_struct.burst_len       = CISX_IDMA_CFG_BURST_LEN_4;
    reg_struct.burst_len_valid = true;

    reg_struct.replicate       = 0;
    reg_struct.replicate_valid = true;

    cisx_idma_set_cfg(cisx_instance, dma_instance, &reg_struct);

    // Enable all the error interrupts
    memset(&irq_struct, 0, sizeof(struct cisx_idma_irqs));
    irq_struct.rresp         = true;
    irq_struct.bresp         = true;
    irq_struct.who           = true;

    cisx_idma_enable_irqs(cisx_instance, dma_instance, &irq_struct);
}

void cisx_idma_channel_reset( cisx_idma_chan_t channel )
{
    struct cisx_idma_reset_reg      rst_struct;
    struct cisx_idma_line_width_reg lw_struct;
    uint8_t cisx_instance, dma_instance;

    get_instance_from_idma_channel(channel, &cisx_instance, &dma_instance);

    // Reset IDMA
    memset(&rst_struct, 0, sizeof(struct cisx_idma_reset_reg));
    rst_struct.soft_reset       = 1;
    rst_struct.soft_reset_valid = true;
    cisx_idma_set_reset(cisx_instance, dma_instance, &rst_struct);

    cpu_spin_delay( 10 );  /* wait a little while to allow hw to reset */

    //rst_struct.soft_reset       = 0;
    //rst_struct.soft_reset_valid = true;
    //cisx_idma_set_reset(cisx_instance, dma_instance, &rst_struct);

    // Disable and ack all interrupts
    cisx_idma_disable_irqs(cisx_instance, dma_instance, NULL);
    cisx_idma_clear_irqs(cisx_instance, dma_instance, NULL);

    // Write back select power-on defaults
    // TODO: anything else needed?
    memset(&lw_struct, 0, sizeof(struct cisx_idma_line_width_reg));
    lw_struct.line_width       = 0;
    lw_struct.line_width_valid = true;
    cisx_idma_set_line_width(cisx_instance, dma_instance, &lw_struct);
}

static void cisx_idma_channel_launch( struct cisx_channel *cch )
{
    int i;
    struct ddma_descriptor_list *desc_list;
    struct cisx_idma_line_width_reg lw_struct;
    struct cisx_idma_desc_write_reg dw_struct;
    uint8_t cisx_instance, dma_instance;

    XASSERT( cch->is_open, cch->channel );

    cisx_dbg2( "%s: %s %d\n", __FUNCTION__, cch->name, cch->channel );

    get_instance_from_idma_channel(cch->channel, &cisx_instance, &dma_instance);

    /* use a shortcut to keep expression size down */
    desc_list = &cch->desc_list;

    /* once we start CISX, this list never changes. So flush *everything* */
    for( i=0 ; i<desc_list->num_descriptors ; i++ ) {
        ddma_desc_flush( &desc_list->list[i] );
    }

    /* fire in the hole! */
    memset(&lw_struct, 0, sizeof(struct cisx_idma_line_width_reg));
    lw_struct.line_width       = cch->bytes_per_row;
    lw_struct.line_width_valid = true;
    cisx_idma_set_line_width(cisx_instance, dma_instance, &lw_struct);

    memset(&dw_struct, 0, sizeof(struct cisx_idma_desc_write_reg));
    dw_struct.desc       = (uint32_t)desc_list->list[0].dma_ptr_self;
    dw_struct.desc_valid = true;
    cisx_idma_set_desc_write(cisx_instance, dma_instance, &dw_struct);
}


/*
 *  CISX DMA (both O & I) Function
 *  
 */

static void cisx_channel_sanity( struct cisx_channel *cch )
{
    struct ddma_descriptor_list *desc_list;

    cisx_dbg2( "%s %s %d\n", __FUNCTION__, cch->name, cch->channel );

    /* use a shortcut to reduce ->'s */
    desc_list = &cch->desc_list;

    if( !cch->is_open ) {
        XASSERT( desc_list->num_descriptors==0, desc_list->num_descriptors );
        XASSERT( desc_list->list==NULL, (uint32_t)desc_list->list );
        return;
    }

    if( desc_list->num_descriptors ) {
        cisx_desc_list_sanity( desc_list );
    }
    else {
        XASSERT( desc_list==NULL, (uint32_t)desc_list );
    }
}

static void cisx_channel_init( struct cisx_channel *cch, const char *name, uint8_t channel )
{
    memset( cch, 0, sizeof(struct cisx_channel) );

    strncpy( cch->name, name, MAX_CISX_CHANNEL_NAME );
    cch->channel = channel;
}

static scan_err_t cisx_channel_open( struct cisx_channel *cch, const char *name, 
                                    uint8_t channel, uint32_t bytes_per_row )
{
    scan_err_t scerr;
    int i;
    struct ddma_descriptor *desc;

    cisx_dbg2( "%s name=\"%s\" ch=%d bpr=%d\n", __FUNCTION__, name, channel, bytes_per_row );

    /* beware of memory leaks */
    XASSERT( cch->desc_list.list==NULL, (uint32_t)cch->desc_list.list );
    XASSERT( cch->desc_list.num_descriptors==0, cch->desc_list.num_descriptors );

    cisx_channel_init( cch, name, channel );

    cch->bytes_per_row = bytes_per_row;

    /* allocate a block of descriptors */
    scerr = ddma_descriptor_list_alloc( &cch->desc_list, name, MAX_CISX_DMA_DESCRIPTORS );
    if( scerr != SCANERR_NONE ) {
        /* ddma logs error */
        return scerr;
    }

    /* clear all the flags (we never stop, want no interrupts */
    for( i=0 ; i<cch->desc_list.num_descriptors ; i++ ) {
        desc = &cch->desc_list.list[i];
        desc->config_flags = 0;
        ddma_desc_flush( desc );
    }

    /* open for business! */
    cch->is_open = true;

    return SCANERR_NONE;
}

static void cisx_channel_close( struct cisx_channel *cch )
{
    cisx_dbg2( "%s %d\n", __FUNCTION__, cch->channel );

    XASSERT( cch->is_open, cch->channel );

    ddma_descriptor_list_free( &cch->desc_list );
    cch->is_open = false;
}

static void cisx_channel_load( struct cisx_channel *cch, 
                               uint8_t *cpu_addr, dma_addr_t dma_addr,
                               uint32_t num_rows, uint32_t bytes_per_row )
{
    int i;
    struct ddma_descriptor_list *desc_list;
    struct ddma_descriptor *desc;

    cisx_dbg2( "%s ch=%d name=\"%s\" nr=%d bpr=%d\n", __FUNCTION__, 
            cch->channel, cch->name, num_rows, bytes_per_row  );

    /* stupid human checks */
    XASSERT( bytes_per_row, cch->channel );
    XASSERT( num_rows, cch->channel );
    XASSERT( cpu_addr, cch->channel );
    XASSERT( dma_addr, cch->channel );

    desc_list = &cch->desc_list;

    XASSERT( desc_list->num_descriptors, cch->channel );
    XASSERT( desc_list->list != NULL, cch->channel );

    /* CISX is a circular list. All memory allocated before we start. We never
     * add new buffers once we're running. 
     *
     * I want simple. Don't bother with head/tail pointers because CISX is
     * pretty much autonomous once we're running. Fire/forget. 
     *
     * Thus we're doing a stupid linear search to add buffers to our desc_list.
     */
    for( i=0 ; i<desc_list->num_descriptors ; i++ ) {
        if( desc_list->list[i].src_addr==0 ) {
            /* found an empty slot! claim it */
            desc = &desc_list->list[i];
            desc->fw_src_addr = (void *)cpu_addr;

            dbg2( "%s cpu_addr=%x dma_addr=%x len=%d\n", __FUNCTION__, 
                    cpu_addr, dma_addr, num_rows*bytes_per_row );
            desc->src_addr = (uint32_t)dma_addr;
            desc->transfer_len_bytes = num_rows * bytes_per_row;
            break;
        }
    }

    /* if this assert fails, trying to add more buffers than we have
     * descriptors allocated. Shouldn't happen because of the way we're
     * allocating buffers.
     */
    XASSERT( i!=desc_list->num_descriptors, i );
}

/* 
 * CISX Control Functions
 *
 */

void cisx_dump( void )
{
   uint8_t cisx_instance;

    for (cisx_instance = 0; cisx_instance < NUM_CISX_INSTANCES; cisx_instance++)
    { 
        cisx_int_dump_regs(cisx_instance);
    }

    cisx_dma_dump();
}

void cisx_set_bypass( bool bypass )
{
    uint8_t cisx_instance;
    struct cisx_int_CisxCfg_reg cfg_struct;

    memset(&cfg_struct, 0, sizeof(struct cisx_int_CisxCfg_reg));

    // Bypass and enable are indeed connected: we can only enable
    // CISX when it is not bypassed.

    for (cisx_instance = 0; cisx_instance < NUM_CISX_INSTANCES; cisx_instance++)
    { 
        if( bypass ) {
            cfg_struct.bypass = 1;
            cfg_struct.enable = 0;
        }
        else {
            cfg_struct.bypass = 0;
            cfg_struct.enable = 1;
        }

        cfg_struct.bypass_valid = true;
        cfg_struct.enable_valid = true;
        cisx_int_set_CisxCfg(cisx_instance, &cfg_struct);

        /* 30-Jan-2013 ; only the first sensor is enabled if we're doing single sided scan */
        if (!bypass && SCANIMG_SINGLE_SENSOR(cisx_mm.sensor_bitmask)) {
            break;
        }
    }
}

void cisx_set_cmode( scan_cmode_t cmode )
{
    struct cisx_int_CisxCfg_reg cfg_struct;
    uint8_t cisx_instance;

    memset(&cfg_struct, 0, sizeof(struct cisx_int_CisxCfg_reg));

    for (cisx_instance = 0; cisx_instance < NUM_CISX_INSTANCES; cisx_instance++)
    { 
        if ( cmode == SCAN_CMODE_MONO ) 
        {
            cfg_struct.mode = CISX_INT_CFG_MODE_MONO_NO_SHUFFLE;
        }
        else 
        { 
            cfg_struct.mode = CISX_INT_CFG_MODE_CLR_NO_SHUFFLE;
        }
        cfg_struct.mode_valid   = true;
        cisx_int_set_CisxCfg(cisx_instance, &cfg_struct);
    }
}

void cisx_set_3channel_order( uint8_t cisx_3chan_order )
{
    uint8_t cisx_instance;
    struct cisx_int_CisxCfg_reg cfg_struct;

    memset(&cfg_struct, 0, sizeof(struct cisx_int_CisxCfg_reg));

    for (cisx_instance = 0; cisx_instance < NUM_CISX_INSTANCES; cisx_instance++)
    { 
        cfg_struct.ccd_order       = cisx_3chan_order;
        cfg_struct.ccd_order_valid = true;
        cisx_int_set_CisxCfg(cisx_instance, &cfg_struct);
    }
}

void cisx_set_pixels( cisx_odma_chan_t odma_channel, uint32_t num_dummy_pixels, uint32_t num_data_pixels, int append_flag )
{
    struct cisx_int_CisxCfg_reg  cfg_struct;
    struct cisx_int_Chan0Pix_reg ch0_struct;
    struct cisx_int_Chan1Pix_reg ch1_struct;
    struct cisx_int_Chan2Pix_reg ch2_struct;
    uint8_t cisx_instance, dma_instance;

    cisx_dbg2( "%s ch=%d dummy=%d data=%d append=%d\n", __FUNCTION__, 
            odma_channel, num_dummy_pixels, num_data_pixels, append_flag );

    get_instance_from_odma_channel(odma_channel, &cisx_instance, &dma_instance);

    memset(&cfg_struct, 0, sizeof(struct cisx_int_CisxCfg_reg));

    switch( odma_channel ) 
    {
        case CISX0_ODMA0 :
        case CISX1_ODMA0 :
            memset(&ch0_struct, 0, sizeof(struct cisx_int_Chan0Pix_reg));
            ch0_struct.chan0dum       = num_dummy_pixels;
            ch0_struct.chan0dum_valid = true;
            ch0_struct.chan0dat       = num_data_pixels;
            ch0_struct.chan0dat_valid = true;
            cisx_int_set_Chan0Pix(cisx_instance, &ch0_struct);

            cfg_struct.appchan0       = append_flag ? 1 : 0;
            cfg_struct.appchan0_valid = true;
            cisx_int_set_CisxCfg(cisx_instance, &cfg_struct);

            break;

        case CISX0_ODMA1 : 
        case CISX1_ODMA1 : 
            memset(&ch1_struct, 0, sizeof(struct cisx_int_Chan1Pix_reg));
            ch1_struct.chan1dum       = num_dummy_pixels;
            ch1_struct.chan1dum_valid = true;
            ch1_struct.chan1dat       = num_data_pixels;
            ch1_struct.chan1dat_valid = true;
            cisx_int_set_Chan1Pix(cisx_instance, &ch1_struct);

            cfg_struct.appchan1       = append_flag ? 1 : 0;
            cfg_struct.appchan1_valid = true;
            cisx_int_set_CisxCfg(cisx_instance, &cfg_struct);

            break;

        case CISX0_ODMA2 : 
        case CISX1_ODMA2 : 
            memset(&ch2_struct, 0, sizeof(struct cisx_int_Chan2Pix_reg));
            ch2_struct.chan2dum       = num_dummy_pixels;
            ch2_struct.chan2dum_valid = true;
            ch2_struct.chan2dat       = num_data_pixels;
            ch2_struct.chan2dat_valid = true;
            cisx_int_set_Chan2Pix(cisx_instance, &ch2_struct);

            cfg_struct.appchan2       = append_flag ? 1 : 0;
            cfg_struct.appchan2_valid = true;
            cisx_int_set_CisxCfg(cisx_instance, &cfg_struct);

            break;

        default : 
            XASSERT( 0, odma_channel );
            break;
    }
}

scan_err_t cisx_open( scan_cmode_t cmode, 
                      const struct cisx_sensor_conf *cisxsensor,
                      uint32_t sensor_bitmask)
{
    struct cisx_mm *mm;
    scan_err_t scerr;
    int i, channel_idx, num_descriptors;
    uint32_t bytes_per_row, bytes_per_row_padded, bytes_per_segment, num_active_pixels;
    struct cisx_channel *cch;
    struct ice_dma_buffer *icebuf;
    char name[DDMA_NAME_LEN+1];
    uint8_t *ptr;

    cisx_dbg2( "%s cmode=%d\n", __FUNCTION__, cmode );

    mm = &cisx_mm;

    /* 03-Apr-2013 ; adding run-time enabling of dual scan */
    mm->sensor_bitmask = sensor_bitmask;

    /* how many pixels are we going to get out of CISX? */
    num_active_pixels = 0;
    for( i=0 ; i<cisxsensor->num_segments ; i++ ) {
        num_active_pixels += cisxsensor->segments[i].num_data;
    }
        
    /* the sizeof(uint16_t) is because cisx works with 16-bit pixels */
    bytes_per_row = num_active_pixels * sizeof(uint16_t);

    /* only tested 3 segment sensors */
    XASSERT( cisxsensor->num_segments==3, cisxsensor->num_segments );

    /* 29-Jan-2013 ; adding dual scan */
    if( SCANIMG_SINGLE_SENSOR(cisx_mm.sensor_bitmask) ) {
        num_odma_channels = 3;
        odma_channels[0] = CISX0_ODMA0;
        odma_channels[1] = CISX0_ODMA1;
        odma_channels[2] = CISX0_ODMA2;
    }
    else {
        num_odma_channels = 6;
        odma_channels[0] = CISX0_ODMA0;
        odma_channels[1] = CISX0_ODMA1;
        odma_channels[2] = CISX0_ODMA2;
        odma_channels[3] = CISX1_ODMA0;
        odma_channels[4] = CISX1_ODMA1;
        odma_channels[5] = CISX1_ODMA2;
    }

    /* 25-Mar-2013 ; adding cbiout_pace value to cisx config. Older code
     * might not have this value set so use zero as an 'uninitialized' value.
     * This will probably case problems later when we *DO* want to use zero.
     */
    if( cisxsensor->cbiout_pace ) {
        cisx_dbg2( "%s: cbiout_pace=%d\n", __FUNCTION__,  cisxsensor->cbiout_pace);
        cisx_set_cbiout_pace( cisxsensor->cbiout_pace );
    }

    channel_idx = 0;
    while( channel_idx < num_odma_channels ) {
        for( i=0 ; i<cisxsensor->num_segments ; i++ ) {
            cch = &cisx_odma_channel[ odma_channels[channel_idx] ];

            cisx_set_pixels( odma_channels[channel_idx], 
                             cisxsensor->segments[i].num_dummy, 
                             cisxsensor->segments[i].num_data,  
                             cisxsensor->segments[i].append_flag  );

            /* the sizeof(uint16_t) is because cisx works with 16-bit pixels */
            bytes_per_segment = cisxsensor->segments[i].num_data * sizeof(uint16_t);

            strncpy( name, "cisxodma ", DDMA_NAME_LEN );
            name[8] = '0' + cch->channel;
            scerr = cisx_channel_open( cch, name, odma_channels[channel_idx], bytes_per_segment );
            if( scerr != SCANERR_NONE ) {
                cisx_close();
                return scerr;
            }

            cisx_odma_channel_enable( cch->channel );
            channel_idx += 1;
        }
    }

    /* 29-Jan-2013 ; adding dual scan */
    if( SCANIMG_SINGLE_SENSOR(cisx_mm.sensor_bitmask) ) {
        num_idma_channels = 1;
        idma_channels[0] = CISX0_IDMA0;
    }
    else {
        num_idma_channels = 2;
        idma_channels[0] = CISX0_IDMA0;
        idma_channels[1] = CISX1_IDMA0;
    }

    for( i=0 ; i<num_idma_channels ; i++ ) {
        cch = &cisx_idma_channel[ idma_channels[i] ];
        strncpy( name, "cisxidma ", DDMA_NAME_LEN );
        name[8] = '0' + cch->channel;
        scerr = cisx_channel_open( cch, name, idma_channels[i], bytes_per_row );
        if( scerr != SCANERR_NONE ) {
            cisx_close();
            return scerr;
        }
        cisx_idma_channel_enable( cch->channel );
    }

    /* 
     * At this point, we'll start allocating memory for the CISX buffers.
     */

    /* be paranoid about memory leaks */
    XASSERT( mm->num_readies==0, mm->num_readies );
    XASSERT( mm->num_running==0, mm->num_running );
    ASSERT( ATIsListEmpty( &mm->readies ) );
    ASSERT( ATIsListEmpty( &mm->running ) );

    /* 21-Jun-2011 ; pad up to dma */
    bytes_per_row_padded = ICE_DMA_ALIGN_ME( bytes_per_row );

    num_descriptors = MAX_CISX_DMA_DESCRIPTORS;
    /* if we're doing dual scan, we have two CISX mouths to feed */
    if( !SCANIMG_SINGLE_SENSOR(cisx_mm.sensor_bitmask) ) {
        num_descriptors *= 2;
    }
    mm->num_descriptors_allocated = 0;

    /* 22-Apr-2013 ; allocate an aggregate buffer (instead of multiple
     * smaller buffers). Want to get fast memory (e.g., SRAM) if possible and
     * the fastmem interface is simpler if we use one single buffer.
     */
    mm->cisx_buf_num_bytes = bytes_per_row_padded * num_descriptors;
    mm->cisx_buf_is_fast_memory = false;

    /* beware memory leaks */
    XASSERT( mm->cisx_buf==NULL, (uint32_t)mm->cisx_buf );

    mm->cisx_buf = scanmem_get_fast_memory(SCANMEM_TAG_CISX,mm->cisx_buf_num_bytes, 0);
    if( mm->cisx_buf!=NULL ) {
        /* we got fast memory! hurray! */
        dbg2( "%s using numbytes=%d of fast memory\n", __FUNCTION__, mm->cisx_buf_num_bytes );
        mm->cisx_buf_is_fast_memory = true;
    }
    else {
        mm->cisx_buf = ice_dma_alloc_buffer(mm->cisx_buf_num_bytes);
        if( mm->cisx_buf==NULL ) {
            dbg2( "%s unable to allocate numbytes=%d\n", __FUNCTION__, mm->cisx_buf_num_bytes );
            mm->cisx_buf_num_bytes = 0;
            return SCANERR_OUT_OF_MEMORY;
        }
        dbg2( "%s using numbytes=%d of regular memory\n", __FUNCTION__, mm->cisx_buf_num_bytes );
    }
    ptr = mm->cisx_buf;

    /* one icebuf per odma/idma descriptor */
    for( i=0 ; i<num_descriptors ; i++ ) {
        icebuf = icebuf_new_blank( ICEBUF_TAG_CISX );
        if( icebuf==NULL ) {
            return SCANERR_OUT_OF_MEMORY;
        }
        icebuf->data = ptr;
        /* Don't use icebuf->dma_handle or dma_map_single here. We'll do all
         * that in the launch function below.
         */
        icebuf->num_rows = 1;
        icebuf->max_rows = 1;
        icebuf->bytes_per_row = bytes_per_row_padded;
        icebuf->datalen = icebuf->max_rows * icebuf->bytes_per_row;
        ptr += icebuf->datalen;

        /* put it on the readies list */
        ATInsertTailList( &mm->readies, &icebuf->listnode );
        mm->num_readies++;
        mm->num_descriptors_allocated++;
    }

    cisx_set_cmode( cmode );

    if( !SCANIMG_SINGLE_SENSOR(cisx_mm.sensor_bitmask) ) {
        scerr = cisx_duplex_setup();
        XASSERT( scerr==SCANERR_NONE, scerr );
    }

    cisx_set_3channel_order( cisxsensor->chan_order );

    return SCANERR_NONE;
}

static scan_err_t cisx_launch_single( struct ice_dma_buffer *icebuf, 
                                      struct cisx_block *block )
{
    int i;
    uint8_t *cpu_addr[3];
    dma_addr_t dma_addr[3];
    struct cisx_channel *cch;

    cisx_dbg2( "%s name=%s\n", __FUNCTION__, block->name );

    /* XXX note I've hardwired to three segment sensor */

    cpu_addr[0] = icebuf->data;
    cpu_addr[1] = cpu_addr[0] + cisx_odma_channel[0].bytes_per_row;
    cpu_addr[2] = cpu_addr[1] + cisx_odma_channel[1].bytes_per_row;

#ifdef __KERNEL__
    for( i=0 ; i<3 ; i++ ) {
        /* 23-Apr-2013 ; if we're using memory not CPU dcache mappable
         * (e.g., sram), don't dma map it.
         */
        if( cisx_mm.cisx_buf_is_fast_memory ) {
            dma_addr[i] = (dma_addr_t)cpu_addr[i];
        }
        else {
            dma_addr[i] = dma_map_single( NULL, (void *)cpu_addr[i],
                                          icebuf->num_rows * icebuf->bytes_per_row, 
                                          DMA_BIDIRECTIONAL );
            XASSERT( dma_addr[i], (uint32_t)cpu_addr[i] );
        }
    }
#else 
    for( i=0 ; i<3 ; i++ ) {
        dma_addr[i] = (dma_addr_t)cpu_addr[i];
    }
#endif

    cisx_dbg2( "%s %p %x %x %x\n", __FUNCTION__, icebuf->data, cpu_addr[0], cpu_addr[1], cpu_addr[2] );

    for( i=0 ; i<3 ; i++ ) {
        cch = &cisx_odma_channel[ block->odma_channels[i] ];
        cisx_channel_load( cch, cpu_addr[i], dma_addr[i], icebuf->num_rows, cch->bytes_per_row );
    }

    cch = &cisx_idma_channel[ block->idma_channel ];
    cisx_channel_load( cch, cpu_addr[0], dma_addr[0], icebuf->num_rows, cch->bytes_per_row );

    return SCANERR_NONE;
}

void cisx_launch( void )
{
    scan_err_t scerr;
    struct cisx_mm *mm;
    int i;
    struct ice_dma_buffer *icebuf;
    int num_cisx, cisx_block_idx;
    struct cisx_block cisx_block_list[2];

    cisx_dbg2( "%s\n", __FUNCTION__ );

    mm = &cisx_mm;

    memset( cisx_block_list, 0, sizeof(cisx_block_list) );
    num_cisx = 1;
    strcpy( cisx_block_list[0].name, "CISX0" );
    for( i=0 ; i<3 ; i++ ) {
        cisx_block_list[0].odma_channels[i] = odma_channels[i];
    }
    cisx_block_list[0].idma_channel = idma_channels[0];

    if( !SCANIMG_SINGLE_SENSOR(cisx_mm.sensor_bitmask) ) {
        XASSERT( num_odma_channels==6, num_odma_channels );
        XASSERT( num_idma_channels==2, num_idma_channels );
        num_cisx++;
        strcpy( cisx_block_list[1].name, "CISX1" );
        for( i=0 ; i<3 ; i++ ) {
            cisx_block_list[1].odma_channels[i] = odma_channels[i+3];
        }
        cisx_block_list[1].idma_channel = idma_channels[1];
    }

    cisx_set_bypass( false );

    /* pull everything from the ready list, push into hardware, add to
     * descriptor DMA 
     */
    cisx_block_idx = 0;
    while( 1 ) {
        icebuf = icebuf_pop( &mm->readies );
        if( icebuf==NULL ) {
            break;
        }
        icebuf_sanity( icebuf );

        mm->num_readies--;
        XASSERT( mm->num_readies>=0, mm->num_readies );

        scerr = cisx_launch_single( icebuf, &cisx_block_list[cisx_block_idx] );
        XASSERT( scerr==SCANERR_NONE, scerr ); 

        /* ping pong back and forth between the CISX blocks */
        cisx_block_idx = (cisx_block_idx+1) % num_cisx;

        /* attach to the end of our running queue */
        ATInsertTailList( &mm->running, &icebuf->listnode );
        mm->num_running++;
    }

    for( i=0 ; i<num_odma_channels ; i++ ) {
        cisx_odma_channel_launch( &cisx_odma_channel[ odma_channels[i] ] );
    }
    for( i=0 ; i<num_idma_channels ; i++ ) {
        cisx_idma_channel_launch( &cisx_idma_channel[ idma_channels[i] ] );
    }

    cisx_interrupt_enable();

//    /* 29-Jun-2011 ; XXX temp debug */
//    cisx_dump();
//    cisx_ddma_dump();
}

void cisx_sanity( void )
{
    struct cisx_mm *mm;
    int i;
    int cnt;

    mm = &cisx_mm;

    for( i=0 ; i<CISX_ODMA_NUM_CHANNELS ; i++ ) {
        cisx_channel_sanity( &cisx_odma_channel[i] );
    }
    for( i=0 ; i<CISX_IDMA_NUM_CHANNELS ; i++ ) {
        cisx_channel_sanity( &cisx_idma_channel[i] );
    }

    cisx_dbg2( "%s running=%d readies=%d\n", __FUNCTION__, mm->num_running,
            mm->num_readies );

    cnt = mm->num_running + mm->num_readies;
    XASSERT( cnt==mm->num_descriptors_allocated, cnt );

    cnt = count_list( &mm->readies );
    XASSERT( cnt==mm->num_readies, cnt );

    cnt = count_list( &mm->running );
    XASSERT( cnt==mm->num_running, cnt );
}

void cisx_close( void )
{
    struct cisx_mm *mm;
    int i;
    struct ice_dma_buffer *icebuf;

    cisx_dbg2( "%s\n", __FUNCTION__ );

    mm = &cisx_mm;

    XASSERT( mm->num_readies==0, mm->num_readies );

    /* turn off CISX */
    cisx_interrupt_disable();
    cisx_set_bypass( true );

    /* pull all icebufs from running list, put back on ready list */
    while( 1 ) {
        icebuf = icebuf_pop( &mm->running );
        if( icebuf==NULL ) {
            break;
        }
        mm->num_running--;
        XASSERT( mm->num_running >=0, mm->num_running );

        /* attach to the end of our running queue */
        ATInsertTailList( &mm->readies, &icebuf->listnode );
        mm->num_readies++;
        XASSERT( mm->num_readies <= mm->num_descriptors_allocated, mm->num_readies ); 

        /* The memory was allocated in one aggregate chunk then dividied
         * between the icebufs. Clear out the data pointers. We'll free the
         * aggregate buffer later.
         */
        icebuf->data = NULL;
        icebuf->dma_handle = (dma_addr_t)NULL;
        icebuf->num_rows = 0;
        icebuf->max_rows = 0;
        icebuf->datalen = 0;
        icebuf->bytes_per_row = 0;
    }

    /* free the memory we allocated for this channel */
    ice_free_counted_list( &mm->readies, mm->num_readies );
    mm->num_readies = 0;
    
    XASSERT( mm->num_running==0, mm->num_running );
    ASSERT( ATIsListEmpty( &mm->running ) );

    for( i=0 ; i<num_odma_channels; i++ ) {
        cisx_odma_channel_disable( odma_channels[i] );
        cisx_channel_close( &cisx_odma_channel[ odma_channels[i] ] );
    }
    for( i=0 ; i<num_idma_channels; i++ ) {
        cisx_idma_channel_disable( idma_channels[i] );
        cisx_channel_close( &cisx_idma_channel[ idma_channels[i] ] );
    }

    /* 22-Apr-2013 ; release our aggregate cisx buffer */
    if( mm->cisx_buf_is_fast_memory ) {
        scanmem_free_fast_memory( SCANMEM_TAG_CISX, mm->cisx_buf, mm->cisx_buf_num_bytes, 0 );
        mm->cisx_buf = NULL;
        mm->cisx_buf_num_bytes = 0;
    }
    else {
        PTR_FREE( mm->cisx_buf );
        mm->cisx_buf_num_bytes = 0;
    }

    /* clean up the hardware */
    cisx_reset();
}

void cisx_dma_dump( void )
{
    int i;

    // REVISIT: this will only output data for actively open channels,
    //          may not be what we want to see
    //cisx_dbg2( "%s: num_odma=%d num_idma=%d\n",
    //           __FUNCTION__, num_odma_channels, num_idma_channels );
    
    for( i=0 ; i < num_odma_channels ; i++ ) {
        odma_channel_dump( odma_channels[i] );
    }
    for( i=0 ; i < num_idma_channels ; i++ ) {
        idma_channel_dump( idma_channels[i] );
    }

    cisx_ddma_dump();
}

void cisx_ddma_dump( void )
{
    int i;

    //cisx_dbg2( "%s: num_odma=%d num_idma=%d\n",
    //            __FUNCTION__, CISX_ODMA_NUM_CHANNELS, CISX_IDMA_NUM_CHANNELS );

    for( i=0 ; i<CISX_ODMA_NUM_CHANNELS ; i++ ) {
        if( cisx_odma_channel[i].desc_list.list ) {
            ddma_descriptor_list_dump( &cisx_odma_channel[i].desc_list);
        }
    }
    for( i=0 ; i<CISX_IDMA_NUM_CHANNELS ; i++ ) {
        if( cisx_idma_channel[i].desc_list.list ) {
            ddma_descriptor_list_dump( &cisx_idma_channel[i].desc_list);
        }
    }

    cisx_ddma_data_peek();
}

void cisx_ddma_data_peek( void )
{
    struct cisx_mm *mm;
    struct ice_dma_buffer *icebuf;

    mm = &cisx_mm;

    cisx_dbg2( "%s: num_running=%d\n", __FUNCTION__, mm->num_running );

    // This isn't very useful if we don't have anything running (and could
    // actually explode due to bogus icebuf pointers)
    if (mm->num_running == 0)
    {
        return;
    }

//    XASSERT( mm->num_running==MAX_CISX_DMA_DESCRIPTORS, mm->num_running );
    icebuf = CONTAINING_RECORD( mm->running.nextEntry, struct ice_dma_buffer, listnode );
    ASSERT( icebuf );

    icebuf_sanity( icebuf );
    
//    cpu_dcache_invalidate_region( icebuf->data, 64 );
//    scanlog_hex_dump( icebuf->data, 64 );
    
#if 0
    /* The following little snippet of code will peek at the first buffer of
     * the ODMA and the IDMA channels. The buffer at the head of each
     * descriptor list is hexdump'd.
     */
    int i;
    struct cisx_channel *cch;
    volatile CISX_ODMA_REGS_t *odma_regs;

    for( i=0 ; i<CISX_ODMA_NUM_CHANNELS ; i++ ) {
        cch = &cisx_odma_channel[i];
        dbg2( "%s %d i=%d ch=%d\n", __FUNCTION__, __LINE__, i, cch->channel );
        odma_regs = getregs( cch->channel );
        ddma_data_peek( &cch->desc_list, odma_regs->line_size );
    }
    ddma_data_peek( &cisx_idma_channel[0].desc_list, idma_regs->line_width );
#endif

#if 0
    /* this big snippet of code will peek at every single buffer in the IDMA
     * descriptor chain
     */
    uint8_t *ptr;
    struct ddma_descriptor *head_desc;
    struct ddma_descriptor *desc;
    int idma_idx;
    struct cisx_channel *idma;

    for( idma_idx=0 ; idma_idx<CISX_IDMA_NUM_CHANNELS ; idma_idx++ ) {
        dbg2( "%s idma_idx=%d\n", __FUNCTION__, idma_idx );

        idma = &cisx_idma_channel[idma_idx];
        if( !idma || !idma->desc_list.list ) {

            dbg2( "%s %d should not see this!\n", __FUNCTION__, __LINE__ );

            continue;
        }

        head_desc = &cisx_idma_channel[idma_idx].desc_list.list[0];
        desc = head_desc;
        do { 
            ddma_desc_dump( desc );

            XASSERT( desc->fw_src_addr, (uint32_t)desc );
            ptr = (uint8_t *)desc->fw_src_addr;
            if( ptr ) {
                if( 1 ) {
                    /* dump a small chunk */
                    cpu_dcache_invalidate_region( ptr, 32 );
                    scanlog_hex_dump( ptr, 32 );
                }
                else {
                    /* dump entire buffer */
                    dbg2( "%s ptr=%p len=%d %d\n", __FUNCTION__, 
                                ptr, desc->transfer_len_bytes, cpu_get_dcache_line_size() );
                    cpu_dcache_invalidate_region( ptr, desc->transfer_len_bytes & ICE_DMA_BYTE_ALIGN_MASK );
    //                cpu_dcache_invalidate_region( ptr, desc->transfer_len_bytes );
                    scanlog_hex_dump( ptr, desc->transfer_len_bytes );
                }
            }
            desc = (struct ddma_descriptor *)desc->fw_next_descriptor_addr;
        } while( desc != head_desc );
    }
#endif
}

void cisx_dma_reset( void )
{
    int i;

    for( i=0 ; i<CISX_ODMA_NUM_CHANNELS ; i++ ) {
        cisx_odma_channel_disable( i );
        cisx_odma_channel_reset( i );
    }
    for( i=0 ; i<CISX_IDMA_NUM_CHANNELS ; i++ ) {
        cisx_idma_channel_disable( i );
        cisx_idma_channel_reset( i );
    }
}

void cisx_reset( void )
{
    struct cisx_int_CisxCfg_reg           cfg_struct;
    struct cisx_int_Chan0Pix_reg          ch0_struct;
    struct cisx_int_Chan1Pix_reg          ch1_struct;
    struct cisx_int_Chan2Pix_reg          ch2_struct;
    struct cisx_int_TBL_CNT_reg           tbl_cnt_reg;
    struct cisx_int_ODMA_TBL_LUT_reg      odma_reg;
    //struct cisx_int_COLOR_OUT_TBL_LUT_reg clr_reg;
    uint8_t cisx_instance, i;

    /* shut it all off before we start poking at it */
    cisx_interrupt_disable();
    cisx_set_bypass( true );

    for (cisx_instance = 0; cisx_instance < NUM_CISX_INSTANCES; cisx_instance++)
    { 
        /* restore CISX to power-on defaults */
        memset(&cfg_struct, 0, sizeof(struct cisx_int_CisxCfg_reg));
        cfg_struct.bypass         = 1;
        cfg_struct.bypass_valid   = true;
        cfg_struct.numchans       = CISX_INT_CFG_NUMCHANS_3;
        cfg_struct.numchans_valid = true;
        cisx_int_set_CisxCfg(cisx_instance, &cfg_struct);
 
        /* sanity check to make sure my code is pointed at the right addresses */
        memset(&cfg_struct, 0, sizeof(struct cisx_int_CisxCfg_reg));
        cisx_int_get_CisxCfg(cisx_instance, &cfg_struct);

        if ((cfg_struct.bypass != 1) || (cfg_struct.numchans != CISX_INT_CFG_NUMCHANS_3))
        {
            dbg1( "%s CisxCfg not as expected\n", __FUNCTION__ );
            XASSERT(0, 0);
        }

        // Reset the color and odma luts
        memset(&tbl_cnt_reg, 0, sizeof(struct cisx_int_TBL_CNT_reg));
        tbl_cnt_reg.color_out_tbl_cnt       = 0;
        tbl_cnt_reg.odma_tbl_cnt            = 2;
        tbl_cnt_reg.color_out_tbl_cnt_valid = true;
        tbl_cnt_reg.odma_tbl_cnt_valid      = true;
        cisx_int_set_TBL_CNT(cisx_instance, &tbl_cnt_reg);

        memset(&odma_reg, 0, sizeof(struct cisx_int_ODMA_TBL_LUT_reg));
        //memset(&clr_reg,  0, sizeof(struct cisx_int_COLOR_OUT_TBL_LUT_reg));
        for (i = 0; i <= 2; i++)
        {
            odma_reg.d       = i;
            odma_reg.d_valid = true;
            cisx_int_set_ODMA_TBL_LUT(cisx_instance, &odma_reg, i);

            // NOTE: color table only used for sensors where all three color
            //   planes for a pixel are grouped together, rather than grouping
            //   all pixels for a single color plane together 
            //clr_reg.d       = 0;
            //clr_reg.d_valid = true;
            //cisx_int_set_COLOR_OUT_TBL_LUT(cisx_instance, &clr_reg, i);
        }

        // Reset the pixel counts (ch0, ch1, ch2)
        memset(&ch0_struct, 0, sizeof(struct cisx_int_Chan0Pix_reg));
        ch0_struct.chan0dum       = 0;
        ch0_struct.chan0dum_valid = true;
        ch0_struct.chan0dat       = 0;
        ch0_struct.chan0dat_valid = true;
        cisx_int_set_Chan0Pix(cisx_instance, &ch0_struct);

        memset(&ch1_struct, 0, sizeof(struct cisx_int_Chan1Pix_reg));
        ch1_struct.chan1dum       = 0;
        ch1_struct.chan1dum_valid = true;
        ch1_struct.chan1dat       = 0;
        ch1_struct.chan1dat_valid = true;
        cisx_int_set_Chan1Pix(cisx_instance, &ch1_struct);

        memset(&ch2_struct, 0, sizeof(struct cisx_int_Chan2Pix_reg));
        ch2_struct.chan2dum       = 0;
        ch2_struct.chan2dum_valid = true;
        ch2_struct.chan2dat       = 0;
        ch2_struct.chan2dat_valid = true;
        cisx_int_set_Chan2Pix(cisx_instance, &ch2_struct);

        /* ack all pending interrupts */
        cisx_int_disable_irqs(cisx_instance, NULL);
        cisx_int_clear_irqs(cisx_instance, NULL);
    }

    cisx_dma_reset();

    num_odma_channels = 0;
    num_idma_channels = 0;
    memset( odma_channels, 0, sizeof(odma_channels) );
    memset( idma_channels, 0, sizeof(idma_channels) );
}

scan_err_t cisx_soft_setup( void )
{
    uint8_t cisx_instance;
    uint32_t num32;
    struct cisx_int_CisxCfg_reg  cfg_struct;

    cisx_dbg2( "%s\n", __FUNCTION__ );

    cisx_reset();

    //cisx_interrupt_enable();

    for (cisx_instance = 0; cisx_instance < NUM_CISX_INSTANCES; cisx_instance++)
    { 
        /* 25-Mar-2013 ; The cbiout_pace can be overwritten by cisxsensor
         * structure passed to cisx_open(). The cisxsensor structure is
         * typically read from platform specific code scansen_get_cisx_conf().
         */

        if( !SCANIMG_SINGLE_SENSOR(cisx_mm.sensor_bitmask) ) {
            scands_get_integer_with_default( "cbiout_pace", &num32, 0 );
        }
        else {
            scands_get_integer_with_default( "cbiout_pace", &num32, 0 );
        }
    
        cisx_dbg2( "%s: cbiout_pace=%d\n", __FUNCTION__,  num32);

        memset(&cfg_struct, 0, sizeof(struct cisx_int_CisxCfg_reg));
        cfg_struct.cbiout_pace       = num32;
        cfg_struct.cbiout_pace_valid = true;
        cfg_struct.numchans          = CISX_INT_CFG_NUMCHANS_3;
        cfg_struct.numchans_valid    = true;
        cisx_int_set_CisxCfg(cisx_instance, &cfg_struct);
    }

    /* 28-Jun-2011 ; default to CCD_ORDER_120 which isn't 0,1,2 but compensates for
     * the two pixels the scan block eats (which throws off the state machines)
     */
    cisx_set_3channel_order( CISX_INT_CFG_CCD_ORDER_120 );

    cisx_set_bypass( true );

    return SCANERR_NONE;
}


scan_err_t cisx_onetime_init( void )
{
    struct cisx_mm *mm;
    int i;
    uint8_t cisx_instance;

    cisx_dbg2( "%s\n", __FUNCTION__ );

    mm = &cisx_mm;

    // Disable all interrupts
    cisx_interrupt_disable();

    // Register for interrupt callbacks
    for (cisx_instance = 0; cisx_instance < NUM_CISX_INSTANCES; cisx_instance++)
    { 
        for (i = 0; i < CISX_MAX_IDMA_PER_BLK; i++)
        {
            cisx_idma_clear_irqs(cisx_instance, i, NULL);
            cisx_idma_register_irq_callback(cisx_instance, i, cisx_idma_interrupt);
        }

        for (i = 0; i < CISX_MAX_ODMA_PER_BLK; i++)
        {
            cisx_odma_clear_irqs(cisx_instance, i, NULL);
            cisx_odma_register_irq_callback(cisx_instance, i, cisx_odma_interrupt);
        }

        cisx_int_clear_irqs(cisx_instance, NULL);
        cisx_int_register_irq_callback(cisx_instance, &cisx_int_interrupt);
    }

    cisx_soft_setup();

    /* our dma descriptors must be padded to a cache line size so we can flush
     * them from cache without breaking neighboring cache lines
     */
    XASSERT( sizeof(struct ddma_descriptor)==cpu_get_dcache_line_size(), 
            sizeof(struct ddma_descriptor));

    /* initialize global data structures */
    for( i=0 ; i<CISX_ODMA_NUM_CHANNELS ; i++ ) {
        cisx_channel_init( &cisx_odma_channel[i], "odma", i );
    }
    for( i=0 ; i<CISX_IDMA_NUM_CHANNELS ; i++ ) {
        cisx_channel_init( &cisx_idma_channel[i], "idma", i );
    }

    ATInitList( &mm->readies );
    ATInitList( &mm->running );
    mm->num_readies = 0;
    mm->num_running = 0;

//    /* 25-Apr-2011 ; XXX temp debug */
//    cisx_dump();

    return SCANERR_NONE;
}

scan_err_t cisx_cleanup_module( void )
{
//    cisx_release_interrupt();

    return SCANERR_NONE;
}

void cisx_get_pixels_per_row( uint32_t *ppr ) 
{
    uint32_t total;
    uint8_t  cisx_instance;
    struct cisx_int_Chan0Pix_reg ch0_struct;
    struct cisx_int_Chan1Pix_reg ch1_struct;
    struct cisx_int_Chan2Pix_reg ch2_struct;

    /* test/debug function ; get total pixel count from chanNpix registers */
    total = 0;

    /* Note only reading the first CISX. */
    cisx_instance = 0;

    /* only capture one channel ; all channels should be identical */
    memset(&ch0_struct, 0, sizeof(struct cisx_int_Chan0Pix_reg));
    cisx_int_get_Chan0Pix(cisx_instance, &ch0_struct);
    total += ch0_struct.chan0dum;
    total += ch0_struct.chan0dat;

    memset(&ch1_struct, 0, sizeof(struct cisx_int_Chan1Pix_reg));
    cisx_int_get_Chan1Pix(cisx_instance, &ch1_struct);
    total += ch1_struct.chan1dum;
    total += ch1_struct.chan1dat;

    memset(&ch2_struct, 0, sizeof(struct cisx_int_Chan2Pix_reg));
    cisx_int_get_Chan2Pix(cisx_instance, &ch2_struct);
    total += ch2_struct.chan2dum;
    total += ch2_struct.chan2dat;

    *ppr = total;
}

uint32_t cisx_get_int_pending( void )
{
    // FIXME: the new driver architecture does not allow for direct polling
    //        of IRQ registers.  This only used by test code, should be OK ...
    return 0;
#if 0
    volatile CISX_INT_REGS_t * cisxregs;

    /* This function used during CISX regression testing. Rather than
     * triggering the interrupt then trying to do huge logging dumps from
     * interrupt context, we can poll on the CISX interrupt status.
     */

    /* 08-Mar-2013 ; XXX for now, hardwire to 0'th CISX */
    cisxregs = cisx_regs_list[0];
    return cisxregs->IntPend;
#endif
}

/**
 * \brief  allow Cfg.cbiout_pace to be platform specific. 
 *
 * "CBI Output Pacing"
 *
 * VallidOut frequency = BusClkFreq/(cbiout_pace + 2) 
 * ValidOut frequency (max) = BusClkFreq/2
 * 
 * \author David Poole
 * \date 25-Mar-2013
 */

void cisx_set_cbiout_pace( uint32_t cbiout_pace )
{
    uint8_t  cisx_instance;
    uint32_t num32;
    struct cisx_int_CisxCfg_reg  cfg_struct;

    if( !SCANIMG_SINGLE_SENSOR(cisx_mm.sensor_bitmask) ) {
        scands_get_integer_with_default( "cbiout_pace", &num32, cbiout_pace );
    }
    else {
        scands_get_integer_with_default( "cbiout_pace", &num32, cbiout_pace );
    }

    cisx_dbg2( "%s: cbiout_pace=%d\n", __FUNCTION__, num32 );

    memset(&cfg_struct, 0, sizeof(struct cisx_int_CisxCfg_reg));
    cfg_struct.cbiout_pace       = num32;
    cfg_struct.cbiout_pace_valid = true;

    for (cisx_instance = 0; cisx_instance < NUM_CISX_INSTANCES; cisx_instance++)
    { 
        cisx_int_set_CisxCfg(cisx_instance, &cfg_struct);
    }
}

/**
 * \brief  Get CISX Dual Channel enable/disable state
 *
 * Created for sanity checking. All of Scan block, CISX, and PIC must be in
 * agreement for dual channel mode.
 *
 * \author David Poole
 * \date 03-Apr-2013
 */

bool cisx_get_dual_channel_enabled( void )
{
    bool enabled;
    int  cisx_instance;
    struct cisx_int_CisxCfg_reg  cfg_struct;

#if !defined HAVE_NSENSOR_SUPPORT
    /* no chance in chocolate syrup you'll ever have dual scan enabled */
    return false;
#endif

    /* all CISX blocks enabled => dual scan */
    enabled = true;
    for (cisx_instance = 0; cisx_instance < NUM_CISX_INSTANCES; cisx_instance++)
    { 

        memset(&cfg_struct, 0, sizeof(struct cisx_int_CisxCfg_reg));
        cisx_int_get_CisxCfg(cisx_instance, &cfg_struct);

        if( cfg_struct.bypass == 1 ) {
            /* bit set => this channel in bypass */

            enabled = false;
        }
    }

    /* sneak in some sanity checking */
    if( enabled ) {
        XASSERT( num_odma_channels==6, num_odma_channels );
        XASSERT( num_idma_channels==2, num_idma_channels );
    }

    /* make sure hardware agrees with our global flag */
    XASSERT( enabled==(!SCANIMG_SINGLE_SENSOR(cisx_mm.sensor_bitmask)), cisx_mm.sensor_bitmask);

    return enabled;
}

