/*
**************************************************************************
This Source Code Form is subject to the terms of the Mozilla Public
License, v. 2.0. If a copy of the MPL was not distributed with this file,
You can obtain one at http://mozilla.org/MPL/2.0/.

Copyright (c) 2006-2015, Marvell International Ltd.

Alternatively, this software may be distributed under the terms of the GNU
General Public License Version 2, and any use shall comply with the terms and
conditions of the GPL.  A copy of the GPL is available at
http://www.gnu.org/licenses/old-licenses/gpl-2.0.html

THE FILE IS DISTRIBUTED AS-IS, WITHOUT WARRANTY OF ANY KIND, AND THE
IMPLIED WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE
ARE EXPRESSLY DISCLAIMED.  The GPL license provides additional details about
this warranty disclaimer.
******************************************************************************
*/


/**
 * \file cal.c
 *
 * \brief Scanner calibration ASIC-dependent  routines.
 *
 */

#include <math.h>
#include <stdlib.h>
#include <string.h>
#include <stdarg.h>

#include "scos.h" 

#include "list.h"           
#include "lassert.h"

#include "scancore.h"
#include "scantypes.h"
#include "scandbg.h"
#include "scansen.h"
#include "scanvars.h"
#include "scanmech.h"
#include "safetylock.h"
#include "icedma.h"

#include "pic_handle_if.h"
#include "pic_if.h"
#include "pic_convenience_if.h"

#include "pic.h"
#include "cal.h"
#include "scands.h"
#include "cal_common.h"



/* forward declarations */

static int cal_hw_setup(struct pic_handle_t *pic_handle, uint32_t sensor_num, const CAL_CONFIG *pconfig);

static cal_err_t processCalibrationCase(struct pic_handle_t **pic_handle, CalibrationCase *pCCase);


static const CAL_CONFIG calconf_32BITPRNUDSNU_NOQUAD = { 
                                              USE_QUAD:false,
                                              PRNUDSNU_LUT_ENTRY_SIZE:4,
                                              PRNUDSNU_LUT_USE_16BIT_PRNU_DSNU:true,
                                              DSNU_SHIFT:0,
                                              MAX_DIGITAL_CORRECTION:0xFFFF,
                                              DIGITAL_GAIN_PRECISION:14,
                                              MAX_DIGITAL_GAIN_TIMES_10:50,
                                              ANALOG_GAIN_CAP_TIMES_10:79,
                                              COLOR_SAT_SLOPE_THRESHOLD:95,
                                              MONO_SAT_SLOPE_THRESHOLD:95,
                                              REPLACE_BAD_PIXEL:0xFFFF,
                                              BYPASS_REPLACE_BAD_PIXEL: false,
                                              hw_setup: cal_hw_setup,
                                              build_case: create_calibration_case,
                                              process_case: processCalibrationCase };

static const CAL_CONFIG calconf_24BITPRNUDSNU_NOQUAD = { 
                                              USE_QUAD:false,
                                              PRNUDSNU_LUT_ENTRY_SIZE:3,
                                              PRNUDSNU_LUT_USE_16BIT_PRNU_DSNU:false,
                                              DSNU_SHIFT:4,
                                              MAX_DIGITAL_CORRECTION:0xFFF,
                                              DIGITAL_GAIN_PRECISION:10,
                                              MAX_DIGITAL_GAIN_TIMES_10:50,
                                              ANALOG_GAIN_CAP_TIMES_10:79,
                                              COLOR_SAT_SLOPE_THRESHOLD:95,
                                              MONO_SAT_SLOPE_THRESHOLD:95,
                                              REPLACE_BAD_PIXEL:0xFFF,
                                              BYPASS_REPLACE_BAD_PIXEL: false,
                                              hw_setup: cal_hw_setup,
                                              build_case: create_calibration_case,
                                              process_case: processCalibrationCase };

static const CAL_CONFIG calconf_24BITPRNUDSNU_QUAD = {  
                                              USE_QUAD:true,
                                              PRNUDSNU_LUT_ENTRY_SIZE:4,
                                              PRNUDSNU_LUT_USE_16BIT_PRNU_DSNU:false,
                                              DSNU_SHIFT:4,
                                              MAX_DIGITAL_CORRECTION:0xFFF,
                                              DIGITAL_GAIN_PRECISION:10,
                                              MAX_DIGITAL_GAIN_TIMES_10:50,
                                              ANALOG_GAIN_CAP_TIMES_10:79,
                                              COLOR_SAT_SLOPE_THRESHOLD:95,
                                              MONO_SAT_SLOPE_THRESHOLD:95,
                                              REPLACE_BAD_PIXEL:0xFFF,
                                              BYPASS_REPLACE_BAD_PIXEL: false,
                                              hw_setup: cal_hw_setup,
                                              build_case: create_calibration_case,
                                              process_case: processCalibrationCase };

const CAL_CONFIG *cal_get_config( void )
{
    /* davep 22-Oct-2012 ; disable quad until cal strip overscan is stable */
    return &calconf_24BITPRNUDSNU_NOQUAD;
//    return &calconf_24BITPRNUDSNU_QUAD;
}

/*
    One time configuration of the calibration
*/
static int cal_hw_setup(struct pic_handle_t *pic_handle, uint32_t sensor_num, const CAL_CONFIG *pconfig)
{
    int pd_corr_bits, pd_coeff_width;

    pd_corr_bits = 12;
    pd_coeff_width = 12;
    
    if((pconfig->PRNUDSNU_LUT_ENTRY_SIZE != 4) &&  (pconfig->PRNUDSNU_LUT_ENTRY_SIZE != 3))
        return 1;
    
    /* Check everything before set hardware*/
    if(pconfig->USE_QUAD)
    { 
#ifndef HAVE_PIC_PC_QUAD
        return 1;           
#endif
        /* can not do 32-bit PRNU/DSNU when QUAD is enabled*/
        if (pconfig->PRNUDSNU_LUT_USE_16BIT_PRNU_DSNU)
            return 2;
        
        if(pconfig->PRNUDSNU_LUT_ENTRY_SIZE != 4)
            return 3;
    }else
    {
        if(pconfig->PRNUDSNU_LUT_ENTRY_SIZE ==4 && !pconfig->PRNUDSNU_LUT_USE_16BIT_PRNU_DSNU)
            return 4;
        
        if(pconfig->PRNUDSNU_LUT_ENTRY_SIZE !=4 && pconfig->PRNUDSNU_LUT_USE_16BIT_PRNU_DSNU)
            return 5;
    }
    
    /* TBD: Do more thorough check*/
    if(pconfig->DIGITAL_GAIN_PRECISION >=12 && !pconfig->PRNUDSNU_LUT_USE_16BIT_PRNU_DSNU)
        return 6;
    
    // make sure that the saturation slope thresholds are reasonable
    if((pconfig->COLOR_SAT_SLOPE_THRESHOLD > 100) || (pconfig->MONO_SAT_SLOPE_THRESHOLD > 100))
        return 7;
    
    if(pconfig->USE_QUAD || !pconfig->PRNUDSNU_LUT_USE_16BIT_PRNU_DSNU)
    {
        pd_corr_bits = 24;
    }
    else
    {
        pd_corr_bits = 32;
    }

    if(pconfig->USE_QUAD)
    {
        /* hardwire the thing to 12/12/8 (quad/prnu/dsnu) */
        pd_coeff_width = 32;
    }
    else
    {
        if(!pconfig->PRNUDSNU_LUT_USE_16BIT_PRNU_DSNU) {
            pd_coeff_width = 24;
        }
        else {
            pd_coeff_width = 32;
        }
    }

    pic_pd_set_config1(pic_handle, pd_corr_bits, 0, 0, 0);
    // set coefficient width
    pic_pd_set_coeffwidth(pic_handle, pd_coeff_width);

    if(pconfig->MAX_DIGITAL_GAIN_TIMES_10!=50)
        return 7;

    if(pconfig->MAX_DIGITAL_CORRECTION==0xFFF)
    {
          /* 1010/2011 Eric: we Only support MAX_DIGITAL_GAIN=5.0 
       Here is how I get the multiple factor (checked with Paul), note we don't use 
       scale factor
       1) Start with a 12 bit fractional 0.12, which has a range of [0,0.9999..]
       2) Since we want to PRNU to have a max= 4.99999, which means we want
          to use the 0.12 to represent (max-1)=3.99999, that requires a multi factor
           f0=4
       3) Considering we want to use 0x8000 as offset, which means we want 0x8000 
          to be 1.0, that means we want a x.15 as a final PRNU value.

       4) To get 0.12 to x.15, we need another multi factor f1=8 (3 bit shift)
         
        5) Final multi factor is f=f0*f1=32
        */
        pic_pd_set_prnu_mult_scale(pic_handle, 32, 0, 32, 0, 32, 0);
        pic_pd_set_prnu_offset(pic_handle, 0x8000, 0x8000, 0x8000);
    }
    else if(pconfig->MAX_DIGITAL_CORRECTION == 0xFFFF)
    {
        /* For 16-bit PRNU, here is how the  factor 2 is calcualted:
          Starting from 0.16, we need f0=4, to convert a 0.16 to a 0.15, we need f1=1/2. Final f=f0*f1=2
        */
        pic_pd_set_prnu_mult_scale(pic_handle, 2, 0, 2, 0, 2, 0);
        pic_pd_set_prnu_offset(pic_handle, 0x8000,  0x8000,  0x8000);
    }else
    {
        return 8; //Error
    }

    pic_pd_set_dsnu_mult_scale(pic_handle,
                               1<<(pconfig->DSNU_SHIFT), 0,
                               1<<(pconfig->DSNU_SHIFT), 0,
                               1<<(pconfig->DSNU_SHIFT), 0);
    pic_pd_set_dsnu_offset(pic_handle, 0, 0, 0);

    return 0;
} 

/**\brief Routine to start a calibration process.
 * 
 * \param[in] pCCase Calibration case to be processed.
 * 
 * \author Eric Huang
 * \author Brad Simith
 * \author David Poole
 * 
 **/

static cal_err_t
processCalibrationCase(struct pic_handle_t **pic_handles, CalibrationCase *pCCase)
{
    scan_err_t scerr;
    cal_cache_t *cached_cal;
    cal_err_t calerr;
    uint32_t num32;
    struct pic_handle_t *pic_handle = pic_handles[pCCase->sensor_num];

    calerr = 0;

    cal_send_all_calcase_metadata( pCCase );

    cal_send_uint32_metadata( 2, 
            CAL_METADATA_BITS_PER_PIXEL, 16,
            CAL_METADATA_ROWS_PER_STRIP, get_needed_rows_per_buffer( pCCase->horiz_res) );

    // Disable motor motion for the first few steps.
    scan_enable_motor_motion(false);

    calerr = cal_setup( pic_handle, pCCase );
    if( calerr != 0 ) {
        scanlog(0,  "%s cal_setup failed, err=%d\n", __FUNCTION__ ,calerr);
        return calerr;
    }

    /* Eric H.: To avoid saturated sensor output show up in calibration, do a summy scan
                before calibrating.
    */
    calerr= cal_dummy_scan( pic_handles, pCCase );
    if( calerr != 0 ) {
        scanlog(0,  "%s cal_dummy_scan failed, err=%d\n", __FUNCTION__ ,calerr);
        return calerr;
    }

    calerr = cal_calc_analog_offset(pic_handles, pCCase);
    if( calerr != 0 ) {
        scanlog(0, "%s calc_analog_offset() failed with calerr=%d\n", 
                    __FUNCTION__, calerr );
        return calerr;
    }
 
    calerr = cal_calc_pwm_gain(pic_handles, pCCase);
    if( calerr != 0 ) {
        scanlog(0, "%s cal_calc_led_pwm() failed with calerr=%d\n", 
                    __FUNCTION__, calerr ); 
        return calerr;
    }
   
    // Enable motor motion for the remaining steps
    scan_enable_motor_motion(true);
    calerr = cal_calc_digital_correction(pic_handles, pCCase);
    if( calerr != 0 ) {
        scanlog(0, "%s cal_calc_digital_correction() failed with calerr=%d\n", 
                    __FUNCTION__, calerr ); 
        return calerr;
    }
    scan_enable_motor_motion(true);

    /* store everything (except LUTs!) to cache */
    cached_cal = getCalCache(pCCase);
    ASSERT(cached_cal);
    store_to_cal_cache( cached_cal, pCCase );

    scerr = scands_get_integer_with_default( "cal_debug_scans", &num32, 0 );
    XASSERT( scerr==SCANERR_NONE, scerr );
    
    /* To enable debug scan at end of cal, set the scan_defines to a nonzero value*/
    cal_run_debug_scans( pic_handles, pCCase, cached_cal, num32 );

    /* davep 18-Sep-2012 ; capture a bunch of scans  */
    if( num32 ) {
        cal_debug_scan_setup_pdlut( pic_handle, cached_cal, pCCase->cmode );
        cal_send_scan_id( CAL_CAP_DEBUG );
        calerr = cal_debug_scan( pic_handles, pCCase );

        cal_debug_scan_setup_pdlut( pic_handle, cached_cal, pCCase->cmode );
        cal_send_scan_id( CAL_CAP_DEBUG );
        calerr = cal_debug_scan( pic_handles, pCCase );

        cal_debug_scan_setup_pdlut( pic_handle, cached_cal, pCCase->cmode );
        cal_send_scan_id( CAL_CAP_DEBUG );
        calerr = cal_debug_scan( pic_handles, pCCase );

        cal_debug_scan_setup_pdlut( pic_handle, cached_cal, pCCase->cmode );
        cal_send_scan_id( CAL_CAP_DEBUG );
        calerr = cal_debug_scan( pic_handles, pCCase );
    }

    /* davep 17-Mar-2008 ; make sure motor enabled before we leave (?) */
    scan_enable_motor_motion(true);

    dbg1("calibration complete\n");

    return 0;
}

