/*
**************************************************************************
This Source Code Form is subject to the terms of the Mozilla Public
License, v. 2.0. If a copy of the MPL was not distributed with this file,
You can obtain one at http://mozilla.org/MPL/2.0/.

Copyright (c) 2011-2016, Marvell International Ltd.

Alternatively, this software may be distributed under the terms of the GNU
General Public License Version 2, and any use shall comply with the terms and
conditions of the GPL.  A copy of the GPL is available at
http://www.gnu.org/licenses/old-licenses/gpl-2.0.html

THE FILE IS DISTRIBUTED AS-IS, WITHOUT WARRANTY OF ANY KIND, AND THE
IMPLIED WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE
ARE EXPRESSLY DISCLAIMED.  The GPL license provides additional details about
this warranty disclaimer.
******************************************************************************
******************************************************************************
*
*  NOTE: Lots of debate over which allocation flags to use:
*
* - GFP_KERNEL: 
*     may cause cached items to be flushed if needed. This is a very good
*     thing if we are in very low memory, but the cache flush will introduce
*     a delay (which can stall our message processing loop).  This may also
*     cause a kernel nastygram if memory cannot be obtained. Experiments show
*     this doesn't always flush as expected ...
*
* - __GFP_NOWARN|__GFP_NORETRY:
*     will not cause cached items to be flushed, will not complain if memory
*     cannot be allocated.  This is great for our message processing loop, but
*     we can get us stuck in a memory out (if nobody else flushes the cache).
*
* If you change flags, make sure the math in scan_check_freemem is still 
* valid.
*
* TODO: pretty sure the dma pool spinlock implementation leaves something to be
*       desired.
*/


#include <stdint.h>

#include <linux/slab.h>
#include <linux/spinlock.h>

#include "icetest_if.h"

#include "scos.h"

#include "lassert.h"
#include "regAddrs.h"
#include "list.h"
#include "memAPI.h"
#include "cpu_api.h"

#include "scantypes.h"
#include "scancore.h"
#include "scandbg.h"
#include "scanmem.h"
#include "notaheap.h"

//#define SCANMEM_DEBUG

#ifdef SCANMEM_DEBUG
#define scanmem_dbg2 dbg2
#else
#define scanmem_dbg2(...)
#endif

// TODO: PORT - is there a better way of getting this info
#define LCM_SIZE 1024 * 96

/* davep 29-Mar-2013 ; lets try CISX in LCM */
//static volatile LCM_REGS_t * const lcm_regs = (volatile LCM_REGS_t *)LCM_BASE ;

/* davep 23-Apr-2013 ; cheesy memory manager to share LCM between CISX and
 * PRNU/DSNU
 */
static struct notaheap scan_fast_memory;

/* davep 14-May-2013 ; track my own memory usage. Only manages the DMA buffers
 * (via scanmem_alloc_aligned)
 *
 * Note signed integer to detect underflow;
 */
#define SCANMEM_MAX_MEMORY 384*1024*1024
static int max_memory = SCANMEM_MAX_MEMORY;

static spinlock_t scanmem_lock;

#if 1 /* RICOH original light weight drop cache */
extern int sgw_refrain_fcache(int);
#define REFRAIN_FCACHE_RATE 108 /* 8%(force) drop cache */
#endif

#define lock() spin_lock(&scanmem_lock)
#define unlock() spin_unlock(&scanmem_lock)


#if 1 /* RICOH */

#define KB (1024)
#define MB (1024 * 1024)
#define MAX_PREALLOC_1M_BUFFER_COUNT 15
#define MAX_PREALLOC_2M_BUFFER_COUNT 10

static void* prealloc_buffer1MB[MAX_PREALLOC_1M_BUFFER_COUNT];
static void* prealloc_buffer2MB[MAX_PREALLOC_2M_BUFFER_COUNT];

static bool push_prealloc_buffer(void** prealloc_buffer, int buf_len, void *data) {
	int i;
	bool result = false;

	lock();
	for (i = 0; i < buf_len; i++) {
		if (prealloc_buffer[i] == NULL) {
			prealloc_buffer[i] = data;
			dbg1( "%s[%d] 0x%p\n", __FUNCTION__, i, data);
			result = true;

			break;
		}
	}
	unlock();

	return result;
}

static void* pop_prealloc_buffer(void** prealloc_buffer, int buf_len) {
	int i;
	void *data = NULL;

	lock();
	for (i = 0; i < buf_len; i++) {
		if (prealloc_buffer[i] != NULL) {
			data = prealloc_buffer[i];
			dbg1( "%s[%d] 0x%p\n", __FUNCTION__, i, data);
			prealloc_buffer[i] = NULL;

			break;
		}
	}
	unlock();

	return data;
}

static void create_prealloc_buffer(void) {
	static bool is_first = true;
	int i;

	if (is_first) {
		is_first = false;

		lock();
		for (i = 0; i < MAX_PREALLOC_1M_BUFFER_COUNT; i++) {
			prealloc_buffer1MB[i] = kmalloc( 1 * MB, GFP_DMA|__GFP_NOWARN|__GFP_NORETRY );
			dbg1( "%s [%d] 1MB:0x%p \n", __FUNCTION__, i, prealloc_buffer1MB[i] );
		}

		for (i = 0; i < MAX_PREALLOC_2M_BUFFER_COUNT; i++) {
			prealloc_buffer2MB[i] = kmalloc( 2 * MB, GFP_DMA|__GFP_NOWARN|__GFP_NORETRY );
			dbg1( "%s [%d] 2MB:0x%p\n", __FUNCTION__, i, prealloc_buffer2MB[i] );
		}
		unlock();
	}
}

#endif /* RICOH */

void *scanmem_alloc( uint32_t datalen )
{
    void *ptr;
	int ret;

    /* davep 14-May-2013 ;  Note this function does not count against the
     * max_memory. Only the DMA buffers count. I can easily track the
     * size of an allocation of the DMA buffers but not the regular heap
     * memory. (I want to maintain the malloc/free semantics as much as
     * possible.)
     */

	ptr = kmalloc( datalen, __GFP_NOWARN|__GFP_NORETRY );

#if 1 /* RICOH : kmalloc retry */
	while (ptr == 0) {
		dbg1("%s sgw_refrain_fcache %d(size:%d)\n", __FUNCTION__, REFRAIN_FCACHE_RATE, datalen);
		ret = sgw_refrain_fcache(REFRAIN_FCACHE_RATE); /* light weight drop cache */

		/* retry kmalloc */
		ptr = kmalloc( datalen, __GFP_NOWARN|__GFP_NORETRY );
		dbg1("  sgw_refrain_fcache ret=%d, ptr=%p\n", ret, ptr);
	}
#endif /* RICOH : kmalloc retry */

    return ptr;
}

void *scanmem_alloc_aligned( uint32_t datalen )
{
    void *ptr;
    int free_bytes;
	int ret;
	
#if 1 /* RICOH */
	if (datalen > 512 * KB && datalen <= 1 * MB) {
		datalen = 1 * MB;
	} else if (datalen > 1 * MB && datalen <= 2 * MB) {
		datalen = 2 * MB;
	}
#endif

    /* davep 14-May-2013 ; adding self limiting memory */
    free_bytes = scanmem_get_free_size();
    if( free_bytes - (int)datalen < 0 ) {
        dbg2( "%s Fail: free = %d, requested = %d\n", __FUNCTION__, free_bytes, datalen );
        return NULL;
    }

    lock();
    max_memory -= datalen;
    unlock();

#if 1 /* RICOH */
	if (datalen > 512 * KB && datalen <= 1 * MB) {
		dbg1(" %s pop 1MB buffer\n", __FUNCTION__);
		ptr = pop_prealloc_buffer(prealloc_buffer1MB, MAX_PREALLOC_1M_BUFFER_COUNT);
	} else if (datalen > 1 * MB && datalen <= 2 * MB) {
		dbg1(" %s pop 2MB buffer\n", __FUNCTION__);
		ptr = pop_prealloc_buffer(prealloc_buffer2MB, MAX_PREALLOC_2M_BUFFER_COUNT);
	} else {
		ptr = NULL;
	}

	if (ptr) {
		return ptr;
	}

	if (datalen > 512 * KB) {
		dbg1("WARNING: %s alloc lagrge buffer:%d\n", __FUNCTION__, datalen);
	}
#endif

	ptr = kmalloc( datalen, GFP_DMA|__GFP_NOWARN|__GFP_NORETRY );

#if 1 /* RICOH : kmalloc retry */
	while (ptr == 0) {
		dbg1("%s sgw_refrain_fcache %d(size:%d)\n", __FUNCTION__, REFRAIN_FCACHE_RATE, datalen);
		ret = sgw_refrain_fcache(REFRAIN_FCACHE_RATE); /* light weight drop cache */

		/* retry kmalloc */
		ptr = kmalloc( datalen, GFP_DMA|__GFP_NOWARN|__GFP_NORETRY );
		dbg1("  sgw_refrain_fcache ret=%d, ptr=%p\n", ret, ptr);
	}
#endif /* RICOH : kmalloc retry */

    /* if we actually we're able to get it, return the value to our max (our
     * allocation limit doesn't reflect the actual state of the heap)
     */
    if( ptr==NULL ) {
        lock();
        max_memory += datalen;
        unlock();
    }

//    dbg2( "%s %p %d %d\n", __FUNCTION__, ptr, datalen, scanmem_get_free_size());

    return ptr;
}

void scanmem_free_aligned( uint8_t **p_ptr, uint32_t datalen )
{
    uint32_t s;

//    dbg2( "%s %p %d\n", __FUNCTION__, *p_ptr, datalen );

#if 1 /* RICOH */
	if (datalen > 512 * KB && datalen <= 1 * MB) {
		datalen = 1 * MB;
		dbg1(" %s push 1MB buffer\n", __FUNCTION__);
		if (push_prealloc_buffer(prealloc_buffer1MB, MAX_PREALLOC_1M_BUFFER_COUNT, *p_ptr)) {
			*p_ptr = NULL;
		}
	} else if (datalen > 1 * MB && datalen <= 2 * MB) {
		datalen = 2 * MB;
		dbg1(" %s push 2MB buffer\n", __FUNCTION__);
		if (push_prealloc_buffer(prealloc_buffer2MB, MAX_PREALLOC_2M_BUFFER_COUNT, *p_ptr)) {
			*p_ptr = NULL;
		}
	}

	if (*p_ptr) {
	    kfree( *p_ptr );
	    *p_ptr = NULL;
	}
#else /* RICOH */
    kfree( *p_ptr );
    *p_ptr = NULL;
#endif /* RICOH */

    lock();
    max_memory += datalen;
    s = (uint32_t)max_memory;
    unlock();

    /* davep 14-May-2013 ; beware of fake free'ing */
    XASSERT( s <= SCANMEM_MAX_MEMORY, s );
}

uint32_t scanmem_get_heap_size( void )
{
    return SCANMEM_MAX_MEMORY;
}

uint32_t scanmem_get_free_size( void )
{
    uint32_t s;

    lock();
    s = (uint32_t)max_memory;
    unlock();

    return s;
}

void scanmem_free_fast_memory( scanmem_tag tag, void *ptr, uint32_t datalen, int instance )
{
    scanmem_dbg2( "%s tag=%d ptr=%p datalen=%d instance=%d\n", __FUNCTION__,
                        tag, ptr, datalen, instance );

    notaheap_free( &scan_fast_memory, tag, ptr, datalen, instance );
}

/**
 * \brief  Get a pointer into fast memory, if available.
 *
 * Get a DMA'able pointer from some sort of faster-than-DRAM memory such as
 * SRAM. Some platforms may not have this sort of memory.
 *
 * \author David Poole
 * \date 23-Apr-2013
 */

void * scanmem_get_fast_memory( scanmem_tag tag, uint32_t datalen, int instance )
{
    scanmem_dbg2( "%s tag=%d datalen=%d instance=%d\n", __FUNCTION__, 
                    tag, datalen, instance);

    return notaheap_alloc( &scan_fast_memory, tag, datalen, instance );
}

/**
 * \brief  Initialize the scan memory portability layer subsystem
 *
 * \author David Poole
 * \date 23-Apr-2013
 */

scan_err_t scanmem_onetime_init( void )
{
    scan_err_t scerr;

    request_mem_region((uint32_t)IPS_LCM_BASE, LCM_SIZE, "scan_fastmem");

    dbg2( "%s fastmem=%p\n", __FUNCTION__, IPS_LCM_BASE );

    spin_lock_init(&scanmem_lock);

#if 1 /* RICOH */
	dbg1("%s called, and call create_prealloc_buffer\n", __FUNCTION__);
	create_prealloc_buffer();
#endif /* RICOH */

    /* initialize the fast memory sharing subsystem */
    // ToDo - we need to put this info into the DTSI file
    if (icetest_get_rev0() == 3)    // RevA is 3
    {
        dbg2( "Looks like a REV_A asic\n" );
        scerr = notaheap_init( &scan_fast_memory, (uint8_t *)IPS_LCM_BASE, LCM_SIZE );
    }
    else
    {
        dbg2( "Looks like a REV_B asic\n" );
        #define IPS_LCM_BASE_REVB ((IPS_BASE) + 0xD0FE8000)
        scerr = notaheap_init( &scan_fast_memory, (uint8_t *)IPS_LCM_BASE_REVB, LCM_SIZE );
    }
    if( scerr != SCANERR_NONE ) {
        return scerr;
    }

    return SCANERR_NONE;
}

