/*
**************************************************************************
This Source Code Form is subject to the terms of the Mozilla Public
License, v. 2.0. If a copy of the MPL was not distributed with this file,
You can obtain one at http://mozilla.org/MPL/2.0/.

Copyright (c) 2014-2015, Marvell International Ltd.

Alternatively, this software may be distributed under the terms of the GNU
General Public License Version 2, and any use shall comply with the terms and
conditions of the GPL.  A copy of the GPL is available at
http://www.gnu.org/licenses/old-licenses/gpl-2.0.html

THE FILE IS DISTRIBUTED AS-IS, WITHOUT WARRANTY OF ANY KIND, AND THE
IMPLIED WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE
ARE EXPRESSLY DISCLAIMED.  The GPL license provides additional details about
this warranty disclaimer.
******************************************************************************
*/
#include <linux/module.h>
#include <linux/irqreturn.h>
#include <linux/interrupt.h>
#include <linux/slab.h>
#include <linux/of_device.h>
#include <linux/dma-mapping.h>
#include <linux/of_irq.h>
#include <linux/of_address.h>
#include <linux/cdev.h>
#include <linux/proc_fs.h> //proc stuff
#include <linux/uaccess.h>
#include "pic_data.h"
#include "pic_handle_if.h"
#include "pic_if.h"
#include "pic_convenience_if.h"
#include "pic_driverlib_if.h"

#include "scantypes.h"
#include "scanmem.h"

struct pd_dev
{
    struct cdev cdev; /*Char device structure*/
    int pic_instance;
    size_t lut_size;
    dma_addr_t lut_phys_addr;
    uint8_t *lut_virt_addr;
};

#define MAX_PIC_INSTANCES 2

static struct pd_dev pd_cdevs[MAX_PIC_INSTANCES];
static dev_t dev_no;

static int get_pic_instance(struct device *dev)
{
    return (int)(dev_get_drvdata(dev));
}

static ssize_t pd_bypass_store(struct device *dev,
                               struct device_attribute *attr,
                               const char *buf, size_t count)
{
    struct pic_handle_t *pic_handle;
    int pic_instance = get_pic_instance(dev);
    bool val = 0;
    if (buf[0] == '0')
    {
        val = false;
    } else
    {
        val = true;
    }
    
    pic_handle = pic_create_new_default_handle();
    pic_do_get_current(pic_handle, pic_instance);
    
    if (!strncmp(attr->attr.name, "bypass_all", sizeof("bypass_all")))
    {
        print("bypass_all\n");
        pic_pd_set_bypass_all(pic_handle, val);
    } else if (!strncmp(attr->attr.name, "bypass_prnu", sizeof("bypass_prnu")))
    {
        print("bypass_prnu\n");
        pic_pd_set_bypass_prnu(pic_handle, val);
    } else if (!strncmp(attr->attr.name, "bypass_dsnu", sizeof("bypass_dsnu")))
    {
        print("bypass_dsnu\n");
        pic_pd_set_bypass_dsnu(pic_handle, val);
    } else if (!strncmp(attr->attr.name, "bypass_bad_replace", sizeof("bypass_bad_replace")))
    {
        print("bypass_bad_replace\n");
        pic_pd_set_bypass_bad_pix_replace(pic_handle, val);
    } else if (!strncmp(attr->attr.name, "bypass_exp_comp", sizeof("bypass_exp_comp")))
    {
        print("bypass_exp_comp\n");
        pic_pd_set_bypass_exp_comp(pic_handle, val);
    } else if (!strncmp(attr->attr.name, "bypass_quad", sizeof("bypass_quad")))
    {
        print("bypass_quad\n");
        pic_pd_set_bypass_quadratic(pic_handle, val);
    }
    else
    {
        print("Not supported attribute name=%s\n", attr->attr.name);
    }
    
    pic_do_configure(pic_handle, pic_instance);
    
    pic_do_free_handle(pic_handle);
    
    return count;
}

static ssize_t pd_bypass_show(struct device *dev,
                              struct device_attribute *attr,
                              char *buf)
{
    struct pic_handle_t *pic_handle;
    int pic_instance = get_pic_instance(dev);
    bool val = false;
    
    buf[0] = 0;
    
    pic_handle = pic_create_new_default_handle();
    pic_do_get_current(pic_handle, pic_instance);
    
    if (!strncmp(attr->attr.name, "bypass_all", sizeof("bypass_all")))
    {
        //print("bypass_all\n");
        val = pic_pd_get_bypass_all(pic_handle);
    } else if (!strncmp(attr->attr.name, "bypass_prnu", sizeof("bypass_prnu")))
    {
        // print("bypass_prnu\n");
        val = pic_pd_get_bypass_prnu(pic_handle);
    } else if (!strncmp(attr->attr.name, "bypass_dsnu", sizeof("bypass_dsnu")))
    {
    	//print("bypass_dsnu\n");
    	val = pic_pd_get_bypass_dsnu(pic_handle);
    } else if (!strncmp(attr->attr.name, "bypass_bad_replace", sizeof("bypass_bad_replace")))
    {
    	//print("bypass_bad_replace\n");
    	val = pic_pd_get_bypass_bad_pix_replace(pic_handle);
    } else if (!strncmp(attr->attr.name, "bypass_exp_comp", sizeof("bypass_exp_comp")))
    {
    	//print("bypass_exp_comp\n");
    	val = pic_pd_get_bypass_exp_comp(pic_handle);
    } else if (!strncmp(attr->attr.name, "bypass_quad", sizeof("bypass_quad")))
    {
    	//print("bypass_quad\n");
    	val = pic_pd_get_bypass_quadratic(pic_handle);
    } else
    {
        print("Not supported attribute name=%s\n", attr->attr.name);
    }
    
    if (val)
    {
        buf[0]='1';
    }else{
        buf[0]='0';
    }
    
    buf[1]='\0';
    pic_do_free_handle(pic_handle);
    return 2;
}



static int pd_dsnu_show(struct device *dev,
                        struct device_attribute *attr,
                        char  *buf)
{
    
    struct pic_handle_t *pic_handle;
    int pic_instance = get_pic_instance(dev);
    uint16_t offset, mult;
    uint8_t scale;
    
    pic_handle = pic_create_new_default_handle();
    pic_do_get_current(pic_handle, pic_instance);
    
    pic_pd_get_dsnu_offset_c0(pic_handle, &offset);
    pic_pd_get_dsnu_mult_scale_c0(pic_handle, &scale, &mult);
    
    pic_do_free_handle(pic_handle);
    
    sprintf(buf, "0x%x 0x%x 0x%x\n", offset, mult, scale);
    return strlen(buf)+1;
}

static int pd_dsnu_store(struct device *dev,
                         struct device_attribute *attr,
                         const char *buf, size_t count)
{
    int pic_instance = get_pic_instance(dev);
    struct pic_handle_t *pic_handle;
    int offset,mult, scale;
    
    sscanf(buf, "%x %x %x", &offset, &mult , &scale);
    
    pic_handle = pic_create_new_default_handle();
    pic_do_get_current(pic_handle, pic_instance);
    
    print("set dsnu offset=%x, mult=%x scale=%x\n", offset, mult, scale);
    pic_pd_set_dsnu_offset(pic_handle, offset, offset, offset);
    pic_pd_set_dsnu_mult_scale(pic_handle, mult, scale, mult, scale, mult, scale);
    
    pic_do_configure(pic_handle, pic_instance);
    
    pic_do_free_handle(pic_handle);
    
    return count;
}


static int pd_prnu_show(struct device *dev,
                        struct device_attribute *attr,
                        char  *buf)
{
    struct pic_handle_t *pic_handle;
    int pic_instance = get_pic_instance(dev);
    uint16_t offset, mult;
    uint8_t scale;
    
    pic_handle = pic_create_new_default_handle();
    pic_do_get_current(pic_handle, pic_instance);
    
    pic_pd_get_prnu_offset_c0(pic_handle, &offset);
    pic_pd_get_prnu_mult_scale_c0(pic_handle, &scale, &mult);
    
    pic_do_free_handle(pic_handle);
    
    sprintf(buf, "0x%x 0x%x 0x%x\n", offset, mult, scale);
    return strlen(buf)+1;
}

static int pd_prnu_store(struct device *dev,
                         struct device_attribute *attr,
                         const char *buf, size_t count)
{
    struct pic_handle_t *pic_handle;
    int pic_instance = get_pic_instance(dev);
    int offset,mult, scale;
    
    sscanf(buf, "%x %x %x" , &offset, &mult , &scale);
    
    pic_handle = pic_create_new_default_handle();
    pic_do_get_current(pic_handle, pic_instance);
    
    //print("set prnu offset=%x, mult=%x scale=%x\n", offset, mult, scale);
    pic_pd_set_prnu_offset(pic_handle, offset, offset, offset);
    pic_pd_set_prnu_mult_scale(pic_handle, mult, scale, mult, scale, mult, scale);
    
    pic_do_configure(pic_handle, pic_instance);
    pic_do_free_handle(pic_handle);
    
    return count;
}

static int pd_lut_show(struct device *dev,
                       struct device_attribute *attr,
                       char  *buf)
{
    struct pic_handle_t *pic_handle;
    int pic_instance = get_pic_instance(dev);
    uint8_t corrbits, allocc;
    int coff_size, dsnu, prnu;
    int entry_size;
    
    pic_handle = pic_create_new_default_handle();
    pic_do_get_current(pic_handle, pic_instance);
    
    entry_size = pic_pd_get_coeffwidth(pic_handle);
    
    pic_pd_get_cfg1(pic_handle, &corrbits, &allocc);
    /*Only show the bits width and color0, since we always use same setup for all
     * 3 colors
     */
    coff_size = 12 + 4* (corrbits);
    switch(allocc)
    {
    case 0:
        prnu = coff_size/2;
        break;
    case 1:
        prnu = coff_size/2 +1;
        break;
    case 2:
        prnu = coff_size/2 +2;
        break;
    case 3:
        prnu = coff_size/2 -1;
        break;
    case 4:
        prnu = coff_size/2 -2;
        break;
    case 5:
        prnu = coff_size/2 + 3;
        break;
    case 6:
        prnu = coff_size/2 -3;
        break;
    default:
        print("Invalid alloc0 read out %d\n", allocc);
        return -EFAULT;
        break;
    }
    dsnu = coff_size - prnu;
    
    sprintf(buf, "%d %d %d %d\n",
            pd_cdevs[pic_instance].lut_size*8/entry_size,
            entry_size, prnu, dsnu);
    
    pic_do_free_handle(pic_handle);
    
    return strlen(buf)+1;
}


static ssize_t pd_lut_store(struct device *dev,
                            struct device_attribute *attr,
                            const char *buf, size_t count)
{
    struct pic_handle_t *pic_handle;
    int pic_instance = get_pic_instance(dev);
    size_t i, line_width;
    dma_addr_t phys_add;
    uint8_t    *virt_add;
    int entry_size;
    int prnu, dsnu;
    int alloc;
    size_t lut_size;
    
    sscanf(buf, "%d %d %d %d", &line_width, &entry_size , &prnu, &dsnu);
    
    pic_handle = pic_create_new_default_handle();
    pic_do_get_current(pic_handle, pic_instance);
    
    pic_pd_set_coeffwidth(pic_handle, entry_size);
    
    if ( (prnu + dsnu) != 12 &&
         (prnu + dsnu) != 16 &&
         (prnu + dsnu) != 20 &&
         (prnu + dsnu) != 24 &&
         (prnu + dsnu) != 28 &&
         (prnu + dsnu) != 32)
    {
        print("Invalid corrbits size %d\n", prnu + dsnu);
        return -EFAULT;
    }
    alloc = (prnu - dsnu)/2;
    
    if (alloc < -3 || alloc > 3)
    {
        print("Invalid allocation %d\n", alloc);
        return -EFAULT;
    }
    /*Use same alloc for all colors */
    pic_pd_set_config1(pic_handle, prnu + dsnu, alloc, alloc, alloc);
    
    pic_do_configure(pic_handle, pic_instance);
    pic_do_free_handle(pic_handle);
    
    lut_size = line_width *entry_size/8;
    print("To set lut_size to %d\n", lut_size);
    
    if (lut_size != pd_cdevs[pic_instance].lut_size)
    {
        if (pd_cdevs[pic_instance].lut_size!=0)
        {
            iounmap((void *)pd_cdevs[pic_instance].lut_virt_addr);
            scanmem_free_fast_memory( SCANMEM_TAG_PDLUT,
                                      (void *)pd_cdevs[pic_instance].lut_phys_addr,
                                      pd_cdevs[pic_instance].lut_size, pic_instance );
//          dma_free_coherent(NULL,
//          pd_cdevs[pic_instance].lut_size,
//          pd_cdevs[pic_instance].lut_virt_addr,
//          pd_cdevs[pic_instance].lut_phys_addr);
        }
        
        pd_cdevs[pic_instance].lut_size = lut_size;
        
        if (lut_size > 0)
        {
            phys_add = (dma_addr_t)scanmem_get_fast_memory(SCANMEM_TAG_PDLUT,
                                                           lut_size, pic_instance);
            virt_add = ioremap_nocache( phys_add, lut_size );
            
//          virt_add = dma_alloc_coherent(NULL, lut_size,  &phys_add, GFP_DMA);
            if (!virt_add || !phys_add)
            {
                print("fail to allocate dma memory size %d\n", lut_size);
                return 0;
            }
            pd_cdevs[pic_instance].lut_phys_addr = phys_add;
            pd_cdevs[pic_instance].lut_virt_addr = virt_add;
            
            //clean it up to known values
            for (i=0;i<lut_size; i++)
            {
                virt_add[i] = 0;
            }
        }
    }
    return count;
}


static int pd_bsc_show(struct device *dev,
                       struct device_attribute *attr,
                       char  *buf)
{
    struct pic_handle_t *pic_handle;
    int pic_instance = get_pic_instance(dev);
    uint8_t leftgainintc, leftgaindecc, rightgainintc, rightgaindecc, lslope, rslope;
    uint16_t lpmax, rpstart;
    
    pic_handle = pic_create_new_default_handle();
    pic_do_get_current(pic_handle, pic_instance);
    
    pic_pd_get_gain(pic_handle, &leftgainintc, &leftgaindecc, &rightgainintc, &rightgaindecc);
    
    pic_pd_get_left_slope(pic_handle, &lslope);
    
    pic_pd_get_left_max_c0(pic_handle, &lpmax);
    
    pic_pd_get_right_slope(pic_handle, &rslope);
    
    pic_pd_get_right_start_c0(pic_handle, &rpstart);
    
    pic_do_free_handle(pic_handle);
    
    /*
     * lgMaxC slope LpmaxC : RgminC slope Rpstart
     */
    
    sprintf(buf, "%d.%d %x %d : %d.%d %x %d \n",
            leftgainintc, leftgaindecc, lslope, lpmax,
            rightgainintc, rightgaindecc, rslope, rpstart);
    
    return strlen(buf)+1;
}


static ssize_t pd_bsc_store(struct device *dev,
                            struct device_attribute *attr,
                            const char *buf, size_t count)
{
    struct pic_handle_t *pic_handle;
    int i;
    int pic_instance = get_pic_instance(dev);
    int lgmaxcint, lgmaxcdec, lslopec, rgmincint, rgmincdec, rslopec;
    uint8_t LgmaxC_int[3], LgmaxC_dec[3], LslopeC[3];
    uint8_t RgminC_int[3], RgminC_dec[3], RslopeC[3];
    int LpmaxC, Rpstart;
    
    pic_handle = pic_create_new_default_handle();
    
    sscanf(buf, "%d.%d %x %d: %d.%d %d %d",
           &lgmaxcint, &lgmaxcdec, &lslopec, &LpmaxC,
           &rgmincint, &rgmincdec, &rslopec, &Rpstart);
    
    for (i=0; i<3; i++) {
        LgmaxC_int[i] = lgmaxcint;
        LgmaxC_dec[i] = lgmaxcdec;
        RgminC_int[i] = rgmincint;
        RgminC_dec[i] = rgmincdec;
        LslopeC[i] = lslopec;
        RslopeC[i] = rslopec;
    }
    
    pic_pd_set_gain(pic_handle, LgmaxC_int, LgmaxC_dec, RgminC_int, RgminC_dec);
    
    pic_pd_set_left_slope(pic_handle, LslopeC);
    pic_pd_set_right_slope(pic_handle, RslopeC);
    
    pic_pd_set_left_max_c0(pic_handle, LpmaxC);
    pic_pd_set_left_max_c1(pic_handle, LpmaxC);
    pic_pd_set_left_max_c2(pic_handle, LpmaxC);
    
    Rpstart = Rpstart & 0xFFFF;
    pic_pd_set_right_start_c0(pic_handle, Rpstart);
    pic_pd_set_right_start_c1(pic_handle, Rpstart);
    pic_pd_set_right_start_c2(pic_handle, Rpstart);
    
    pic_do_configure(pic_handle, pic_instance);
    
    pic_do_free_handle(pic_handle);
    
    return count;
}



static int pd_debug_store(struct device *dev,
                          struct device_attribute *attr,
                          const char *buf, size_t count)
{
    int pic_instance = get_pic_instance(dev);
    struct pic_handle_t *pic_handle;
    int cmd;
    int i;
    
    pic_handle = pic_create_new_default_handle();
    pic_do_get_current(pic_handle, pic_instance);
    
    sscanf(buf, "%d", &cmd);
    switch(cmd)
    {
    case 0:
        //print out
        if (pd_cdevs[pic_instance].lut_size > 0)
        {
            print("Dump LUT. Total ENtries %d, coeffwidth=%d \n",
                  pd_cdevs[pic_instance].lut_size,
                  pic_pd_get_coeffwidth(pic_handle));
            
            for (i = 0;i< pd_cdevs[pic_instance].lut_size; i++)
            {
                if (i%32 == 0)
                    print("\n[%d] ", i);
                print("%02x ", pd_cdevs[pic_instance].lut_virt_addr[i]);
            }
            print("\n");
        } else
        {
            print("LUT is not filled yet!\n");
        }
        break;
    default:
        break;
    }
    
    pic_do_free_handle(pic_handle);
    
    return count;
    
}

DEVICE_ATTR(pd_debug,       S_IWUSR , NULL,  pd_debug_store);
DEVICE_ATTR(bypass_all,    S_IWUSR | S_IRUGO, pd_bypass_show, pd_bypass_store);
DEVICE_ATTR(bypass_prnu,   S_IWUSR | S_IRUGO, pd_bypass_show, pd_bypass_store);
DEVICE_ATTR(bypass_dsnu,   S_IWUSR | S_IRUGO, pd_bypass_show, pd_bypass_store);
DEVICE_ATTR(bypass_bad_replace, S_IWUSR | S_IRUGO, pd_bypass_show, pd_bypass_store);
DEVICE_ATTR(bypass_exp_comp,  S_IWUSR | S_IRUGO, pd_bypass_show, pd_bypass_store);
DEVICE_ATTR(bypass_quad,   S_IWUSR | S_IRUGO, pd_bypass_show, pd_bypass_store);
DEVICE_ATTR(dsnu,   S_IWUSR | S_IRUGO, pd_dsnu_show, pd_dsnu_store);
DEVICE_ATTR(prnu,   S_IWUSR | S_IRUGO, pd_prnu_show, pd_prnu_store);
DEVICE_ATTR(lut,   S_IWUSR | S_IRUGO, pd_lut_show, pd_lut_store);
DEVICE_ATTR(bsc,   S_IWUSR | S_IRUGO, pd_bsc_show, pd_bsc_store);

static struct attribute *pd_attrs[] = {
    &dev_attr_pd_debug.attr,
    &dev_attr_bypass_all.attr,
    &dev_attr_bypass_prnu.attr,
    &dev_attr_bypass_dsnu.attr,
    &dev_attr_bypass_bad_replace.attr,
    &dev_attr_bypass_exp_comp.attr,
    &dev_attr_bypass_quad.attr,
    &dev_attr_dsnu.attr,
    &dev_attr_prnu.attr,
    &dev_attr_lut.attr,
    &dev_attr_bsc.attr,
    NULL,
};

static struct attribute_group pd_attrgrp = {
    .attrs = pd_attrs
};

static const struct attribute_group *pd_attrgrp_set[] = {
    &pd_attrgrp,
    NULL
};

static struct class pic_ctrl_class = {
    .name = "pic_ctrl",
    .owner = THIS_MODULE,
    .dev_groups = pd_attrgrp_set,
};

static ssize_t pd_read(struct file *f, char __user *to,
                       size_t count, loff_t *offp)
{
    struct pd_dev *dev= f->private_data;
    ssize_t retval = 0;
    
    if ( dev->lut_size > 0)
    {
        uint8_t *kernel_temp;
        
        if (dev->lut_size <= *offp)
        {
            goto out;
        }
        if (*offp + count > dev->lut_size)
            count = dev->lut_size - *offp;
        
        kernel_temp = kmalloc( count, GFP_KERNEL );
        if ( !kernel_temp )
        {
            retval= -ENOMEM;
            goto out;
        }
        
        memcpy_fromio(kernel_temp, dev->lut_virt_addr + (*offp), count);
        if (copy_to_user(to, kernel_temp, count))
        {
            retval= -EFAULT;
            goto out;
        }
        *offp =(*offp) + count;
        retval = count;
        
        kfree( kernel_temp );
    }
out:
    return retval;
}

static ssize_t pd_write(struct file *f, const char __user* from,
                        size_t count, loff_t *offp)
{
    struct pd_dev *dev= f->private_data;
    ssize_t retval = 0;
    
    if ( dev->lut_size > 0)
    {
        uint8_t *kernel_temp;
        
        if (dev->lut_size <= *offp)
        {
            goto out;
        }
        if ((*offp) + count > dev->lut_size)
            count = dev->lut_size - *offp;
        
        kernel_temp = kmalloc(count, GFP_KERNEL);
        if (!kernel_temp)
        {
            retval= -ENOMEM;
            goto out;
        }
        
        //print("%s.%d: count=%d\n", __func__, __LINE__, count);
        if (copy_from_user(kernel_temp, from, count))
        {
            retval= -EFAULT;
            goto out;
        }
        memcpy_toio(dev->lut_virt_addr + (*offp), kernel_temp, count);
        *offp =(*offp) + count;
        retval = count;
        
        kfree( kernel_temp );
    } else
    {
        print("lut_size is not set yet!");
        retval= -EFAULT;
        goto out;
    }
out:
    return retval;
}

static int pd_open(struct inode *node, struct file *filp)
{
    struct pd_dev *dev; //device information
    
    print(KERN_DEBUG "pd_open\n");
    dev = container_of(node->i_cdev, struct pd_dev, cdev);
    filp->private_data = dev; /*for other methods*/
    return 0;
}

struct file_operations pd_ops = {
    .owner = THIS_MODULE,
    .read = pd_read,
    .write = pd_write,
    .open = pd_open,
    //.release = pd_release,
    //.unlocked_ioctl= fishing_ioctl,
};

size_t pic_pd_get_lut(int pic_instance, dma_addr_t *lut)
{
    if (pd_cdevs[pic_instance].lut_size > 0)
    {
        *lut = pd_cdevs[pic_instance].lut_phys_addr;
        return pd_cdevs[pic_instance].lut_size;
    }
    return 0;
}
EXPORT_SYMBOL(pic_pd_get_lut);


static void pic_control_add(int index)
{
    int err;
    struct device *dev;
    struct pic_handle_t *pic_handle;
    
    pic_handle = pic_create_new_default_handle();
    pic_do_get_current(pic_handle, index);
    pic_pd_set_bypass_all(pic_handle, true);
    pic_do_configure(pic_handle, index);
    pic_do_free_handle(pic_handle);
    
    memset(&pd_cdevs[index], 0, sizeof(struct pd_dev));
    
    dev = device_create(&pic_ctrl_class, NULL,
                        MKDEV(MAJOR(dev_no), index), (void *)index,
                        "pic_ctrl%u", index);
    
    cdev_init(&(pd_cdevs[index].cdev), &pd_ops);
    
    pd_cdevs[index].cdev.owner = THIS_MODULE;
    
    err = cdev_add(&(pd_cdevs[index].cdev), MKDEV(MAJOR(dev_no), index), 1);
    
    if (err)
        print(KERN_ALERT "Error %d adding pd %d\n", err, index);
    else
        print(KERN_ALERT "PIC_LUT[%d] dev_no is %d:%d\n", index, MAJOR(dev_no), MINOR(dev_no));
    
    pd_cdevs[index].pic_instance = index;
    
    //print("dev instance add =%x  dev add=%x\n", (uint32_t)&(pd_cdevs[index]), (uint32_t)dev);
    
}

void pic_control_remove(int pic_instance)
{
    struct pic_handle_t *pic_handle;
    
    pic_handle = pic_create_new_default_handle();
    pic_do_get_current(pic_handle, pic_instance);
    pic_pd_set_bypass_all(pic_handle, true);
    pic_do_configure(pic_handle, pic_instance);
    pic_do_free_handle(pic_handle);
    
    sysfs_remove_group(&(pd_cdevs[pic_instance].cdev.kobj), &pd_attrgrp);
    
    if (pd_cdevs[pic_instance].lut_size != 0)
    {
        iounmap((void *)pd_cdevs[pic_instance].lut_phys_addr);
        scanmem_free_fast_memory( SCANMEM_TAG_PDLUT,
                                  (void *)pd_cdevs[pic_instance].lut_phys_addr,
                                  pd_cdevs[pic_instance].lut_size, pic_instance );
//      dma_free_coherent(NULL, pd_cdevs[pic_instance].lut_size,
//      pd_cdevs[pic_instance].lut_virt_addr,
//      pd_cdevs[pic_instance].lut_phys_addr);
        
        pd_cdevs[pic_instance].lut_size = 0;
    }
    
    cdev_del(&(pd_cdevs[pic_instance].cdev));
    //unregister_chrdev_region(&(f_cdev.cdev), 1);
}

static int pic_control_init(void)
{
    int err;
    
    class_register(&pic_ctrl_class);
    
    err = alloc_chrdev_region(&dev_no, 0, MAX_PIC_INSTANCES, "pic_ctrl");
    if (err)
    {
        print(KERN_ALERT "Error %d allocate major number\n", err);
        return 1;
    }
    
    pic_control_add(0);
    pic_control_add(1);
    
    pic_set_external_pd_lut_provider(pic_pd_get_lut);
    
    /*
     * A non 0 return means init_module failed; module can't be loaded.
     */
    return 0;
}

static void pic_control_exit(void)
{
    print("%s\n", __FUNCTION__ );
    pic_control_remove(0);
    pic_control_remove(1);
    
    pic_set_external_pd_lut_provider( NULL );
    
    class_unregister(&pic_ctrl_class);
}

module_init(pic_control_init);
module_exit(pic_control_exit);


MODULE_AUTHOR("Copyright (c) 2014-2015 Marvell International Ltd.");
MODULE_DESCRIPTION("Marvell series Scan Driver");

MODULE_LICENSE("Dual MPL/GPL");
MODULE_VERSION("2014_JULY 18");

