/*
**************************************************************************
This Source Code Form is subject to the terms of the Mozilla Public
License, v. 2.0. If a copy of the MPL was not distributed with this file,
You can obtain one at http://mozilla.org/MPL/2.0/.

Copyright (c) 2010-2015, Marvell International Ltd.

Alternatively, this software may be distributed under the terms of the GNU
General Public License Version 2, and any use shall comply with the terms and
conditions of the GPL.  A copy of the GPL is available at
http://www.gnu.org/licenses/old-licenses/gpl-2.0.html

THE FILE IS DISTRIBUTED AS-IS, WITHOUT WARRANTY OF ANY KIND, AND THE
IMPLIED WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE
ARE EXPRESSLY DISCLAIMED.  The GPL license provides additional details about
this warranty disclaimer.
******************************************************************************
*/


/**
 * \file icefileapp.c
 *
 * \brief Thread and OBA interface to icefile
 *
 */

/* davep 23-Jul-2008 ; added -D__AGMESSAGE_H_ to makefile to prevent agMessage.h
 * from being included in the scan code.  But we need agMessage.h in this file.
 */
#ifdef __AGMESSAGE_H__
#undef __AGMESSAGE_H__
#endif

#include <string.h>
#include <stdint.h>
#include <stdbool.h>

//#include "tx_api.h"
#include "scos.h"

//#include "ATypes.h"
#include "lassert.h"
#include "debug.h"
#include "list.h"
#include "cpu_api.h"
#include "dprintf.h"

#include "scantypes.h"
#include "scancore.h"
#include "scandbg.h"
#include "safetylock.h"
#include "icedma.h"

#include "pic_handle_if.h"
#include "pic_if.h"

#include "pic.h"
#include "scansen.h"
#include "scanvars.h"
#include "icefile.h"
#include "icefileapp.h"
#include "scanlib.h"
//#include "txtools.h"
#include "safetylock.h"
#include "ddma.h"

#include "icetest_if.h"

#include "icetest.h"
#include "scantask.h"
#include "icetesttest.h"
#include "cisx.h"

#include "scanmech.h"

#include <linux/uaccess.h>
#include <linux/delay.h>

//#define ICEFILE_DEBUG 1

#ifdef ICEFILE_DEBUG
    #define ice_dbg2 dbg2
#else
    #define ice_dbg2(...)
#endif

#define MAX_ROWS_PER_ICEFILE_BUFFER  16
//#define MAX_ROWS_PER_ICEFILE_BUFFER  128
//#define MAX_ROWS_PER_ICEFILE_BUFFER  24

#define ICEFILE_CLOSED 0
#define ICEFILE_OPEN   1
static int icefile_state;

static struct ddma_descriptor *icetest_desc;
static struct ice_dma_buffer *icetest_icebuf;
static struct icefile_image_info image_info;
static bool use_pipecutter = false;

int icefile_get_rows_per_buffer( void )
{
    return MAX_ROWS_PER_ICEFILE_BUFFER;
}
EXPORT_SYMBOL(icefile_get_rows_per_buffer);

void icetest_free_memory( void )
{
    if( icetest_desc ) {
        ddma_descriptor_free( &icetest_desc );
    }
    if( icetest_icebuf ) {
        icebuf_free( &icetest_icebuf );
    }
}

static scan_err_t icetest_alloc_memory( int total_rows, int pixels_per_row )
{
    scan_err_t scerr, final_scerr;
    int icetest_bytes_per_row, icetest_pixels_per_row;
    int rows_per_buffer;
    uint32_t total_bytes;

    dbg2( "%s ppr=%d total_rows=%d\n", __FUNCTION__, pixels_per_row, total_rows);

    final_scerr = SCANERR_NONE;
    icetest_desc = NULL;

    /* allocate dma descriptors */
    scerr = ddma_descriptor_alloc( &icetest_desc );
    if( scerr != SCANERR_NONE ) {
        final_scerr = scerr;
        goto fail;
    }

    /* TODO get icetest pixels per row from sensor? */
    icetest_pixels_per_row = pixels_per_row;

    icetest_bytes_per_row = icetest_pixels_per_row * sizeof(uint32_t);

    /* davep 09-Feb-2012 ; hail mary ; put full image in one buffer */
//    rows_per_buffer = total_rows;
    /* well, that didn't work. Too big */
    rows_per_buffer = MAX_ROWS_PER_ICEFILE_BUFFER;

    total_bytes = icetest_bytes_per_row * rows_per_buffer;
    dbg2( "%s tr=%d rpb=%d total_bytes=%d\n", __FUNCTION__, total_rows, rows_per_buffer, total_bytes );

    icetest_icebuf = icebuf_new( rows_per_buffer, icetest_bytes_per_row, ICEBUF_TAG_ICETEST_IDMA );
    if( icetest_icebuf==NULL ) {
        final_scerr = SCANERR_OUT_OF_MEMORY;
        goto fail;
    }

    /* ready to be filled from the host! */
    icetest_icebuf->num_rows = 0;
    icetest_icebuf->datalen = 0;

    return SCANERR_NONE;

fail : 
    icetest_free_memory();

    return final_scerr;
}

static scan_err_t icetest_start( void )
{
    scan_err_t scerr;
    int sensor = 0;

    scerr = icetest_soft_setup();
    XASSERT( scerr==SCANERR_NONE, scerr );

    /* davep 09-Feb-2012 ; make sure cisx is properly clean and in bypass */
    // NOTE: Daves icefile tools are all geared to provide input directly to 
    //       PIC, so CISX is bypassed here.
    dbg1( "%s cisx in bypass for icefile\n", __FUNCTION__ );
    cisx_soft_setup();
    cisx_set_bypass(true);

    /* turn on the ICETest hardware */
    icetest_enable( true );
    icetest_idma_enable();

    icetest_set_test_mode( sensor, ICETEST_CONFIG_DATA_TO_CISX );

    return SCANERR_NONE;
}

static scan_err_t run_icebuf( struct ice_dma_buffer *icebuf )
{
    static uint32_t total_rows=0;
    int sensor;

    /* cheap kcah for debugging */
    if( icebuf==NULL ) {
        total_rows = 0;
        return SCANERR_NONE;
    }
    icebuf_sanity( icebuf );

    sensor = (*((uint32_t*)icebuf->data) & PIC_INDATA_TAGOUT_MASK) >> PIC_INDATA_TAGOUT_SHIFT;
    icetest_set_test_mode( sensor, ICETEST_CONFIG_DATA_TO_CISX );

    ice_dbg2( "%s num_rows=%d bytes=%d sensor=%d\n", __FUNCTION__, icebuf->num_rows, icebuf->datalen, sensor );

#ifdef ICEFILE_DEBUG
    scanlog_hex_dump( (unsigned char *)icebuf->data, 32 );
#endif

    /* davep 07-Jun-2011 ; sanity check ; if this assert fails, I have a
     * partially filled icebuf and I need to add code to better handle
     * fragmented rows
     */
    ice_dbg2( "%s mr=%d nr=%d bpr=%d datalen=%d\n", __FUNCTION__,
                icebuf->max_rows, icebuf->num_rows, icebuf->bytes_per_row,
                icebuf->datalen );
    XASSERT( icebuf->num_rows*icebuf->bytes_per_row==icebuf->datalen, icebuf->datalen );

//    cpu_dcache_writeback_region( icebuf->data, icebuf->datalen );

    /* set up simple icetest input and pic output dma descriptors */
    icetest_desc->config_flags = DDMA_DESCRIPTOR_CONFIG_OWNER_BLOCK
                       | DDMA_DESCRIPTOR_CONFIG_STOP 
                       | DDMA_DESCRIPTOR_CONFIG_INT  
                       ;
    icetest_desc->src_addr = (uint32_t)dma_map_single( NULL, (void *)icebuf->data, icebuf->datalen, DMA_TO_DEVICE );
    icetest_desc->transfer_len_bytes = icebuf->datalen;
    icetest_desc->fw_src_addr = (void *)icetest_desc->src_addr;

    ddma_desc_flush( icetest_desc );

    /* blocking wait for icetest DMA to complete */
    icetest_idma_desc_write_and_wait( icetest_desc, icebuf->bytes_per_row );

    if ( sensor == 0 ) {
        total_rows += icebuf->num_rows;
    }

    ice_dbg2( "%s total_rows=%d\n", __FUNCTION__, total_rows );

    /* davep 21-Jun-2011 ; shotgun debug for PIC overflows; separate icetest
     * buffers to allow PIC WDMA a chance to drain. Otherwise, we'll need
     * pause/resume here in icetest.  TODO hook icetest into scancmdq?  we'll
     * get pause/resume as a bonus.
     */
//    tx_thread_sleep( 1 ); /* 1 tick == 10ms */
//    tx_thread_sleep( SECONDS/2 );
//    tx_thread_relinquish();
    //msleep(1);
     schedule();

    /* ready to be filled again! */
    icebuf->num_rows = 0;
    icebuf->datalen = 0;

    return SCANERR_NONE;
}

/**
 * \brief  Parse an ASP icefile open message and open icefile
 *
 * Does a series of scanity checks on the incoming ASP icefile open message. If
 * everything looks fine, will do the actual icefile open.
 *
 * A scan must be running before icefile is opened.
 *
 * \author David Poole
 * \date 12-Nov-2010
 *
 */

scan_err_t handle_icefile_open( void )
{
    scan_err_t scerr;
    struct scanvars *sv;
//    int i;

    dbg2( "%s\n", __FUNCTION__ );

    sv = scanvar_get();
    if ( (sv->doc_src == SCAN_DOCUMENT_SOURCE_FLATBED) ||
         (sv->doc_src == SCAN_DOCUMENT_SOURCE_FLATBED_NSENSOR) ) 
    {
        use_pipecutter = false;
    }
    else {
        use_pipecutter = true;
        smech_adf_paper_event_callback(SCAN_ADF_PAPER_TOF, true, 0);
    }

    /* sanity check data */
    if( icefile_state!=ICEFILE_CLOSED ) {
        /* already open */
        return SCANERR_NOT_PERMITTED;
    }

    if( !scanlib_is_locked() ) {
        dbg2( "%s scan not running; no icefile for you\n", __FUNCTION__ );
        return SCANERR_NOT_PERMITTED;
    }

//    if( open_hdr->num_channels!=1 && open_hdr->num_channels!=3 ) {
//        dbg2( "%s num_channels=%d not in (1,3)\n", __FUNCTION__, open_hdr->num_channels );
//        return SCANERR_INVALID_PARAM;
//    }

//    for( i=0 ; i<open_hdr->num_channels ; i++ ) {
//        /* stupid human check ; icefile uses PIC channel numbers so make sure
//         * we're not being told something stupid
//         */
//        if( open_hdr->channels[i] >= PIC_WDMA_NUM_CHANNELS ) {
//            dbg2( "%s channel=%d not a valid PIC channel\n", __FUNCTION__, 
//                        open_hdr->channels[i] );
//            return SCANERR_INVALID_PARAM;
//        }
//    }

    icefile_get_image_info( &image_info );

    if( image_info.total_rows==0 || image_info.pixels_per_row==0 ) {
        /* gently ignore invalid image info */
        dbg2( "%s invalid image info total_rows=%d pixels_per_row=%d\n", __FUNCTION__, 
                    image_info.total_rows, image_info.pixels_per_row );
        return SCANERR_INVALID_PARAM;
    }

    /* allocate global data structures and buffers  */
    scerr = icetest_alloc_memory( image_info.total_rows, image_info.pixels_per_row );
    if( scerr != SCANERR_NONE ) {
        return scerr;
    }

    /* set up and start the hardware */
    scerr = icetest_start();
    if( scerr != SCANERR_NONE ) {
        icetest_free_memory();
        return scerr;
    }

    icefile_state = ICEFILE_OPEN;

    /* davep 14-Jun-2011 ; clear some test/debug counters */
    run_icebuf( NULL );

    return SCANERR_NONE;
}
EXPORT_SYMBOL(handle_icefile_open);

static int launch( const char __user *buf, size_t byte_count, struct ice_dma_buffer *icebuf )
{
    int bytes_read, bytes_to_read, retval;
    uint8_t *ptr;
    int err;

    /* we return the total number of bytes read in this pass */
    bytes_read = 0;

    /* stay in this loop as long as :
     *  we have a place to read data into &&
     *  we still haven't read everything we're supposed to read in this pass
     *
     *  We will also jump out of this loop if the read function returns
     *  retval<0 (error) or retval==zero (no data).
     */
    while( byte_count>0 ) {

//        dbg2( "%s %ld %ld %ld\n", __FUNCTION__, icebuf->max_rows,
//                    icebuf->bytes_per_row, icebuf->datalen );

        bytes_to_read = MIN( byte_count, 
                            icebuf->max_rows*icebuf->bytes_per_row - icebuf->datalen );

        /* Get a pointer to the next place we want data to land.
         * NOTE THIS IS A CACHE BYPASS POINTER!
         */
        ptr = icebuf->data + icebuf->datalen;

        /* call the callback function to get data; the callback function is
         * required to have the same semantics as a C-library non-blocking
         * read: return 0 on no data, <0 on error, >0 as number of bytes read.
         */
//        retval = iopipe->pipe_read( iopipe, ptr, bytes_to_read, PARSEREAD_DELAY );
        err = copy_from_user(ptr, buf + bytes_read, bytes_to_read);
        ASSERT(err == 0);
        retval = bytes_to_read;

        ice_dbg2( "icefile bytes_to_read=%d retval=%d\n", bytes_to_read, retval );

        if( retval<0 ) {
            /* error! immediately push the error up to our caller so we can
             * stop reading 
             */
            return retval;
        }

//        if( retval==0 ) {
//            /* no data read this time; leave and come back later */
//            posix_sleep_seconds(1);
//            break;
//        }

        icebuf->datalen += retval;
        bytes_read += retval;
        byte_count -= retval;
        XASSERT( byte_count>=0, byte_count );

        /* This is a bit of a kludge; we need to be able to count rows so we
         * know when to stop. However we are receiving bytes so we could
         * wind up mid-row. Set num_rows to our closest full row count. 
         */
        icebuf->num_rows = icebuf->datalen / icebuf->bytes_per_row;
       
        ice_dbg2( "%s icebuf=%p datalen=%d num_rows=%d\n", __FUNCTION__,
                          icebuf, icebuf->datalen, icebuf->num_rows );

        /* we filled up the buffer; pass it off for processing then go back for
         * more
         */
        if( icebuf->num_rows == icebuf->max_rows ) {
            XASSERT( icebuf->datalen==icebuf->max_rows*icebuf->bytes_per_row, icebuf->datalen );
            run_icebuf( icebuf );
        }
    }

    return bytes_read;
}

/**
 * \brief  Wait for icefile to completely close.
 *
 * We handle icefile's flush and close in a blocking mode. The icefile flush
 * and close polls the message queue. 
 *
 * \author David Poole
 * \date 22-Mar-2006
 *
 */

static scan_err_t blocking_close_icefile( void )
{
    dbg2( "%s\n", __FUNCTION__ );

    if( icefile_state != ICEFILE_OPEN ) {
        /* whoops! shouldn't have been sent a close */
        return SCANERR_NOT_PERMITTED;
    }

    icetest_idma_disable();
    icetest_enable( false );

    icetest_free_memory();

    icefile_state = ICEFILE_CLOSED;

    return SCANERR_NONE;
}

static scan_err_t icefileapp_run_internal_page( void )
{
    scan_err_t scerr;
    scan_err_t final_scerr;
    uint32_t completed_rows, remaining_rows;
    scan_cmode_t cmode;
    struct ice_dma_buffer *icebuf;
    struct ddma_channel dch;
    ATLISTENTRY running;
    int num_running;
    int bytes_per_row;
    int rows_per_buffer;
    bool end_of_image = false;

    dbg2( "%s\n", __FUNCTION__ );

    XASSERT( icefile_state==ICEFILE_CLOSED, icefile_state );

    final_scerr = SCANERR_NONE;

    icefile_get_image_info( &image_info );

    if( image_info.total_rows==0 || image_info.pixels_per_row==0 ) {
        /* gently ignore invalid image info */
        dbg2( "%s invalid image info total_rows=%d pixels_per_row=%d\n", __FUNCTION__, 
                    image_info.total_rows, image_info.pixels_per_row );
        return SCANERR_INVALID_PARAM;
    }

    /* we're not using the globals in this function */
    XASSERT( icetest_icebuf==NULL, (uint32_t)icetest_icebuf  );
    XASSERT( icetest_desc==NULL, (uint32_t)icetest_desc );

    /* hardwire to mono for now */
    cmode = SCAN_CMODE_MONO;

    ATInitList( &running );
    num_running = 0;

    /* allocate a big descriptor chain */
    memset( &dch, 0, sizeof(struct ddma_channel) );
    ddma_channel_open( &dch, "icefileapp", 0 );
    scerr = ddma_channel_alloc_descriptors( &dch, 256 );
//    scerr = ddma_channel_alloc_descriptors( &dch, 64 );    
    if( scerr != SCANERR_NONE ) {
        ddma_channel_close( &dch );
        return scerr;
    }

    
    rows_per_buffer = MAX_ROWS_PER_ICEFILE_BUFFER;
    bytes_per_row = image_info.pixels_per_row * sizeof(uint32_t);

    /* Loop allocating icebufs for the full image. Push icebufs into descriptor
     * chain.
     */
    completed_rows = 0;
    remaining_rows = image_info.total_rows;

    while( completed_rows < image_info.total_rows  )
    {
        icebuf = icebuf_new( rows_per_buffer, bytes_per_row, ICEBUF_TAG_ICETEST_IDMA );
        if( icebuf==NULL ) {
            dbg1( "%s out of memory for icebufs at count=%d\n", __FUNCTION__, num_running  );
            final_scerr = SCANERR_OUT_OF_MEMORY;
            goto leave;
        }
        ATInsertTailList( &running, &icebuf->listnode );
        num_running++;

        icebuf->num_rows = MIN( icebuf->max_rows, remaining_rows );
        icebuf->datalen = icebuf->num_rows * icebuf->bytes_per_row;

        if (completed_rows + icebuf->num_rows >= image_info.total_rows)
            end_of_image = true;
        // pixels_per_row
        icetest_test_buffer_init(cmode, icebuf->data, 
                                 icebuf->datalen,
                                 image_info.pixels_per_row,
                                 icebuf->num_rows, end_of_image);
        icebuf_sanity( icebuf );

#if 0
        if (completed_rows == 0)
        {
            dbg1("%s printing row 0\n", __func__);
            scanlog_hex_dump(icebuf->data, icebuf->bytes_per_row);
        }
#endif        
        completed_rows += icebuf->num_rows;
        remaining_rows -= icebuf->num_rows;
#if 0
        if (completed_rows >= image_info.total_rows)
        {
            dbg1("%s printing row %d (total rows=%d)\n", __func__, completed_rows, image_info.total_rows);
            scanlog_hex_dump((icebuf->data + ((icebuf->num_rows-1) * icebuf->bytes_per_row)), icebuf->bytes_per_row);
        }
#endif        

//        cpu_dcache_writeback_region( icebuf->data, icebuf->datalen );

        /* davep 02-Jul-2013 ; TODO port this to Linux kernel. The dma_addr_t
         * cast is WRONG for linux kernel. Must add dma handle to this function
         */
        ddma_channel_load( &dch, icebuf->data, icebuf->num_rows, icebuf->bytes_per_row );

//        dbg2( "%s completed=%d remaining=%d total=%d\n", __FUNCTION__, 
//                    completed_rows, remaining_rows, image_info.total_rows );
    }

    icefile_state = ICEFILE_OPEN;

    /* modify my chain so we only get an interrupt on the LAST descriptor. The
     * 'write_and_wait' function below will block waiting for a single
     * interrupt from an icetest interrupt handler.
     */
    {
        struct ddma_descriptor *curr;
        curr = dch.head_desc;
        while( curr != dch.tail_desc ) {
            curr->config_flags &= ~DDMA_DESCRIPTOR_CONFIG_INT;
            ddma_desc_flush( curr );
            curr = (struct ddma_descriptor *)curr->fw_next_descriptor_addr;
        }
    }
//    ddma_desc_chain_dump( dch.head_desc, dch.num_descriptors );

    /* set up and start the hardware */
    scerr = icetest_start();
    if( scerr != SCANERR_NONE ) {
        goto leave;
    }

    /* fire in the hole! push the head of the descriptor chain into hardware */

    icetest_idma_desc_write_and_wait( dch.head_desc, bytes_per_row );

    /* done! */
    scerr = blocking_close_icefile();
    XASSERT( scerr==SCANERR_NONE, scerr );

leave:
    /* clean up after ourselves */
    ddma_channel_free_descriptors( &dch );
    ddma_channel_close( &dch );
    ice_free_counted_list( &running, num_running );

    /* additional safety net */
    XASSERT( icetest_icebuf==NULL, (uint32_t)icetest_icebuf  );
    XASSERT( icetest_desc==NULL, (uint32_t)icetest_desc );

    dbg2( "%s done scerr=%d\n", __FUNCTION__, final_scerr );

    return final_scerr;
}

scan_err_t handle_new_icefile_data( const char __user *buf, size_t count )
{
    int retval;

    ice_dbg2( "%s datalen=%d\n", __FUNCTION__, count );
    
    if( icefile_state != ICEFILE_OPEN ) {
        /* ignore it; this will probably break the protocol sequencing but
         * screw the host if he's sending us data when he's not supposed to
         */
        dbg2( "%s icefile not open!\n", __FUNCTION__ );
        return SCANERR_NOT_PERMITTED;
    }

//    /* the *4 is because icetest has a 32-bit "pixel" so the incoming
//     * bytes_per_row must match
//     */
//    if( datahdr->bytes_per_row != image_info.pixels_per_row*sizeof(uint32_t) ) {
//        dbg2( "%s invalid data header bytes_per_row packet=%d expected=%d\n", __FUNCTION__, 
//                    datahdr->bytes_per_row, image_info.pixels_per_row*4 );
//        return SCANERR_INVALID_PARAM;
//    }

//    if( datahdr->channel >= ICEFILE_NUM_CHANNELS ) {
//        /* again, this will break the protocol sequencing but this is Q&D for
//         * now  (I just know I'll regret this later...)
//         */
//        return SCANERR_INVALID_PARAM;
//    }

//    /* deep sanity check; only allow 16M packets max (this is to catch protocol
//     * errors more than anything else)
//     */
//    if( datahdr->datalen_bytes > 0x1000000 ) {
//        return SCANERR_INVALID_PARAM;
//    }

//    bytes_read = 0;

    /* we stay here until we've ready the entire data packet */
//    while( bytes_read < datahdr->datalen_bytes ) {

//        ice_dbg2( "%s read bytes=%d\n", __FUNCTION__, datahdr->datalen_bytes-bytes_read );

        retval = launch( buf, count, icetest_icebuf );
        if( retval < 0 ) {
            /* read error */
            return SCANERR_GENERIC_FAIL;
        }

//        bytes_read += retval;
//
//        /* Wee! Go back for more data! */
//    }

    ice_dbg2( "%s done\n", __FUNCTION__ );

    return SCANERR_NONE;
}
EXPORT_SYMBOL(handle_new_icefile_data);

scan_err_t handle_icefile_close( void )
{
    scan_err_t scerr;

    dbg2( "%s icefile close message\n", __FUNCTION__ );

    /* send any partial data (flush) */
    if( icetest_icebuf && icetest_icebuf->num_rows > 0 ) {
        run_icebuf( icetest_icebuf );
    }

    if ( use_pipecutter ) {
        smech_adf_paper_event_callback(SCAN_ADF_PAPER_TOF, false, 0);
    }

    scerr = blocking_close_icefile();

    return scerr;
}
EXPORT_SYMBOL(handle_icefile_close);

void icefileapp_onetime_init( void )
{

}

scan_err_t icefileapp_run_internal_page_msg( void )
{
    scan_err_t scerr;
    scerr = icefileapp_run_internal_page();
    if( scerr != SCANERR_NONE ) {
        dbg1( "icefile internal page failed to run scerr=%d\n", scerr );
    }
    return scerr;
}

