/*
**************************************************************************
This Source Code Form is subject to the terms of the Mozilla Public
License, v. 2.0. If a copy of the MPL was not distributed with this file,
You can obtain one at http://mozilla.org/MPL/2.0/.

Copyright (c) 2012-2015, Marvell International Ltd.

Alternatively, this software may be distributed under the terms of the GNU
General Public License Version 2, and any use shall comply with the terms and
conditions of the GPL.  A copy of the GPL is available at
http://www.gnu.org/licenses/old-licenses/gpl-2.0.html

THE FILE IS DISTRIBUTED AS-IS, WITHOUT WARRANTY OF ANY KIND, AND THE
IMPLIED WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE
ARE EXPRESSLY DISCLAIMED.  The GPL license provides additional details about
this warranty disclaimer.
******************************************************************************
*/


/* 
 * Icefile, Linux Kernel edition.
 *
 * Provides a path for test image data to flow into the kernel.  The data is
 * then fed through the icetest hardware block as input to the scanpipe.  This
 * serves as a replacement for the acl path used by non-linux systems.
 *
 */

#include <linux/module.h> /* Needed by all modules */
#include <linux/kernel.h> /* Needed for KERN_INFO */
#include <linux/fs.h>
#include <linux/kthread.h>
#include <linux/cdev.h>
#include <linux/kfifo.h>
#include <linux/hardirq.h>
#include <linux/list.h>
#include <linux/uaccess.h>
#include <linux/ioctl.h>
#include <linux/uio.h>

#include "lassert.h"
#include "memAPI.h"

#include "scantypes.h"
#include "scancore.h"
#include "scandbg.h"
#include "siqdata.h"
#include "adfsensor.h"
#include "smsg_decode.h"
#include "chipgap.h"

#include "icefile.h"

// ----------- linux device code -------------------

#define ICEFILE_DEV_COOKIE 0xD782B55C

struct icefile_dev {
    uint32_t cookie;
    struct mutex lock;
    struct cdev cdev;
};

static struct icefile_dev icefile_device;

static dev_t icefile_dev_num;

// (Define in scantask)
scan_err_t handle_icefile_open( void );
scan_err_t handle_new_icefile_data( const char __user *buf, size_t count );
scan_err_t handle_icefile_close( void );

static int icefile_open( struct inode *inode, struct file *filp )
{
    dbg2( "%s\n", __FUNCTION__ );

    filp->private_data = NULL;

    /* success! */
    return 0;
}

static int icefile_release( struct inode *inode, struct file *filp )
{
    scan_err_t scerr;

    dbg2( "%s\n", __FUNCTION__ );

    if (filp->private_data != NULL)
    {
        scerr = handle_icefile_close();
        XASSERT(scerr == SCANERR_NONE, scerr);
    }

    /* success! */
    return 0;
}

static ssize_t icefile_read(struct file *filp, char __user *buf, size_t count, loff_t *offp )
{
    int bytes_copied = 0;

    dbg2( "%s\n", __FUNCTION__ );

    if (*offp == 0)
    {
        struct icefile_image_info image_info;

        icefile_get_image_info( &image_info );

        image_info.rows_per_buffer = icefile_get_rows_per_buffer();
        bytes_copied = snprintf(buf, count, "pixels_per_row=%d\ntotal_rows=%d\nrows_per_buffer=%d\n", 
                                image_info.pixels_per_row, image_info.total_rows, image_info.rows_per_buffer);
        buf[count] = '\0';
        *offp += bytes_copied;
    }

    return bytes_copied;
}

static ssize_t icefile_write(struct file *filp, const char __user *buf, size_t count, loff_t *offp )
{
    scan_err_t scerr;

    dbg2( "%s (%d bytes @ %d)\n", __FUNCTION__, count, (int)*offp );

    if (*offp == 0)
    {
        scerr = handle_icefile_open();
        XASSERT(scerr == SCANERR_NONE, scerr);

        filp->private_data = (void *) 1;
    }

    scerr = handle_new_icefile_data(buf, count);
    XASSERT(scerr == SCANERR_NONE, scerr);

    *offp += count;

    return count;
}

static long icefile_unlocked_ioctl(struct file *filp, unsigned int cmd, unsigned long arg)
{
    int retcode = 0;

    dbg2( "%s cmd=%d arg=%ld\n", __FUNCTION__, cmd, arg );

    return retcode;
}

static const struct file_operations icefile_fops = {
    .owner = THIS_MODULE,
    .open = icefile_open,
    .read = icefile_read,
    .write = icefile_write,
    .unlocked_ioctl = icefile_unlocked_ioctl,
    .release = icefile_release
};

void icefile_cleanup( void )
{
    struct icefile_dev *dev;

    dev = &icefile_device;

    unregister_chrdev_region( icefile_dev_num, 1 );

    cdev_del( &dev->cdev );
}

static int __init icefile_init_module(void)
{
//    scan_err_t scerr;
    int retcode;
    struct icefile_dev *dev;

    dbg2( "%s\n", __FUNCTION__ );

    dev = &icefile_device;
    dev->cookie = ICEFILE_DEV_COOKIE;

    mutex_init( &dev->lock );

    retcode = alloc_chrdev_region( &icefile_dev_num, 0, 1, "icefile" );
    if( retcode != 0 ) {
        // A non 0 return means init_module failed; module can't be loaded. 
        return -1;
    }

    cdev_init( &dev->cdev, &icefile_fops );

    retcode = cdev_add( &dev->cdev, icefile_dev_num, 1 );
    if( retcode != 0 ) {
        unregister_chrdev_region( icefile_dev_num, 1 );
        // A non 0 return means init_module failed; module can't be loaded. 
        return -1;
    }

    return 0;
}

static void __exit icefile_cleanup_module(void)
{
    dbg2( "%s\n", __FUNCTION__ );
}

module_init( icefile_init_module );
module_exit( icefile_cleanup_module );


MODULE_AUTHOR("Copyright (c) 2014 Marvell International Ltd. All Rights Reserved");
MODULE_DESCRIPTION("Marvell Scan test path");

MODULE_LICENSE("GPL");
MODULE_VERSION("2014_MAY_15");

