/*
**************************************************************************
This Source Code Form is subject to the terms of the Mozilla Public
License, v. 2.0. If a copy of the MPL was not distributed with this file,
You can obtain one at http://mozilla.org/MPL/2.0/.

Copyright (c) 2008-2016, Marvell International Ltd.

Alternatively, this software may be distributed under the terms of the GNU
General Public License Version 2, and any use shall comply with the terms and
conditions of the GPL.  A copy of the GPL is available at
http://www.gnu.org/licenses/old-licenses/gpl-2.0.html

THE FILE IS DISTRIBUTED AS-IS, WITHOUT WARRANTY OF ANY KIND, AND THE
IMPLIED WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE
ARE EXPRESSLY DISCLAIMED.  The GPL license provides additional details about
this warranty disclaimer.
******************************************************************************
*/


/**
 * \file scanplat.h
 *
 * \brief  Platform-specific scanner and motor settings interface
 *
 */


#ifndef SCANPLAT_H
#define SCANPLAT_H

/* forward decl; see scancap.h */
struct scan_platform_capabilities;

/**
 * \brief  Initialize the motor hardware
 *
 * One-time initialization for motors - create and initialize the specific mech
 * driver.
 *
 * \remark A Null motor driver is supported for debug and early bring-up
 * functionality.  The null motor driver does not have a motor attached,
 * but has enough of the Scanmech interface to let the scan system think it's
 * talking to a motor, and allow a scan to proceed
 */
void scanplat_init_mech(void);


/**
 * \brief  Turn on the GPIOs to allow communication to the scanner
 *
 * The ASIC, by default, may have all comm lines to the scanners
 * turned off.  This function is called by the scanner initialization
 * code to turn on the lines.
 */
void scanplat_init_scan_gpio( void );

/**
 * \brief  Gets the maximum scan length
 *
 * Created for configuring scannable area between ADF and
 * flatbed designs.
 *
 * \retval uint32_t  the maximum scan length in 1/1000's of an inch
 */
uint32_t scanplat_get_max_scan_length( scan_document_source_t doc_src );

/**
 * \brief  Return a structure indicating hardware capabilities of the current
 * scan platform.
 *
 * Members of this structure will not change at run-time.
 *
 * \author David Poole
 * \date 04-Jan-2009
 *
 */

void scanplat_get_capabilities( struct scan_platform_capabilities *caps );


/*\brief Get/set PIC ESD settings
 *
 *
 */
scan_err_t scanplat_get_esd_settings(bool *p_esd_dir,
                         uint16_t *p_esd_accum_size,
                                     uint16_t *p_esd_start_offset,
                                     uint8_t *p_esd_change,
                                     uint8_t *p_esd_tolerance);

scan_err_t scanplat_debug_set_esd_settings(bool *p_esd_dir,
                                           uint16_t *p_esd_accum_size,
                                           uint16_t *p_esd_start_offset,
                                           uint8_t *p_esd_change,
                                           uint8_t *p_esd_tolerance);


scan_err_t scanplat_get_psense_settings(uint16_t *p_start_pixel,
                                           uint8_t  *p_region_width,
                                           uint8_t *p_region_tolerance,
                                           uint8_t *p_total_region_tolerance,
                                           uint8_t *p_toggle01,
                                           uint8_t *p_toggle10,
                                           uint8_t *p_count0,
                                           uint8_t *p_count1 );

scan_err_t scanplat_debug_set_psense_settings(uint16_t *p_start_pixel,
                                              uint8_t  *p_reg_tolerance, uint8_t *p_total_reg_tolerance,
                                              uint8_t  *p_toggle01, uint8_t *p_toggle10,
                                              uint8_t  *p_count0,   uint8_t *p_count1);

uint32_t scanplat_calc_sclkper(scan_cmode_t cmode, uint32_t dpi, uint32_t pixeltimes_per_line,
                               scan_document_source_t doc_src);

typedef struct 
{
    /* Override these values if you need to use a starting default gain other than 1.0
     */
    bool    analog_cal_default_gain_not_null;
    float   analog_cal_default_gain;

    uint8_t illum_start_val;           // 0 - 100
    uint8_t illum_reduction_percent;   // 0 - 100 (percent)
    uint8_t expected_output_reduction; // 0 - 100 (percent) - amount we expect output to reduce when we reduce illumination by specified percent

} scanplat_cal_config_t;

/**
 * \brief  Retrieves a platform specific structure that should 
 *         be used when doing a softreset for a given AFE.
 *
 * Allows each platform to customize the way specific AFEs are 
 * reset on their platform. 
 *
 * \retval uint32_t  index of the AFE
 */
void *scanplat_afe_get_softsetup_param(int afe);

/**
 * \brief  Provides platform specific calibration settings for 
 *         each sensor.
 *
 * Allows each platform to customize starting gains, etc.
 *
 * \retval uint32_t  index of the sensor being calibrated
 */
scanplat_cal_config_t *scanplat_get_cal_params(int sensor);


#endif /* SCANPLAT_H */

