/*
 *
 *  Copyright (C) 2008-2009 RICOH Co.,LTD.
 *  All rights reserved.
 *
 *  affiliation	:DSS Development Center
 *  		 Document Solutions & Services Division
 * 
 *  purpose	: Performs user authentication using PanelService API.
 *
 */

package jp.co.ricoh.dsdk.ap.sample.localauth.auth;

import java.io.IOException;

import jp.co.ricoh.dsdk.ap.sample.localauth.app.CurrentInfo;
import jp.co.ricoh.dsdk.ap.sample.localauth.cs.IDService;
import jp.co.ricoh.dsdk.opt.auth.ccm.AlreadySetException;
import jp.co.ricoh.dsdk.opt.auth.ccm.Ccm;
import jp.co.ricoh.dsdk.opt.auth.ccm.CcmRequest;
import jp.co.ricoh.dsdk.opt.auth.ccm.CcmRequestHandler;
import jp.co.ricoh.dsdk.opt.auth.ccm.CcmService;
import jp.co.ricoh.dsdk.opt.auth.ccm.InvalidPINException;
import jp.co.ricoh.dsdk.opt.auth.ccm.Name;
import jp.co.ricoh.dsdk.opt.auth.ccm.Password;
import jp.co.ricoh.dsdk.opt.auth.ccm.UnsupportedRequestException;
import jp.co.ricoh.dsdk.scard.framework.card.Card;
import jp.co.ricoh.dsdk.scard.framework.card.CardAccessException;
import jp.co.ricoh.dsdk.scard.framework.service.CardServiceException;

/**
 * このクラスは、PanelServiceを使用してユーザ認証を行うクラスです。
 * This class performs user authentication using PanelService. 
 */
public class LoginService {
	
	private LocalAuth localAuth= null;
	private CcmService ccms = null;
	private CcmRequestHandler handler = null;
	private Card card = null;
	private LoginUser currentUser = null;
	
	/**
	 * LoginServiceのインスタンスを作成します。
	 * Creates a LoginService instance.
	 */
	public LoginService() throws IOException {
		localAuth = new LocalAuth();
		currentUser = new LoginUser(new byte[0], "", "");
		
		System.setProperty("jp.co.ricoh.dsdk.opt.auth.ccm.debug", "1");
		ccms = new Ccm().getService();
		handler = new CcmRequestHandler() {
			public void handle(CcmRequest[] req) 
			throws UnsupportedRequestException, InvalidPINException, IOException {
				handleCcmRequest(req);
			}
		};
	}
	
	/**
	 * 認証要求のハンドルを開始します。
	 * Starts handling authentication requests. 
	 */
	public void start() throws AlreadySetException {
		ccms.setRequestHandler(handler);
	}
	
	/**
	 * 認証要求のハンドルを停止します。
	 * Stops handling the authentication requests.
	 */
	public void stop() {
		ccms.removeRequestHandler(handler);
	}
	
	/**
	 * ユーザ認証ログイン要求を出します。
	 * Requests the user authentication login.
	 */
	public void requestLogin(Card card) {
		this.card = card;
		ccms.requestLogin();
	}
	
	/**
	 * ユーザ認証ログアウト要求を出します。
	 * Requests the user authentication logout.
	 */
	public void requestLogout() {
		ccms.requestLogout();
		CurrentInfo.clear();
	}
	
	/**
	 * 現在のユーザを取得します。
	 * Returns the current LoginUser.
	 */
	public LoginUser getCurrentUser() {
		return currentUser;
	}
	
	/**
	 * 認証要求の処理を行います。
	 * Processes the authentication requests. 
	 */
	private void handleCcmRequest(CcmRequest[] req) 
	throws UnsupportedRequestException, CardAccessException, CardServiceException {
		currentUser = new LoginUser(new byte[0], "", "");
		
		Name name = null;
		Password password = null;
		
		/* 要求の振り分け */
		/* Categorizes each request */
		for (int i = 0; i < req.length; i++) {
			
			if (req[i] instanceof Name) {               
				name = (Name) req[i];
				
			} else if (req[i] instanceof Password) {  
				password = (Password) req[i];
				
			} else {
				throw new UnsupportedRequestException("Unsupported " + req[i]);
			}
		}
		
		try{
			card.lock();
			
			IDService idService = 
				(IDService) card.getCardService(IDService.class);
			
			byte[] tokenID = idService.getID();
			
			currentUser.setId(tokenID);
			
			/* IDに対応するユーザを取得 */
			/* Obtains the corresponding user to the ID */
			if (localAuth.authenticate(tokenID)) {
				currentUser.setLoginName(localAuth.getAuthenticatedUser().getLoginName());
				currentUser.setLoginPassword(localAuth.getAuthenticatedUser().getLoginPassword());
			}
			
			/* ログインユーザ名のセット */
			/* Sets the login user name */
			if (name != null) {
				name.setName(currentUser.getLoginName().getBytes());
			}
			
			/* ログインパスワードのセット */
			/* Sets the login password */
			if (password != null) {
				password.setPassword(currentUser.getLoginPassword().getBytes());
			}
			
		} finally {
			card.unlock();
		}
		
		/* 現在のID、ユーザ情報をセット */
		/* Sets the current ID and current user */
		CurrentInfo.set(currentUser);
	}
	
}
