/*
 *
 *  Copyright (C) 2008-2009 RICOH Co.,LTD.
 *  All rights reserved.
 *
 *  affiliation	:DSS Development Center
 *  		 Document Solutions & Services Division
 * 
 *  purpose	: PanelService sample. SCF is also used in this sample.
 *
 */

package pssample;

import java.io.IOException;

import jp.co.ricoh.dsdk.opt.auth.ccm.AlreadySetException;
import jp.co.ricoh.dsdk.opt.auth.ccm.Ccm;
import jp.co.ricoh.dsdk.opt.auth.ccm.CcmRequest;
import jp.co.ricoh.dsdk.opt.auth.ccm.CcmRequestHandler;
import jp.co.ricoh.dsdk.opt.auth.ccm.CcmService;
import jp.co.ricoh.dsdk.opt.auth.ccm.InvalidPINException;
import jp.co.ricoh.dsdk.opt.auth.ccm.Name;
import jp.co.ricoh.dsdk.opt.auth.ccm.Password;
import jp.co.ricoh.dsdk.opt.auth.ccm.UnsupportedRequestException;
import jp.co.ricoh.dsdk.opt.auth.ccm.VerifyPIN;
import jp.co.ricoh.dsdk.scard.framework.card.Card;
import jp.co.ricoh.dsdk.scard.framework.card.CardAccessException;
import jp.co.ricoh.dsdk.scard.framework.card.CardManager;
import jp.co.ricoh.dsdk.scard.framework.card.InitializeException;
import jp.co.ricoh.dsdk.scard.framework.card.Slot;
import jp.co.ricoh.dsdk.scard.framework.event.CardEvent;
import jp.co.ricoh.dsdk.scard.framework.event.CardEventListener;
import jp.co.ricoh.dsdk.scard.framework.service.CardServiceException;
import jp.co.ricoh.dsdk.scard.framework.service.CardServiceRegister;
import jp.co.ricoh.dsdk.scard.framework.service.CardServiceRegistryException;
import jp.co.ricoh.dsdk.scard.option.security.CHVException;
import jp.co.ricoh.dsdk.scard.option.security.VerificationException;

import org.osgi.framework.BundleActivator;
import org.osgi.framework.BundleContext;

/**
 * This is a sample program of an authentication bundle using SCF.
 * If more than one card readers are connected to the device, 
 * a card which is firstly inserted will be efficient.
 * 
 * SCFを利用した認証バンドルのサンプルです。
 * 複数のリーダーを同時に使用した場合、最初に認識されたカードを有効とします。
 */
public class PanelServiceSampleBundle implements BundleActivator{
	
	/**
	 * The CcmService object
	 * 
	 * CCMサービス
	 */
	private CcmService ccmService;
	
	/**
	 * The CcmHandler object
	 * 
	 * CCMハンドラ
	 */
	private CcmRequestHandler ccmReqHandler;
	
	/**
	 * The CardManager object
	 * 
	 * カードマネージャ
	 */
	private CardManager cardManager;

	/**
	 * The CardEventListener object
	 * 
	 * カードイベントリスナ
	 */
	private CardEventListener cardEventListener;
	
	/**
	 * The slot where a card is inserted
	 * 
	 * カードがセットされているスロット
	 */
	private Slot slot;
	
	/**
	 * Starts bundle.
	 * 
	 * バンドル開始。
	 */
	public void start(BundleContext context) throws Exception {
		
		/*
		 * Set the debug log property for PanelService
		 * (PanelService does not print out its debug log by default.)
		 * 
		 * PanelService のデバッグログを出すためのプロパティを設定
		 * (PanelServiceはデフォルトではデバッグログ出力しません)
		 */
		System.setProperty("jp.co.ricoh.dsdk.opt.auth.ccm.debug", "1");
		
		try {
			/*
			 * registers a CardService
			 * カードサービス登録
			 */
			new CardServiceRegister().add(MockCardService.class);
			
			/*
			 * obtains a CcmService object
			 * CCMサービス取得
			 */
			ccmService = new Ccm().getService();
		
			/*
			 * creates a CardManager object
			 * カードマネージャ生成
			 */
			cardManager = new CardManager();
			
			/*
			 * adds a CardEventListener
			 * カードイベントリスナ追加
			 */
			cardManager.addCardListener(
					cardEventListener = 
						new CardEventListener() {
							public void inserted(CardEvent e) {
								cardInserted(e);
							}
							public void removed(CardEvent e) {
								cardRemoved(e);
							}
						}
					);
			
			/*
			 * sets a CcmRequestHandler
			 * CCMリクエストハンドラ設定
			 */
			ccmService.setRequestHandler(
					ccmReqHandler = 
						new CcmRequestHandler() {
							public void handle(CcmRequest[] req) 
								throws UnsupportedRequestException, 
								        InvalidPINException, 
								        IOException {
								ccmRequest(req);
							}
						}
					);
			
		/*
		 * if a CardManager initialization failed
		 * カードマネージャ初期化エラーの場合
		 */
		} catch (InitializeException e) {
			e.printStackTrace();
			release();
			throw e;
			
		/*
		 * if a CcmRequestHandler already set
		 * リクエストハンドラがすでに設定されていた場合
		 */
		} catch (AlreadySetException e) {
			e.printStackTrace();
			release();
			throw e;
			
		/*
		 * other error
		 * その他エラー
		 */
		} catch (Exception e) {
			e.printStackTrace();
			release();
			throw e;
		}
	}
	
	/**
	 * Stops bundle.
	 * 
	 * バンドル停止。
	 */
	public void stop(BundleContext context) {
		release();
	}
	
	/**
	 * Releases resources.
	 * 
	 * リソース解放。
	 */
	synchronized private void release() {
		/*
		 * removes the CardService
		 * カードサービス削除
		 */
		try {
			new CardServiceRegister().remove(MockCardService.class);
		} catch (CardServiceRegistryException e) {}
		
		/*
		 * removes the CcmRequestHandler
		 * CCMリクエストハンドラ削除
		 */
		if (ccmReqHandler != null) {
			ccmService.removeRequestHandler(ccmReqHandler);
			ccmReqHandler = null;
		}
		
		/*
		 * removes the CardEventListener
		 * カードイベントリスナ削除
		 */
		if (cardEventListener != null) {
			cardManager.removeCardListener(cardEventListener);
			cardEventListener = null;
		}
		
		/*
		 * requests to logout if a card exists in the reader
		 * カードがセットされていた場合はログアウト要求を行う
		 */
		if (slot!=null) {
			
			slot=null;
			
			// ログアウトリクエスト
			ccmService.requestLogout();
		}
	}
	/**
	 * The process followed after a card is inserted to the reader.
	 * 
	 * カードがセットされた時の処理。
	 */
	synchronized private void cardInserted(CardEvent e) {
		if (slot==null) {			
			/*
			 * obtains the slot
			 * スロット取得
			 */
			slot = e.getSlot();
			
			/*
			 * requests to login
			 * ログイン要求
			 */
			ccmService.requestLogin();
		}
	}
	/**
	 * The process followed after the card is removed from the reader.
	 * 
	 * カードがはずされた時の処理。
	 */
	synchronized private void cardRemoved(CardEvent e) {
		if (slot!=null) {
			if (slot.equals(e.getSlot())) {
				slot = null;
			
				/*
				 * requests to logout
				 * ログアウト要求
				 */
				ccmService.requestLogout();
			}
		}
	}
	/**
	 * The process followed after receiving a/some CcmRequest(s) from the CCM.
	 * 
	 * CCMからリクエストがきた時の処理
	 */
	synchronized private void ccmRequest(CcmRequest[] req) 
	throws UnsupportedRequestException, InvalidPINException, 
			CardAccessException, VerificationException, CardServiceException {
		
		Card card = null;
		Name name = null;
		VerifyPIN pin = null;
		Password password = null;
		
		// リクエスト振分け
		for (int i = 0; i < req.length; i++) {
			
			/*
			 * Name request
			 * 名前要求
			 */
			if (req[i] instanceof Name) {               
				name = (Name) req[i];
				
			/*
			 * Password request
			 * パスワード要求
			 */
			} else if (req[i] instanceof Password) {  
				password = (Password) req[i];
			
			/*
			 * VerifyPIN request
			 * PIN認証要求
			 */
			} else if (req[i] instanceof VerifyPIN) {  
				pin = (VerifyPIN) req[i];
				
			/*
			 * Unsupported request
			 * 非サポート要求
			 */
			} else {
				throw new UnsupportedRequestException("Unsupported " + req[i]);
			}
		}
		
		try{
			/*
			 * obtains the card
			 * カード取得
			 */
			card = slot.getCard(); 
			
			/*
			 * obtains a mutex lock of the card
			 * カードロック
			 */
			card.lock();
			
			/*
			 * obtains a card service
			 * カードサービス取得
			 */
			MockCardService service = 
				(MockCardService) card.getCardService(MockCardService.class);
			
			/*
			 * if a pin verification is requested
			 * PIN認証要求がある場合
			 */
			if (pin != null) {
				service.verifyPIN(pin.getPIN());
			}
			
			/*
			 * if a name is requested
			 * 名前要求がある場合
			 */
			if (name != null) {
				name.setName(service.getUser());
			}
			
			/*
			 * if a password is requested
			 * パスワード要求がある場合
			 */
			if (password != null) {
				password.setPassword(service.getPassword());
			}
			
		/* 
		 * if the PIN verification failed
		 * PIN認証エラーの場合
		 */
		} catch (CHVException e) {
			throw new InvalidPINException(e.getMessage());
		
		} finally {
			/*
			 * releases the mutex lock of the card
			 * カード解放
			 */
			card.unlock();
		}
	}
}
