#!/bin/bash
#
usage () {
	echo $@
	echo "usage: $0 [<config>] <interface> [-o <options>]"
	echo ""
	echo "Options are:"
	echo "    debug    : be verbose"
	echo "    rc       : indicates that we are called from rcnetwork"
	echo ""
	echo "Any another options are ignored"
	exit $R_USAGE
}

xm_cmd="xm"
pidof -x /usr/sbin/xend >/dev/null 2>&1 || xm_cmd="xl"

######################################################################
# change the working direcory and source some common files
#
R_INTERNAL=1      # internal error, e.g. no config or missing scripts
cd /etc/sysconfig/network || exit $R_INTERNAL
test -f ./config && . ./config
test -f scripts/functions && . scripts/functions || exit $R_INTERNAL

######################################################################
# check arguments and how we are called (in case of links)
#
SCRIPTNAME=${0}
debug $*
case $1 in ""|-h|*help*) usage ;; esac
CONFIG="$1"
shift
if [ "x$1" != x -a "x$1" != "x-o" ] ; then
        INTERFACE="$1"
else
        INTERFACE="$CONFIG"
fi
shift
test "x$1" = "x-o" && shift
DEBUG=no
RUN_FROM_RC=no
while [ $# -gt 0 ]; do
	case $1 in
		debug)       DEBUG=yes ;;
		rc)          RUN_FROM_RC=yes ;;
		*)           debug unknown option $1 ;;
	esac
	shift
done

# usage: ifprint <err_mesg|mesg|...> message....
ifprint() {
	func=$1 ; shift
	test "x$func" = x && return 1
	if [ "$RUN_FROM_RC" = yes -a "$INTERFACE" != all ] ; then
		$func "`printf "    %-9s " "$INTERFACE"`$*"
	else
		$func "$*"
	fi
}

#
# xen related code
#

# check if xen is running
is_xend_running() {
	test -x /etc/init.d/xend && \
	/etc/init.d/xend status &>/dev/null && return 0
	return 1
}
exit_if_xend_not_running() {
	is_xend_running || {
		debug "$0: xend is not running - nothing to do"
		exit 0
	}
}	

# (modified) functions from /etc/init.d/xendomains
parseln()
{
	name=${1:0:$((${#1}-36))}
	name=${name%% *}
	rest="${1: -36}"
	id=${rest:0:4}
	id=`echo $id`
	mem=${rest:4:6}
	mem=`echo $mem`
	vcpu=${rest:10:6}
	vcpu=`echo $vcpu`
	state=${rest:16:11}
	state=`echo $state`
	tm=${rest:27}
	tm=`echo $tm`
}

xm_list()
{
	TERM=vt100 ${xm_cmd} list | grep -v '^Name *ID'
}

# For the specified vm, return a list of vifs that are connected to $INTERFACE
list_vifs()
{
	id=$1
	vifs=()
	for vif in $(ls -1 "/sys/class/net/$INTERFACE/brif/" 2>/dev/null) ; do
		eval BRIDGE_PORTS="" `grep "^[[:space:]]*BRIDGE_PORTS=" \
			"/etc/sysconfig/network/ifcfg-$INTERFACE" 2>/dev/null`
		for p in $BRIDGE_PORTS ; do
			test "x$p" = "x$vif" && continue 2
		done
		case $vif in
		(tap${id}\.*|vif${id}\.*)
			vifs=(${vifs[@]} ${vif})
		;;
		esac
	done
	echo "${vifs[@]}"
}

# Write list of concerned vifs to state file
save_sysconfig_state()
{
	[ -d "${RUN_FILES_BASE}/xen/" ] || \
	mkdir -p "${RUN_FILES_BASE}/xen/" || return 1

	rm -f "${RUN_FILES_BASE}/xen/$INTERFACE" && {
		echo "VIFS='${vifs[@]}'"
	} > "${RUN_FILES_BASE}/xen/$INTERFACE"
}

case $SCRIPTNAME in
*if-up.d*)
	exit_if_xend_not_running

	if test -f "${RUN_FILES_BASE}/xen/$INTERFACE" ; then
		. "${RUN_FILES_BASE}/xen/$INTERFACE"

		for vif in ${VIFS}; do
			test -d "/sys/class/net/${vif}" || continue
			test -d "/sys/class/net/${INTERFACE}/brif/${vif}" && \
				continue
			if ! is_iface_up ${vif} ; then
				ip link set dev ${vif} up || continue
			fi
			brctl addif ${INTERFACE} ${vif} &>/dev/null
		done

		# remove sysconfig state
		rm -f "${RUN_FILES_BASE}/xen/$INTERFACE"
	fi
;;
*if-down.d*)
	exit_if_xend_not_running
        test -d "/sys/class/net/$INTERFACE/brif/" || exit 0

	# Remember vifs attached to $INTERFACE
	vifs=()
	num=0
	while read LN; do
		parseln "$LN"
		[ "$id" = 0 ] && continue
		[ -z "$state" ] && continue

		vifs=(${vifs[@]} $(list_vifs $id))
	done < <(xm_list)

	[ -z "${vifs[*]}" ] || save_sysconfig_state

;;
*)
	usage
;;
esac

