/* WS4D-gSOAP - Implementation of the Devices Profile for Web Services
 * (DPWS) on top of gSOAP
 * Copyright (C) 2007 University of Rostock
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301 USA
 */

#include "ws4d_misc.h"
#ifndef WIN32
#include <sys/time.h>
#endif
#include <time.h>

/**
 * Function extracts a string of a speficic language from an array of struct wdp__LocalizedStringType
 *
 * @param string array of structure wdp__LocalizedStringType to
 * extract string from
 * @param size number of elements in array
 * @param lang language to extract
 *
 * @return pointer to extractet string on success, NULL otherwise
 */
const char *
ws4d_locstring_get (struct ws4d_locstring *string, int size, const char *lang)
{
  int i = 0;
  const char *result = NULL;

  /* test parameters */
  ws4d_assert (string && (size > 0) && lang, NULL);

  for (i = 0; i < size; i++)
    {
      if (string[i].lang && !STRCASECMP (string[i].lang, lang))
        {
          result = string[i].string;
          break;
        }
    }

  return result;
}

struct ws4d_locstring *
ws4d_locstring_dup (struct ws4d_locstring *string, int size,
                    struct ws4d_abs_allocator *alist)
{
  struct ws4d_locstring *res;
  int i, err = WS4D_OK;

  /* test parameters */
  ws4d_assert (string && (size > 0) && alist, NULL);

  res = ws4d_malloc_alist (sizeof (struct ws4d_locstring) * size, alist);
  for (i = 0; i < size; i++)
    {
      if (string[i].lang && string[i].string)
        {
          res[i].lang = ws4d_strdup (string[i].lang, alist);
          res[i].string = ws4d_strdup (string[i].string, alist);
        }
      else
        {
          err = WS4D_ERR;
        }
    }

  if (err != WS4D_OK)
    {
      for (; i > 0; i--)
        {
          if (string[i].lang && string[i].string)
            {
              ws4d_free_alist ((void *) res[i].lang);
              ws4d_free_alist ((void *) res[i].string);
            }
        }
    }

  return res;
}

void
ws4d_locstring_free (struct ws4d_locstring *string, int size)
{
  int i;

  ws4d_assert (string && (size > 0),);

  for (i = 0; i < size; i++)
    {
      if (string[i].lang)
        {
          ws4d_free_alist ((void *) string[i].lang);
        }

      if (string[i].string)
        {
          ws4d_free_alist ((void *) string[i].string);
        }
    }
  ws4d_free_alist (string);
}

int
ws4d_dur_to_s (struct ws4d_dur *dur, ws4d_time * s)
{
  if (dur && s)
    {
      *s = ((946080000 * dur->years) +
            (2592000 * dur->months) +
            (86400 * dur->days) + (3600 * dur->hours) +
            (60 * dur->minutes) + dur->seconds);

      return WS4D_OK;
    }
  else
    {
      return WS4D_ERR;
    }
}

int
ws4d_s_to_dur (ws4d_time s, struct ws4d_dur *dur)
{
  if (dur)
    {
      dur->years = s / 946080000;
      s = s % 946080000;
      dur->months = s / 2592000;
      s = s % 2592000;
      dur->days = s / 86400;
      s = s % 86400;
      dur->hours = s / 3600;
      s = s % 3600;
      dur->minutes = s / 60;
      s = s % 60;
      dur->seconds = s;

      return WS4D_OK;
    }
  else
    {
      return WS4D_ERR;
    }
}

int
ws4d_xsddt_to_dur (const char *xsddt, struct ws4d_dur *dur)
{
  char *period = NULL;
  char *dur_time = NULL;

  ws4d_assert (xsddt && dur, WS4D_ERR);

  period = strchr (xsddt, 'P');
  dur_time = strchr (xsddt, 'T');

  if (period)
    {
      int cur = 0, last = 0;

      period++;

      while ((period[cur] != '\0') && (period[cur] != 'T'))
        {
          if (isdigit (period[cur]))
            {
              cur++;
            }
          else if (period[cur] == 'Y')
            {
              period[cur] = '\0';
              dur->years = atoi (period + last);
              cur++;
              last = cur;
            }
          else if (period[cur] == 'M')
            {
              period[cur] = '\0';
              dur->months = atoi (period + last);
              cur++;
              last = cur;
            }
          else if (period[cur] == 'D')
            {
              period[cur] = '\0';
              dur->days = atoi (period + last);
              cur++;
              last = cur;
            }
          else
            {
              /* parse error */
              return WS4D_ERR;
            }
        }
    }
  if (dur_time)
    {
      int cur = 0, last = 0;

      dur_time++;

      while (dur_time[cur] != '\0')
        {
          if (isdigit (dur_time[cur]))
            {
              cur++;
            }
          else if (dur_time[cur] == 'H')
            {
              dur_time[cur] = '\0';
              dur->years = atoi (dur_time + last);
              cur++;
              last = cur;
            }
          else if (dur_time[cur] == 'M')
            {
              dur_time[cur] = '\0';
              dur->months = atoi (dur_time + last);
              cur++;
              last = cur;
            }
          else if (dur_time[cur] == 'S')
            {
              dur_time[cur] = '\0';
              dur->days = atoi (dur_time + last);
              cur++;
              last = cur;
            }
          else
            {
              /* parse error */
              return WS4D_ERR;
            }
        }
    }

  return WS4D_OK;
}

int
ws4d_dur_to_xsddt (struct ws4d_dur *dur, char *buf, int buf_len)
{
  int ret;

  ws4d_assert (dur && buf && (buf_len > 0), WS4D_ERR);

  ret = SNPRINTF (buf, buf_len,
                  "P%dY%dM%dDT%dH%dM%dS", dur->years,
                  dur->months, dur->days, dur->hours, dur->minutes,
                  dur->seconds);

  if (ret > buf_len)
    {
      return WS4D_ERR;
    }
  else
    {
      return WS4D_OK;
    }
}

#ifndef WITH_OTHER_ARCH
ws4d_time
ws4d_systime_ms (void)
{
#ifndef WIN32
  struct timeval tv;
  gettimeofday (&tv, (struct timezone *) 0);

  return (ws4d_time) (tv.tv_sec * 1000) + (tv.tv_usec / 1000);
#else
  return (ws4d_time) GetTickCount ();
#endif
}

ws4d_time
ws4d_systime_s (void)
{
#ifndef WIN32
  struct timeval tv;
  gettimeofday (&tv, (struct timezone *) 0);

  return (ws4d_time) tv.tv_sec;
#else
  return (ws4d_time) GetTickCount () / 1000;
#endif
}
#endif

const char *ws4d_uuid_schema_prefix = "urn:uuid:";
const char *ws4d_uuid_schema_format = "urn:uuid:%s";

static unsigned char
ws4d_generaterandhex ()
{
  int r = rand () % 15;
  return (r > 9) ? (r - 9) + 'a' : (r) + '0';
}

void
ws4d_uuid_generate_random (char *out)
{
  register int i;
  char uuid_dig19_table[4] = { '8', '9', 'a', 'b' };

  /* test parameters */
  ws4d_assert (out,);

  /* random seed */
  srand ((unsigned int) ws4d_systime_ms ());

  /* set random hex for specific digits */
  for (i = 0; i < (WS4D_UUID_SIZE - 1); i++)
    {
      if ((i != 8) && (i != 13) && (i != 14) && (i != 18) && (i != 19)
          && (i != 23))
        {
          out[i] = ws4d_generaterandhex ();
        }
    }

  /* set dashes and digits for remaining parts */
  out[8] = '-';
  out[13] = '-';
  out[14] = '4';
  out[18] = '-';
  out[19] = uuid_dig19_table[rand () % 3];
  out[23] = '-';

  /* terminate string */
  out[WS4D_UUID_SIZE - 1] = '\n';

  return;
}


/*
** Translation Table as described in RFC1113
*/
static const char cb64[] =
  "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/";

/* maybe, http://devenix.wordpress.com/2008/01/18/howto-base64-encode-and-decode-with-c-and-openssl-2/
 * gives a lead to how to do base64-de/encoding with OpenSSL... if this is preferable...
 */

/*
** encodeblock
**
** encode 3 8-bit binary bytes as 4 '6-bit' characters
*/
void
encodeblock (unsigned char in[3], unsigned char out[4], int len)
{
  out[0] = cb64[in[0] >> 2];
  out[1] = cb64[((in[0] & 0x03) << 4) | ((in[1] & 0xf0) >> 4)];
  out[2] =
    (unsigned char) (len >
                     1 ? cb64[((in[1] & 0x0f) << 2) | ((in[2] & 0xc0) >> 6)] :
                     '=');
  out[3] = (unsigned char) (len > 2 ? cb64[in[2] & 0x3f] : '=');
}

/**
 * Function encodes a binary array to base64
 *
 * @param pointer to unsigned char array containing binary data
 *
 * @return pointer to created array containing base64-coded data
 */
unsigned char *
ws4d_bin2b64 (unsigned char *binary)
{
  unsigned char in[3], out[4];
  int loop = 0, i, len, blocksout = 0;
  unsigned char *base64 = NULL;

  while (binary[loop * 3])
    {
      in[0] = in[1] = in[2] = '\0';
      len = 0;
      while (len < 3 && binary[3 * loop + len] != '\0')
        {
          in[len] = binary[loop * 3 + len];
          len++;
        }

      if (len)
        {
          encodeblock (in, out, len);
          base64 =
            (unsigned char *) realloc (base64,
                                       (loop +
                                        1) * 4 * sizeof (unsigned char));
          for (i = 0; i < 4; i++)
            {
              base64[4 * loop + i] = out[i];
            }
          blocksout++;
        }

      loop++;

      if (len != 3)
        break;

    }
  base64 = (unsigned char *) realloc (base64, 4 * loop + 1);
  base64[4 * loop] = (unsigned char) 0;
  return base64;
}

/*
** decodeblock
**
** decode 4 '6-bit' characters into 3 8-bit binary bytes
*/
void
decodeblock (unsigned char in[4], unsigned char out[3])
{
  out[0] = (unsigned char) (in[0] << 2 | in[1] >> 4);
  out[1] = (unsigned char) (in[1] << 4 | in[2] >> 2);
  out[2] = (unsigned char) (((in[2] << 6) & 0xc0) | in[3]);
}

/**
 * Function decodes a base64-coded data
 *
 * @param pointer to unsigned char array containing base64-coded data
 *
 * @return pointer to created array containing binary data
 */
unsigned char *
ws4d_b642bin (unsigned char *base64)
{
  unsigned char in[4], out[3], v;
  int i, len, loop = 0;
  unsigned char *binary = (unsigned char *) NULL;

  /* if length is not a multiple of 4, base64 does not contain a valid
   * base64-coded string. So, in this case, we just return NULL
   */
  for (len = 0; base64[len]; len++);
  if ((int) (len % 4))
    return (unsigned char *) NULL;

  binary = (unsigned char *) malloc (len / 4 * 3 + 1);
  binary[len / 4 * 3] = '\0';   /* so most likely we use too many bytes... */

  /* further, we assume, we can only get 'clean' base64 - so no error-checks */
  while (base64[4 * loop] != '\0')
    {
      for (i = 0; i < 4; i++)
        {
          v = base64[loop * 4 + i];
          if (v >= 'A' && v <= 'Z')
            in[i] = (unsigned char) v - 'A';
          else if (v >= 'a' && v <= 'z')
            in[i] = (unsigned char) v - 71;
          else if (v >= '0' && v <= '9')
            in[i] = (unsigned char) v + 4;
          else if (v == '+')
            in[i] = (unsigned char) 62;
          else if (v == '/')
            in[i] = (unsigned char) 63;
          else if (v == '=')
            in[i] = 0;
          else                  /* Something went wrong */
            return (unsigned char *) NULL;
        }

      decodeblock (in, out);
      for (i = 0; i < 3; i++)
        {
          binary[3 * loop + i] = out[i];
        }
      loop++;
    }
  return binary;
}


#ifndef HAVE_INET_PTON_H
int
ws4d_inet_pton (int af, const char *src, void *dst)
{
#ifdef WIN32
  unsigned long *in4 = NULL;
#else
  struct in_addr *in4 = NULL;
#endif

  ws4d_assert (src && dst, -1);

  switch (af)
    {
    case AF_INET:
      in4 = dst;
      *in4 = inet_addr (src);
      return *in4 != INADDR_NONE;
      break;
    default:
      return -1;
    }
}
#endif

#ifndef HAVE_INET_NTOP_H
const char *
ws4d_inet_ntop (int af, const void *src, char *dst, size_t size)
{
  char *result;
#ifdef WIN32
  struct in_addr *in4 = (void *) src;
#else
  in_addr_t *in4 = src;
#endif

  ws4d_assert (src && dst && (size > 0), NULL);

  switch (af)
    {
    case AF_INET:
      result = inet_ntoa (*in4);
      if (result != NULL)
        {
          if (strlen (result) < size)
            {
              strncpy (dst, result, size);
              return dst;
            }
        }
      return NULL;
      break;
    default:
      return NULL;
    }
}
#endif
