/* WS4D-gSOAP - Implementation of the Devices Profile for Web Services
 * (DPWS) on top of gSOAP
 * Copyright (C) 2007 University of Rostock
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301 USA
 */

#include "ws4d_misc.h"
#include "ws4d_service.h"
#include "ws4d_hostingservice.h"

/*******
 * Hosted Service Logic
 *
 */

#define WS4D_HOSTEDEP_ID "WS4D-HostedEP-0.1"

const char *ws4d_hostedep_plugin_id = WS4D_HOSTEDEP_ID;

#define WS4D_HOSTEDEP_INACTIVE (0)
#define WS4D_HOSTEDEP_ACTIVE (1)

struct ws4d_hostedep_plugin_s
{
  int status;
  const char *wsdl;
  char *__any;
  void *transport_data;
  struct ws4d_abs_allocator alist;
};


INLINE struct ws4d_hostedep_plugin_s
  *ws4d_hostedep_plugindata (struct ws4d_epr *epr);

int ws4d_hostedep_fcopy (struct ws4d_epr *epr, struct ws4d_epr_plugin *dst,
                         struct ws4d_epr_plugin *src);

void ws4d_hostedep_fdelete (struct ws4d_epr *epr, struct ws4d_epr_plugin *p);

int ws4d_hostedep_fcreate (struct ws4d_epr *epr,
                           struct ws4d_epr_plugin *plugin, void *arg);

int ws4d_hostedep_plugin (struct ws4d_epr *epr);

static struct ws4d_epr_plugin_cb ws4d_hostedep_callbacks =
  { ws4d_hostedep_fdelete, ws4d_hostedep_fcopy, NULL };

INLINE struct ws4d_hostedep_plugin_s *
ws4d_hostedep_plugindata (struct ws4d_epr *epr)
{

  return (struct ws4d_hostedep_plugin_s *)
    ws4d_epr_get_plugindata (epr, ws4d_hostedep_plugin_id);

}

int
ws4d_hostedep_fcopy (struct ws4d_epr *epr, struct ws4d_epr_plugin *dst,
                     struct ws4d_epr_plugin *src)
{
  struct ws4d_hostedep_plugin_s *dst_data = NULL, *src_data = NULL;
  int err;

  src_data = (struct ws4d_hostedep_plugin_s *) src->data;

  /* test parameters */
  ws4d_assert (src_data, WS4D_EPARAM);

  /* set plugin structurr */
  dst->id = ws4d_hostedep_plugin_id;
  dst->cb = &ws4d_hostedep_callbacks;

  /* check if plugin data is already there */
  if (dst->data)
    {
      dst_data = dst->data;
    }
  else
    {
      dst_data = ws4d_malloc (sizeof (struct ws4d_hostedep_plugin_s));
      ws4d_fail (!dst_data, WS4D_EOM);

      dst_data->status = WS4D_HOSTEDEP_INACTIVE;

      WS4D_ALLOCLIST_INIT (&dst_data->alist);
    }

  /* update status */
  dst_data->status = src_data->status;

  /* update wsdl */
  err =
    ws4d_strdup2 (src_data->wsdl, (const char **) &dst_data->wsdl,
                  ws4d_epr_get_alist (epr));
  ws4d_fail (err != WS4D_OK, err);

  /* update extensions */
  err =
    ws4d_strdup2 (src_data->__any, (const char **) &dst_data->__any,
                  ws4d_epr_get_alist (epr));
  ws4d_fail (err != WS4D_OK, err);

  /* update transport data */
  dst_data->transport_data = src_data->transport_data;

  dst->data = dst_data;

  return WS4D_OK;
}

void
ws4d_hostedep_fdelete (struct ws4d_epr *epr, struct ws4d_epr_plugin *p)
{
  struct ws4d_hostedep_plugin_s *plugin = p->data;

  WS4D_UNUSED_PARAM (epr);

  if (plugin)
    {
      ws4d_alloclist_done (&plugin->alist);
      ws4d_free (plugin);
    }
}

int
ws4d_hostedep_fcreate (struct ws4d_epr *epr,
                       struct ws4d_epr_plugin *plugin, void *arg)
{
  struct ws4d_hostedep_plugin_s *data = NULL;

  WS4D_UNUSED_PARAM (epr);
  WS4D_UNUSED_PARAM (arg);

  plugin->id = ws4d_hostedep_plugin_id;
  plugin->cb = &ws4d_hostedep_callbacks;

  data = ws4d_malloc (sizeof (struct ws4d_hostedep_plugin_s));
  if (!data)
    {
      return WS4D_EOM;
    }

  data->status = WS4D_HOSTEDEP_INACTIVE;

  WS4D_ALLOCLIST_INIT (&data->alist);

  plugin->data = data;

  return WS4D_OK;
}

int
ws4d_hostedep_plugin (struct ws4d_epr *epr)
{
  struct ws4d_hostedep_plugin_s *plugin = ws4d_hostedep_plugindata (epr);

  /* check if plugin is already registered */
  if (plugin)
    {
      return WS4D_OK;
    }

  return ws4d_epr_register_plugin (epr, ws4d_hostedep_fcreate, NULL);
}

int
ws4d_hosted_activate (struct ws4d_epr *hostedservice)
{
  struct ws4d_hostedep_plugin_s *plugin =
    ws4d_hostedep_plugindata (hostedservice);

  /* test parameters */
  ws4d_assert (plugin
               && (plugin->status == WS4D_HOSTEDEP_INACTIVE), WS4D_EPARAM);

  plugin->status = WS4D_HOSTEDEP_ACTIVE;
  return WS4D_OK;
}

int
ws4d_hosted_deactivate (struct ws4d_epr *hostedservice)
{
  struct ws4d_hostedep_plugin_s *plugin =
    ws4d_hostedep_plugindata (hostedservice);

  /* test parameters */
  ws4d_assert (plugin
               && (plugin->status == WS4D_HOSTEDEP_ACTIVE), WS4D_EPARAM);

  plugin->status = WS4D_HOSTEDEP_INACTIVE;
  return WS4D_OK;
}

int
ws4d_hosted_isactive (struct ws4d_epr *hostedservice)
{
  struct ws4d_hostedep_plugin_s *plugin =
    ws4d_hostedep_plugindata (hostedservice);

  ws4d_assert (plugin, 0);

  return plugin->status == WS4D_HOSTEDEP_ACTIVE;
}

int
ws4d_hosted_set_wsdl (struct ws4d_epr *hostedservice, const char *wsdl)
{
  struct ws4d_hostedep_plugin_s *plugin =
    ws4d_hostedep_plugindata (hostedservice);

  /* test parameters */
  ws4d_assert (plugin, WS4D_EPARAM);

  return ws4d_strdup2 (wsdl, (const char **) &plugin->wsdl,
                       ws4d_epr_get_alist (hostedservice));
}

const char *
ws4d_hosted_get_wsdl (struct ws4d_epr *hostedservice)
{
  struct ws4d_hostedep_plugin_s *plugin =
    ws4d_hostedep_plugindata (hostedservice);

  /* test parameters */
  ws4d_assert (plugin, NULL);

  return plugin->wsdl;
}

int
ws4d_hosted_set_ext (struct ws4d_epr *hostedservice, const char *ext)
{
  struct ws4d_hostedep_plugin_s *plugin =
    ws4d_hostedep_plugindata (hostedservice);

  /* test parameters */
  ws4d_assert (plugin && ext, WS4D_EPARAM);

  /* update extension */
  return ws4d_strdup2 (ext, (const char **) &plugin->__any,
                       ws4d_epr_get_alist (hostedservice));
}

const char *
ws4d_hosted_get_ext (struct ws4d_epr *hostedservice)
{
  struct ws4d_hostedep_plugin_s *plugin =
    ws4d_hostedep_plugindata (hostedservice);

  /* test parameters */
  ws4d_assert (plugin, NULL);

  /* return ext string */
  return plugin->__any;
}

int
ws4d_hosted_set_transportdata (struct ws4d_epr *hostedservice,
                               void *transport_data)
{
  struct ws4d_hostedep_plugin_s *plugin =
    ws4d_hostedep_plugindata (hostedservice);

  /* test parameters */
  ws4d_assert (plugin, WS4D_EPARAM);

  plugin->transport_data = transport_data;

  return WS4D_OK;
}

void *
ws4d_hosted_get_transportdata (struct ws4d_epr *hostedservice)
{
  struct ws4d_hostedep_plugin_s *plugin =
    ws4d_hostedep_plugindata (hostedservice);

  /* test parameters */
  ws4d_assert (plugin, NULL);

  return plugin->transport_data;
}

/*
 * Hosting Service
 */

#include "ws4d_hostingservice.h"
#include "ws4d_abstract_eprlist.h"
#include "ws4d_eprllist.h"

int
ws4d_hosting_init (struct ws4d_hostingservice *hs,
                   struct ws4d_stringlist *netdevs)
{
  /* test parameters */
  ws4d_assert (hs && netdevs, WS4D_EPARAM);

  memset (hs, 0, sizeof (struct ws4d_hostingservice));
  WS4D_ALLOCLIST_INIT (&hs->alist);

  hs->change_tracker = WS4D_HS_UNCHANGED;

  ws4d_stringlist_init2 (&hs->hosts, ws4d_stringlist_tostring (netdevs));

  ws4d_mutex_init (&hs->lock);

  return ws4d_eprlist_init (&hs->services, ws4d_eprllist_init, NULL);
}

int
ws4d_hosting_done (struct ws4d_hostingservice *hs)
{
  int ret;

  /* test parameters */
  ws4d_assert (hs, WS4D_EPARAM);

  ret = ws4d_eprlist_clear (&hs->services);
  ws4d_fail (ret != WS4D_OK, ret);

  ws4d_stringlist_done (&hs->hosts);
  ws4d_eprlist_done (&hs->services);
  ws4d_alloclist_done (&hs->alist);

  ws4d_mutex_destroy (&hs->lock);

  return WS4D_OK;
}

int
ws4d_hosting_change_interf (struct ws4d_hostingservice *hs, const char *hosts)
{
  /* test parameters */
  ws4d_assert (hs && hosts, WS4D_EPARAM);

  /* update host string */
  ws4d_stringlist_clear (&hs->hosts);
  ws4d_stringlist_add (&hs->hosts, hosts);

  /* TODO: should trigger some action here */

  hs->change_tracker |= WS4D_HS_RELCHANGED;

  return WS4D_OK;
}

struct ws4d_epr *
ws4d_hosting_add_service (struct ws4d_hostingservice *hs,
                          const char *serviceid)
{
  int err;
  struct ws4d_epr *service = NULL;

  /* test parameters */
  ws4d_assert (hs && serviceid, NULL);

  /* avoid duplicates */
  service = ws4d_hosting_get_service (hs, serviceid);
  ws4d_fail (service, NULL);

  service = ws4d_eprlist_alloc (&hs->services);
  ws4d_fail (!service, NULL);

  err = ws4d_register_serviceep (service);
  if (err != WS4D_OK)
    {
      ws4d_eprlist_free (&hs->services, service);
      return NULL;
    }

  err = ws4d_hostedep_plugin (service);
  if (err != WS4D_OK)
    {
      ws4d_eprlist_free (&hs->services, service);
      return NULL;
    }

  err = ws4d_serviceep_setid (service, serviceid);
  if (err != WS4D_OK)
    {
      ws4d_eprlist_free (&hs->services, service);
      return NULL;
    }

  err = ws4d_eprlist_add (&hs->services, service);
  if (err != WS4D_OK)
    {
      ws4d_eprlist_free (&hs->services, service);
      return NULL;
    }

  return service;
}

int
ws4d_hosting_bind_service (struct ws4d_hostingservice *hs,
                           struct ws4d_epr *service, char *uri, size_t size)
{
  /* test parameters */
  ws4d_assert (hs && service && uri && (size > 0), WS4D_EPARAM);

  /* check if service is correctly initialized */
  ws4d_fail (ws4d_serviceep_getid (service) == NULL, WS4D_ERR);

  /* check if service was already bound */
  if (ws4d_epr_get_Addrs (service) != NULL)
    {
      /* fail if addresses don't match */
      ws4d_fail (ws4d_epr_compare_Addrs (service, uri) != 1, WS4D_ERR);

      return WS4D_OK;
    }

  /* asign addresses */
  return ws4d_epr_set_Addrs (service, uri);
}

int
ws4d_hosting_activate_service (struct ws4d_hostingservice *hs,
                               struct ws4d_epr *service, char *uri,
                               size_t size)
{
  int err;

  /* test parameters */
  ws4d_assert (hs && service && !ws4d_hosted_isactive (service), WS4D_EPARAM);

  /* bind service to uri */
  err = ws4d_hosting_bind_service (hs, service, uri, size);
  ws4d_fail (err != WS4D_OK, err);

  /* activate service */
  err = ws4d_hosted_activate (service);
  ws4d_fail (err != WS4D_OK, err);

  /* mark as changed */
  hs->change_tracker |= WS4D_HS_RELCHANGED;

  return WS4D_OK;
}

int
ws4d_hosting_deactivate_service (struct ws4d_hostingservice *hs,
                                 struct ws4d_epr *service)
{
  int err;

  /* test parameters */
  ws4d_assert (hs && service && ws4d_hosted_isactive (service), WS4D_EPARAM);

  err = ws4d_hosted_deactivate (service);
  ws4d_fail (err != WS4D_OK, err);

  hs->change_tracker |= WS4D_HS_RELCHANGED;

  return WS4D_OK;
}

int
ws4d_hosting_remove_service (struct ws4d_hostingservice *hs,
                             struct ws4d_epr *service)
{
  int ret;

  /* test parameters */
  ws4d_assert (hs && service, WS4D_EPARAM);

  if (ws4d_hosted_isactive (service))
    {
      ws4d_hosting_deactivate_service (hs, service);
    }

  ret = ws4d_eprlist_remove (&hs->services, service);
  ws4d_fail (ret != WS4D_OK, ret);

  return ws4d_eprlist_free (&hs->services, service);
}

struct ws4d_epr *
ws4d_hosting_get_service (struct ws4d_hostingservice *hs,
                          const char *serviceid)
{
  register struct ws4d_epr *epr, *iter;

  /* test parameters */
  ws4d_assert (hs && serviceid, NULL);

  ws4d_eprlist_foreach (epr, iter, &hs->services)
  {
    const char *id = ws4d_serviceep_getid (epr);

    if (id && (serviceid == id))
      {
        return epr;
      }
  }

  ws4d_eprlist_foreach (epr, iter, &hs->services)
  {
    const char *id = ws4d_serviceep_getid (epr);

    if (id && !strcmp (serviceid, id))
      {
        return epr;
      }
  }

  return NULL;
}

int
ws4d_hosting_get_activeservices (struct ws4d_hostingservice *hs,
                                 struct ws4d_abs_eprlist *list)
{
  register struct ws4d_epr *service, *iter;
  int matches = 0;

  /* test parameters */
  ws4d_assert (hs && list, matches);

  ws4d_eprlist_foreach (service, iter, &hs->services)
  {
    if (ws4d_hosted_isactive (service))
      {
        if (ws4d_eprlist_add2 (list, service) != WS4D_OK)
          {
            break;
          }

        matches++;
      }
  }

  return matches;
}

int
ws4d_hosting_get_active_services_count (struct ws4d_hostingservice *hs)
{
  register struct ws4d_epr *epr, *iter;
  register int count = 0;

  /* test parameters */
  ws4d_assert (hs, WS4D_EPARAM);

  ws4d_eprlist_foreach (epr, iter, &hs->services)
  {
    if (ws4d_hosted_isactive (epr))
      {
        count++;
      }
  }

  return count;
}

struct ws4d_relationship *
ws4d_hosting_get_relationship (struct ws4d_hostingservice *hs,
                               struct ws4d_abs_allocator *alist)
{
  struct ws4d_relationship *result;

  /* test parameters */
  ws4d_assert (hs && alist, NULL);

  result = ws4d_malloc_alist (sizeof (struct ws4d_relationship), alist);
  ws4d_fail (!result, NULL);

  result->hosted_count = ws4d_hosting_get_active_services_count (hs);
  if (result->hosted_count > 0)
    {
      struct ws4d_host *host;
      register struct ws4d_epr *epr, *iter;

      result->hosted =
        ws4d_malloc_alist (sizeof (struct ws4d_host) * result->hosted_count,
                           alist);
      if (!result->hosted)
        {
          ws4d_free_alist (result);
          return NULL;
        }

      host = result->hosted;

      ws4d_eprlist_foreach (epr, iter, &hs->services)
      {
        ws4d_epr_lock (epr);
        if (ws4d_hosted_isactive (epr))
          {
            host->Addr = ws4d_strdup (ws4d_epr_get_Addrs (epr), alist);
            host->ServiceId = ws4d_strdup (ws4d_serviceep_getid (epr), alist);
            ws4d_qnamelist_init (&host->types);
            ws4d_qnamelist_copy (ws4d_serviceep_gettypelist (epr),
                                 &host->types);
            host->__any =
              ws4d_strdup ((char *) ws4d_hosted_get_ext (epr), alist);
            host++;
          }
        ws4d_epr_unlock (epr);
      }
    }

  return result;
}

void
ws4d_hosting_free_relationship (struct ws4d_hostingservice *hs,
                                struct ws4d_relationship *relationship)
{
  int i;

  /* test parameters */
  ws4d_assert (hs && relationship,);

  for (i = 0; i < relationship->hosted_count; i++)
    {
      ws4d_free_alist (relationship->hosted[i].Addr);
      ws4d_free_alist (relationship->hosted[i].ServiceId);
      ws4d_free_alist (relationship->hosted[i].__any);
      ws4d_qnamelist_done (&relationship->hosted[i].types);
    }
  ws4d_free_alist (relationship->hosted);
  ws4d_free_alist (relationship);
}

struct ws4d_thisDevice *
ws4d_hosting_get_thisdevice (struct ws4d_hostingservice *hs)
{
  /* test parameters */
  ws4d_assert (hs, NULL);

  return &hs->device;
}

struct ws4d_thisDevice *
ws4d_hosting_change_thisdevice (struct ws4d_hostingservice *hs)
{
  /* test parameters */
  ws4d_assert (hs, NULL);

  hs->change_tracker |= WS4D_HS_DEVCHANGED;
  return &hs->device;
}

struct ws4d_thisModel *
ws4d_hosting_get_thismodel (struct ws4d_hostingservice *hs)
{
  /* test parameters */
  ws4d_assert (hs, NULL);

  return &hs->model;
}

struct ws4d_thisModel *
ws4d_hosting_change_thismodel (struct ws4d_hostingservice *hs)
{
  /* test parameters */
  ws4d_assert (hs, NULL);

  hs->change_tracker |= WS4D_HS_MODCHANGED;
  return &hs->model;
}
