/* WS4D-gSOAP - Implementation of the Devices Profile for Web Services
 * (DPWS) on top of gSOAP
 * Copyright (C) 2007 University of Rostock
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301 USA
 */

#include "stdsoap2.h"

#include <stdarg.h>

#include "dpwsH.h"
#include "soap_misc.h"

#include "ws4d_localizedstring.h"

struct wsdp__LocalizedStringType *
ws4d_locstring_tosoap (struct ws4d_locstring *string,
                       int size, struct ws4d_abs_allocator *alist)
{
  struct wsdp__LocalizedStringType *res;
  int i, err = WS4D_OK;

  /* test parameters */
  ws4d_assert (string && (size > 0) && alist, NULL);

  res =
    ws4d_malloc_alist (sizeof (struct wsdp__LocalizedStringType) * size,
                       alist);
  for (i = 0; i < size; i++)
    {
      if (string[i].lang && string[i].string)
        {
          res[i].xml__lang = ws4d_strdup (string[i].lang, alist);
          res[i].__item = ws4d_strdup (string[i].string, alist);
        }
      else
        {
          err = WS4D_ERR;
        }
    }

  if (err != WS4D_OK)
    {
      for (; i > 0; i--)
        {
          if (string[i].lang && string[i].string)
            {
              ws4d_free_alist (res[i].__item);
              ws4d_free_alist (res[i].xml__lang);
            }
        }
    }

  return res;
}

struct ws4d_locstring *
soap_locstring_tows4d (struct wsdp__LocalizedStringType *string,
                       int size, struct ws4d_abs_allocator *alist)
{
  struct ws4d_locstring *res;
  int i, err = WS4D_OK;

  /* test parameters */
  ws4d_assert (string && (size > 0) && alist, NULL);

  res = ws4d_malloc_alist (sizeof (struct ws4d_locstring) * size, alist);
  for (i = 0; i < size; i++)
    {
      if (string[i].__item)
        {
          if (string[i].xml__lang)
            {
              res[i].lang = ws4d_strdup (string[i].xml__lang, alist);
            }
          res[i].string = ws4d_strdup (string[i].__item, alist);
        }
      else
        {
          err = WS4D_ERR;
        }
    }

  if (err != WS4D_OK)
    {
      for (; i > 0; i--)
        {
          if (string[i].__item)
            {
              char *loc_string = (char *) res[i].string;

              if (string[i].xml__lang)
                {
                  char *loc_lang = (char *) res[i].lang;
                  ws4d_free_alist (loc_lang);
                }

              ws4d_free_alist (loc_string);
            }
        }
    }

  return res;
}

char *
soap_locstring_get (struct wsdp__LocalizedStringType *string,
                    int size, const char *lang)
{
  int i = 0;
  char *result = NULL;

  /* test parameters */
  ws4d_assert (string && (size > 0) && lang, NULL);

  for (i = 0; i < size; i++)
    {
      if (string[i].xml__lang && !STRCASECMP (string[i].xml__lang, lang))
        {
          result = string[i].__item;
          break;
        }
    }

  return result;
}

void
soap_locstring_free (struct wsdp__LocalizedStringType *string, int size)
{
  int i;

  ws4d_assert (string && (size > 0),);

  for (i = 0; i < size; i++)
    {
      if (string[i].xml__lang)
        {
          ws4d_free_alist (string[i].xml__lang);
        }

      if (string[i].__item)
        {
          ws4d_free_alist (string[i].__item);
        }
    }
  ws4d_free_alist (string);
}

static int
soap_ser_send (struct soap *soap, const char *buf, size_t len)
{
  if (soap->user)
    {
      strncat (soap->user, buf, len);
    }
  return SOAP_OK;
}

char *
soap_elem_to_str (void *ptr, char *tag, int type)
{
  struct soap tmp;
  soap_init (&tmp);
  soap_set_omode (&tmp, SOAP_XML_CANONICAL);
  tmp.fsend = soap_ser_send;
  soap_begin_count (&tmp);
  soap_putelement (&tmp, ptr, tag, -1, type);
  soap_end_count (&tmp);
  tmp.user = malloc (tmp.count + 1);
  memset (tmp.user, 0, tmp.count + 1);
  soap_begin_send (&tmp);
  soap_putelement (&tmp, ptr, tag, -1, type);
  soap_end_send (&tmp);
  return tmp.user;
}

struct soap_recv_buf
{
  int pos;
  size_t size;
  char *buf;
};

static size_t
soap_deser_recv (struct soap *soap, char *buf, size_t len)
{
  if (soap->user)
    {
      struct soap_recv_buf *recv_buf = soap->user;
      int ret = 0;
      if ((unsigned int) recv_buf->pos >= recv_buf->size)
        return 0;
      if (len >= (recv_buf->size - recv_buf->pos))
        {
          ret = recv_buf->size - recv_buf->pos;
        }
      else
        {
          ret = len;
        }

      memcpy (buf, recv_buf->buf + recv_buf->pos, ret);
      recv_buf->pos += ret;
      return ret;
    }
  return 0;
}

int
soap_begin_deser (struct soap *soap, char *str, size_t str_len,
                  struct Namespace *ns)
{
  struct soap_recv_buf *buf;
  int i;
  soap->frecv = soap_deser_recv;
  soap_begin_recv (soap);
  for (i = 0; ns[i].id; i++)
    {
      soap_push_namespace (soap, ns[i].id, ns[i].ns);
    }
  soap->level++;

  buf = (struct soap_recv_buf *) soap_malloc (soap,
                                              sizeof (struct soap_recv_buf));
  buf->pos = 0;
  buf->size = str_len + 1;
  buf->buf = str;
  soap->user = buf;
  return SOAP_OK;
}

int
soap_end_deser (struct soap *soap)
{
  soap->level--;
  soap_pop_namespace (soap);
  soap_end_recv (soap);
  return SOAP_OK;
}

int
soap_header_new (struct soap *soap, size_t size)
{
  if (!soap || (size < 1))
    return SOAP_ERR;

  soap->header = (struct SOAP_ENV__Header *) soap_malloc (soap, size);

  if (soap->header)
    {
      memset (soap->header, 0, size);
      return SOAP_OK;
    }
  else
    {
      return soap->error;
    }
}

struct Namespace *
soap_extend_namespaces (struct Namespace *orig_namespaces,
                        struct Namespace *extns,
                        struct ws4d_abs_allocator *alist)
{
  register struct Namespace *entry = NULL;
  struct Namespace *result = NULL, *tmp_result = NULL;
  int old_size = 0, ext_size = 0, new_size = 0, count = 0;

  /* test parameters */
  ws4d_assert (extns && orig_namespaces && alist, result);

  /* count entries in orig_namespaces */
  for (entry = orig_namespaces; entry && entry->id; entry++)
    {
      old_size++;
    }

  /* count entries in extns */
  for (entry = extns; entry && entry->id; entry++)
    {
      ext_size++;
    }

  /* allocate memory */
  tmp_result = malloc (sizeof (struct Namespace) * (ext_size));
  ws4d_fail(tmp_result == NULL, NULL);
  memset (tmp_result, 0, sizeof (struct Namespace) * (ext_size));

  new_size = old_size;

  for (entry = extns; entry && entry->id && entry->ns; entry++)
    {
      int duplicate = 0;
      register struct Namespace *old_entry = NULL;

      for (old_entry = orig_namespaces; old_entry && old_entry->id
           && old_entry->ns; old_entry++)
        {
          if (!strcmp (old_entry->ns, entry->ns) && !strcmp (old_entry->id,
                                                             entry->id))
            {
              duplicate = 1;
              break;
            }
        }

      if (!duplicate)
        {
          tmp_result[new_size - old_size].id = entry->id;
          tmp_result[new_size - old_size].ns = entry->ns;
          new_size++;
        }
    }

  if (new_size > old_size)
    {
      result = ws4d_malloc_alist (sizeof (struct Namespace) * (new_size + 1),
                                  alist);
      ws4d_fail(result == NULL, NULL);
    }
  else
    {
      if (tmp_result)
      {
        free (tmp_result);
      }

      return orig_namespaces;
    }

  memset (result, 0, sizeof (struct Namespace) * (new_size + 1));
  memcpy (result, orig_namespaces,
          sizeof (struct Namespace) * (old_size + 1));

  entry = result;
  entry += old_size;

  for (count = 0; count < (new_size - old_size); count++, entry++)
    {
      entry->id = tmp_result[count].id;
      entry->ns = tmp_result[count].ns;
    }

  if (tmp_result)
  {
    free (tmp_result);
  }

  return result;
}

#ifndef WITH_NOIO
void *
soap_getpeer (struct soap *soap)
{
  return &soap->peer;
}

size_t
soap_getpeerlen (struct soap * soap)
{
  return soap->peerlen;
}
#endif

char *
soap_gen_prefix (int *num, struct ws4d_abs_allocator *alist,
                 struct Namespace *prefix_namespaces)
{
  char prefix[255];
  int collision = 0;

  /* test parameters */
  ws4d_assert (alist && prefix_namespaces, NULL);

  do
    {
      struct Namespace *cur_ns = prefix_namespaces;
      collision = 0;

      *num += 1;
      SNPRINTF (prefix, 255, "n%d", *num);

      while (cur_ns->ns)
        {
          if (cur_ns->id)
            {
              if (!strcmp (cur_ns->id, prefix))
                {
                  collision = 1;
                }
            }
          cur_ns++;
        }

    }
  while (collision == 1);

  return ws4d_strdup (prefix, alist);
}

struct Namespace *
soap_qnamelist_namespaces (ws4d_qnamelist * head,
                           struct ws4d_abs_allocator *alist,
                           struct Namespace *qname_namespaces)
{
  register struct ws4d_qname *type, *next;
  register struct Namespace *result = NULL, *entry = NULL, *ns = NULL;
  int type_count = 0, prefix_count = 0;

  /* test parameters */
  ws4d_assert (head && alist, result);

  ws4d_qnamelist_foreach (type, next, head)
  {
    if (type->ns)
      {
        int found = 0;
        for (ns = qname_namespaces; ns && ns->id; ns++)
          {
            if (ns && ns->ns && !strcmp (type->ns, ns->ns))
              {
                found = 1;
                break;
              }
          }
        if (found == 0)
          {
            type_count++;
          }
      }
  }

  result = ws4d_malloc_alist (sizeof (struct Namespace) * (type_count + 1),
                              alist);
  ws4d_assert (result != NULL, NULL);
  memset (result, 0, sizeof (struct Namespace) * (type_count + 1));

  entry = result;
  ws4d_qnamelist_foreach (type, next, head)
  {
    if (type->ns)
      {
        int found = 0;
        for (ns = qname_namespaces; ns && ns->id; ns++)
          {
            if (ns && ns->ns && !strcmp (type->ns, ns->ns))
              {
                found = 1;
                break;
              }
          }
        if (found == 0)
          {
            if (type->prefix)
              {
                entry->id = type->prefix;
              }
            else
              {
                /* TODO: should fail somehow if no prefix can be generated */
                entry->id =
                  soap_gen_prefix (&prefix_count, alist, qname_namespaces);
              }
            entry->ns = type->ns;
            entry++;
          }
      }
  }

  return result;
}

/************
 * soap based allocator
 * **********/

#define SOAP_ALLOCATORPLUGIN_ID "gSOAP allocator V 1.0"
const char *soap_allocator_plugin_id = SOAP_ALLOCATORPLUGIN_ID;

#define soap_allocator_checkid(allocator) \
  ((allocator)->id && (((allocator)->id == soap_allocator_plugin_id)||(!strcmp((allocator)->id, soap_allocator_plugin_id))))

static void
soap_allocator_fdone (struct ws4d_abs_allocator *allocator)
{
  WS4D_UNUSED_PARAM (allocator);

  return;
}

static int
soap_allocator_ffreeall (struct ws4d_abs_allocator *allocator)
{
  WS4D_UNUSED_PARAM (allocator);

  return WS4D_OK;
}

static void *
soap_allocator_falloc (struct ws4d_abs_allocator *allocator, size_t size)
{
  struct soap *soap;

  /* test parameters */
  ws4d_assert (soap_allocator_checkid (allocator), NULL);
  soap = allocator->data;

  /* allocate memory */
  return soap_malloc (soap, size);
}

static int
soap_allocator_ffree (struct ws4d_abs_allocator *allocator, void *buf)
{
  struct soap *soap;

  /* test parameters */
  ws4d_assert (soap_allocator_checkid (allocator), WS4D_ERR);
  soap = allocator->data;

  /* free memory */
  soap_dealloc (soap, buf);
  return WS4D_OK;
}

static void *
soap_allocator_fmemdup (struct ws4d_abs_allocator *allocator, const void *buf,
                        size_t size)
{
  struct soap *soap;
  void *result;

  /* test parameters */
  ws4d_assert (soap_allocator_checkid (allocator) && (size > 0), NULL);
  soap = allocator->data;

  /* allocate memory */
  result = soap_malloc (soap, size);
  ws4d_fail (result == NULL, NULL);

  /* free memory */
  memcpy (result, buf, size);

  return result;
}

static struct ws4d_abs_allocator_cbs soap_allocator_cbs = {
  soap_allocator_fdone,
  soap_allocator_ffreeall,
  soap_allocator_falloc,
  soap_allocator_ffree,
  soap_allocator_fmemdup,
  soap_allocator_fmemdup,
  soap_allocator_ffree
};

int
soap_allocator_finit (struct ws4d_abs_allocator *allocator, void *arg)
{
  struct soap *soap = arg;

  /* test parameters */
  ws4d_assert (allocator && soap, WS4D_ERR);

  allocator->id = soap_allocator_plugin_id;
  allocator->callbacks = &soap_allocator_cbs;
  allocator->data = soap;

  return WS4D_OK;
}

/* TODO: fix memory leak that may exist here */
void
soap_copy_local_namespaces (struct soap *src, struct soap *dst)
{
  if (src->local_namespaces && !dst->local_namespaces)
    {
      register const struct Namespace *ns1;
      register struct Namespace *ns2;
      register size_t n = 1;
      for (ns1 = src->local_namespaces; ns1->id; ns1++)
        n++;
      n *= sizeof (struct Namespace);
      ns2 = (struct Namespace *) SOAP_MALLOC (dst, n);
      if (ns2)
        {
          memcpy (ns2, src->local_namespaces, n);
          dst->local_namespaces = ns2;
        }
    }
}

const char *
soap_expand_QNames (struct soap *soap, const char *s)
{
  /* test parameters */
  ws4d_assert(soap && s, NULL);

  soap->labidx = 0;
  while (*s)
    {
      while ((*s > 0) && (*s <= 32))
        s++;
      if (!*s)
      {
        return "";
      }

      if (*s == '"')            /* already expanded QName */
        {
          const char *q;
          q = strchr (s, '"');
          if (!q)
            {
              /* This should never happen (invalid gSOAP escaping!) */
              return NULL;
            }
          while (*q > 32)
          {
            q++;
          }
          soap_append_lab (soap, s, q - s);
          s = q;
        }
      else                      /* expand QName */
        {
          struct Namespace *np;
          const char *ns = NULL;
          const char *q, *r = strchr (s, ':');
          size_t k = r - s;
          for (np = soap->local_namespaces; np->id; np++)
            {
              if (!strncmp (np->id, s, k))
                {
                  ns = np->ns;
                  break;
                }
            }

          /* This should never happen. gSOAP should already
           * replaced the prefix. Using empty namespace. */
          ws4d_assert(ns, NULL);

          soap_append_lab (soap, "\"", 1);
          soap_append_lab (soap, ns, strlen (ns));
          soap_append_lab (soap, "\":", 2);
          r++;
          q = r;
          while (*q > 32)
          {
            q++;
          }
          soap_append_lab (soap, r, q - r);
          s = q;
        }
      /* advance to next and add spacing */
      if (*s)
      {
        soap_append_lab (soap, " ", 1);
      }
    }
  soap_append_lab (soap, "\0", 1);

  return soap_strdup (soap, soap->labbuf);
}
