/* WS4D-gSOAP - Implementation of the Devices Profile for Web Services
 * (DPWS) on top of gSOAP
 * Copyright (C) 2007 University of Rostock
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301 USA
 */

#ifndef WS4D_HOSTINGSERVICE_H_
#define WS4D_HOSTINGSERVICE_H_

/**
 * Service Hosting
 *
 * @addtogroup APIHosting Service Hosting
 * @ingroup WS4D_UTILS
 *
 * @{
 */

/** @} */

#include "ws4d_abstract_eprlist.h"
#include "ws4d_device_description.h"

/**
 * Hosted Service RelationShip meta data
 *
 * @addtogroup APIHostedRelApi Hosted Service RelationShip meta data
 * @ingroup APIHosting
 *
 * @{
 */

struct ws4d_host
{
  char *Addr;
  char *ServiceId;
  ws4d_qnamelist types;
  char *__any;
};

struct ws4d_relationship
{
  struct ws4d_host host;
  struct ws4d_host *hosted;
  int hosted_count;
};

/** @} */

/**
 * Hosted Service Plugin
 *
 * This epr plugin is used by the hosting service to manage hosted services.
 *
 * @addtogroup APIHostedServicePlugin Hosted Service Plugin
 * @ingroup APIHosting
 *
 * @{
 */

/**
 * Get ID of an hosted service
 *
 * See ws4d_serviceep_getid() for more information.
 */
#define ws4d_hosted_get_id(epr)  \
  ws4d_serviceep_getid(epr)

/**
 * Add a type to an hosted service
 *
 * See ws4d_serviceep_addtype() for more information.
 */
#define ws4d_hosted_add_type(epr, qname)  \
  ws4d_serviceep_addtype(epr, qname)

/**
 * Add a list of types to and hosted service
 *
 * See ws4d_serviceep_addtypestr() for more information
 */
#define ws4d_hosted_add_typestring(epr, qnames)  \
  ws4d_serviceep_addtypestr (epr, qnames)

/**
 * Get the list of types of an hosted service
 *
 * See ws4d_serviceep_gettypelist() for more information
 */
#define ws4d_hosted_get_typelist(epr)  \
  ws4d_serviceep_gettypelist(epr)

/**
 * Set uri of an hosted service
 *
 * @param hostedservice service to set uri
 * @param uri_template uri
 * @return WS4D_OK on success, an error code otherwise
 */
int ws4d_hosted_set_uri (struct ws4d_epr *hostedservice,
                         const char *uri_template);

/**
 * Get uri of an hosted service
 *
 * @param hostedservice service to get uri
 * @return pointer to service uri on success or NULL otherwise
 */
const char *ws4d_hosted_get_uri (struct ws4d_epr *hostedservice);

/**
 * Set WSDL of an hosted service
 *
 * @param hostedservice service to set wsdl
 * @param wsdl wsdl
 * @return WS4D_OK on success, an error code otherwise
 */
int ws4d_hosted_set_wsdl (struct ws4d_epr *hostedservice, const char *wsdl);

/**
 * Get WSDL of an hosted service
 *
 * @param hostedservice service to get WSDL
 * @return pointer to service WSDL on success or NULL otherwise
 */
const char *ws4d_hosted_get_wsdl (struct ws4d_epr *hostedservice);

/**
 * Set extension elements of an hosted service
 *
 * This function can be used to add extensions to the element corresponding to
 * the Relationship meta data of this hosted service.
 *
 * @param hostedservice service to set extension elements
 * @param ext ext
 * @return WS4D_OK on success, an error code otherwise
 */
int ws4d_hosted_set_ext (struct ws4d_epr *hostedservice, const char *ext);

/**
 * Get extension elements of an hosted service
 *
 * @param hostedservice service to get the extension elements
 * @return pointer to service extension elements on success or NULL otherwise
 */
const char *ws4d_hosted_get_ext (struct ws4d_epr *hostedservice);

/**
 * Set web service toolkit specific information for transport
 *
 * This function is internally used for the ws toolkit abstraction to relate a
 * service with a ws toolkit specific service structure. When you use the gsoap
 * toolkit this is used to relate a gsoap handle to a service.
 *
 * @param hostedservice service to set ws toolkit specific data
 * @param transport_data pointer to ws toolkit specific data structure
 * @return WS4D_OK on success, an error code otherwise
 */
int ws4d_hosted_set_transportdata (struct ws4d_epr *hostedservice,
                                   void *transport_data);

/**
 * Get web service toolkit specific information for transport
 *
 * This function is internally used for the ws toolkit abstraction to relate a
 * service with a ws toolkit specific service structure. When you use the gsoap
 * toolkit this is used to relate a gsoap handle to a service.
 *
 * @param hostedservice service to get ws toolkit specific data
 * @return pointer to ws toolkit specific data structure on success, NULL otherwise
 */
void *ws4d_hosted_get_transportdata (struct ws4d_epr *hostedservice);

/**
 * Activate a hosted service
 *
 * Services have the state active or inactive. Only active services are published
 * within the Relationship metadata. If you create new services they are
 * inactive by default until you activate them.
 *
 * @param hostedservice service to activate
 * @return WS4D_OK on success, an error code otherwise
 */
int ws4d_hosted_activate (struct ws4d_epr *hostedservice);

/**
 * Deactivate a hosted service
 *
 * Services have the state active or inactive. Only active services are published
 * within the Relationship metadata. If you create new services they are
 * inactive by default until you activate them.
 *
 * @param hostedservice service to deactivate
 * @return WS4D_OK on success, an error code otherwise
 */
int ws4d_hosted_deactivate (struct ws4d_epr *hostedservice);

/**
 * Test if an hosted service is active
 *
 * @param hostedservice service to test
 * @return 1 if service is active, !1 otherwise.
 */
int ws4d_hosted_isactive (struct ws4d_epr *hostedservice);

/** @} */

/**
 * Hosting Service
 *
 * @addtogroup APIHostingServiceMod Hosting Service
 * @ingroup APIHosting
 *
 * @{
 */

#define WS4D_HS_UNCHANGED       0x00000000
#define WS4D_HS_RELCHANGED      0x00000001      /* Relationship changed */
#define WS4D_HS_DEVCHANGED      0x00000002      /* ThisDevice changed   */
#define WS4D_HS_MODCHANGED      0x00000004      /* ThisModel changed    */

struct ws4d_hostingservice
{
  struct ws4d_stringlist hosts;
  int change_tracker;
  struct ws4d_abs_eprlist services;
  struct ws4d_thisDevice device;
  struct ws4d_thisModel model;
  struct ws4d_abs_allocator alist;
#ifdef WITH_MUTEXES
    WS4D_MUTEX (lock);
#endif
};

/**
 * Initialize hosting service structure
 *
 * @param hs hosting service structure to initialize
 * @param interf interface the hosting service is related to
 *
 * @return WS4D_OK on success, an error code otherwise
 */
int ws4d_hosting_init (struct ws4d_hostingservice *hs,
                       struct ws4d_stringlist *netdevs);

/**
 * Destroy hosting service structure
 *
 * @param hs hosting service structure to destroy
 *
 * @return WS4D_OK on success, an error code otherwise
 */
int ws4d_hosting_done (struct ws4d_hostingservice *hs);

#ifdef WITH_MUTEXES
#define ws4d_hosting_lock(hs) ws4d_mutex_lock(&(hs)->lock)
#define ws4d_hosting_unlock(hs) ws4d_mutex_unlock(&(hs)->lock)
#else

/**
 * Lock the hosting service structure
 */
#define ws4d_hosting_lock(hs)

/**
 * Unlock the hosting service structure
 */
#define ws4d_hosting_unlock(hs)
#endif

/**
 * Change interface a hosting service is related to
 *
 * @param hs hosting service structure
 * @param interf new interface the hosting service is related to
 *
 * @return WS4D_OK on success, an error code otherwise
 */
int ws4d_hosting_change_interf (struct ws4d_hostingservice *hs,
                                const char *interf);

/**
 * Add a new service to the hosting service
 *
 * @param hs hosting service structure
 * @param serviceid locally unique id for the service
 *
 * @return WS4D_OK on success, an error code otherwise
 */
struct ws4d_epr *ws4d_hosting_add_service (struct ws4d_hostingservice *hs,
                                           const char *serviceid);

/**
 * Bind a hosted service to an uri template
 *
 * This function binds the a service to an uri template. Uri templates are
 * uris with variables that are replaced by ws4d_hosting_bind_service() in this
 * case. Uri templates like http://host:0/ are replaced with uris that contain
 * interface related to the hs like this: http://192.168.1.1:0/. The content
 * of the uri template buffer is replaced wiht the resulting uri.
 *
 * @param hs hosting service structure
 * @param service service to bind
 * @param uri_template uri_template that is used to bind the service and replaced with the resulting uri
 * @param size size of the uri template buffer
 *
 * @return WS4D_OK on success, an error code otherwise
 */
int ws4d_hosting_bind_service (struct ws4d_hostingservice *hs,
                               struct ws4d_epr *service, char *uri_template,
                               size_t size);

/**
 * Activates a hosted service so that it is part of the relationship metadata
 *
 * @param hs hosting service structure
 * @param service service to activate
 * @param uri uri of the service
 * @param size size of the uri buffer
 *
 * @return WS4D_OK on success, an error code otherwise
 */
int ws4d_hosting_activate_service (struct ws4d_hostingservice *hs,
                                   struct ws4d_epr *service, char *uri,
                                   size_t size);

/**
 * Deactivates a hosted service
 *
 * @param hs hosting service structure
 * @param service service to activate
 *
 * @return WS4D_OK on success, an error code otherwise
 */
int ws4d_hosting_deactivate_service (struct ws4d_hostingservice *hs,
                                     struct ws4d_epr *service);

/**
 * Removes a hosted service from the hosting service
 *
 * @param hs hosting servicce structure
 * @param service service to deactivate
 *
 * @return WS4D_OK on success, an error code otherwise
 */
int ws4d_hosting_remove_service (struct ws4d_hostingservice *hs,
                                 struct ws4d_epr *service);

/**
 * Searches for a hosted service by service id
 *
 * @param hs hosting service structure
 * @param serviceid id of the service to look for
 *
 * @return pointer to service epr if service was found, NULL otherwise
 */
struct ws4d_epr *ws4d_hosting_get_service (struct ws4d_hostingservice *hs,
                                           const char *serviceid);

/**
 * Copy all active services to an epr list
 *
 * @param hs hosting service structure
 * @param list list to copy eprs to
 *
 * @return WS4D_OK on success, an error code otherwise
 */
int ws4d_hosting_get_activeservices (struct ws4d_hostingservice *hs,
                                     struct ws4d_abs_eprlist *list);

/**
 * Get number of active services of a hosting service
 *
 * @param hs hosting service structure
 *
 * @return number of active services
 */
int ws4d_hosting_get_active_services_count (struct ws4d_hostingservice *hs);

/**
 * Get the relationship metadata of a hosting service by copy
 *
 * @param hs hosting service structure
 * @param alist allocation list to copy data
 *
 * @return Copy of relationship metadata on success or NULL otherwise
 *
 * TODO: check if we need this function
 */
struct ws4d_relationship
  *ws4d_hosting_get_relationship (struct ws4d_hostingservice *hs,
                                  struct ws4d_abs_allocator *alist);

/**
 * Free relationship metadata returned by ws4d_hosting_get_relationship
 *
 * @param hs hosting service structure
 * @param relationship data to free
 *
 * TODO: check if we need this function
 */
void ws4d_hosting_free_relationship (struct ws4d_hostingservice *hs,
                                     struct ws4d_relationship *relationship);

/**
 * Get ThisDevice meta data of an hosting service
 *
 * @param hs hosting service structure
 *
 * @return pointer to ThisDevice meta data on success, or NULL otherwise
 */
struct ws4d_thisDevice
  *ws4d_hosting_get_thisdevice (struct ws4d_hostingservice *hs);

/**
 * Get ThisDevice meta data of an hosting service for changeing
 *
 * @param hs hosting service structure
 *
 * @return pointer to ThisDevice meta data on success, or NULL otherwise
 */ struct ws4d_thisDevice
 *ws4d_hosting_change_thisdevice (struct ws4d_hostingservice *hs);

 /**
  * Get ThisModel meta data of an hosting service
  *
  * @param hs hosting service structure
  *
  * @return pointer to ThisModel meta data on success, or NULL otherwise
  */
struct ws4d_thisModel
  *ws4d_hosting_get_thismodel (struct ws4d_hostingservice *hs);

/**
 * Get ThisModel meta data of an hosting service for changing
 *
 * @param hs hosting service structure
 *
 * @return pointer to ThisModel meta data on success, or NULL otherwise
 */

struct ws4d_thisModel
  *ws4d_hosting_change_thismodel (struct ws4d_hostingservice *hs);

/** @} */

#endif /*WS4D_HOSTINGSERVICE_H_ */
