/* Copyright (C) 2007-2010  University of Rostock
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301 USA.
 */

/*
 * security_shell_types.h
 *
 *  Created on: 09.03.2010
 *      Author: cht
 */

#ifndef SECURITY_SHELL_TYPES_H_
#define SECURITY_SHELL_TYPES_H_

#include <openssl/ossl_typ.h>
#include <openssl/x509.h>
#include "ws4d_misc.h"
#include "ws4d_list.h"

/* Errors */
#define ERR_OK		0
#define ERR_SEQUENCE	1
#define ERR_RESPONSE	2
#define ERR_HANDLER     3
#define ERR_NO_ELEM     4
#define ERR_BLOCKED     5
#define ERR_AUTHENTICAT 6
#define ERR_FILEIO      7


/* Authentication Engine */

#define CRYPT_LOG 1             /* REMOVE */

#define CHALLENGE_SIZE 256
#define PIN_SIZE         6
#define REFERENCE_SIZE  128
#define KEY_SIZE       256

typedef unsigned char *ws4dChallenge;
typedef unsigned char *ws4dResponse;
typedef unsigned char *ws4dPin;
typedef unsigned char *ws4dDerivedKey;
typedef X509 *ws4dX509;

#define MAX_WRONG_PINS      3
#define WRONG_PIN_PENALTY 120   // seconds

/* Certificate Database */
typedef char *ws4dCertRef;

struct _cert_list_elem
{
  ws4dX509 cert;
  ws4dPin pin;
  int ack;
  ws4dCertRef ref;
  ws4dChallenge challenge;
  int count;                    /* count how many attempts to enter right pin have been made */
  time_t block_until;           /* block for a certain time */
  char *uri;                    // needed by clients only
  struct ws4d_list_node list;
};

typedef struct _cert_list_elem CertStorage;

/* Authorization Database */

/* An authorization is coped with the following way:
 * An authorization depends on a certain CREDENTIAL, which a client must
 * be able to present. The Authorization Engine uses the HANDLER to judge
 * whether the presented credential matches a saved one or not. Depending
 * on the handlers decision (which can also take the VALIDITY into account)
 * permission to access the RESSOUCRCE is granted or denied. To remember the
 * decision for subsequent requests, the decision is saved in PERMISSION.
 *
 * Validity is ought to be quite versatile. On the one hand, it may serve as
 * a timestamp on which an authorization expires. On the other hand, it can
 * be used to express a duration for a denial.
 */

/* special token to express dynamic authorization */
#define AUTHORIZATION_DEACTIVATED  0
#define AUTHORIZATION_ACTIVATED    1

#define DYNAMIC_AUTHORIZATION (void *)"DYNAMIC"
#define DYNAUTSZ 7              /* for malloc */

#define DEFAULT_DENIAL_DURATION 60      /* how long does a denial last? */

typedef enum
{ DENIED = 0,
  PERMITTED_ONCE,
  PERMITTED_TEMPORARY,
  PERMITTED,
  UNKNOWN
} permission_t;

typedef unsigned int credential_type_t;

#define CredDYNAMIC      0x00000000     /* no credential at all - consider it a dummy */
#define CredX509Cert     0x00000001     /* a real embedded X509-Certificate */
#define CredX509CertRef  0x00000002     /* a reference as generated by authenticaten engine */
#define CredUserPass     0x00000004     /* User-Password-Combination */
#define CredSharedSecret 0x00000008     /* Knowledge of shared secret */
                                    /* to be extended */

typedef enum
{ SERVICE, METHOD } ressource_type_t;


struct authorization_s
{
  void *credential;             /* Nachweis */
  credential_type_t credential_type;
  int (*cred_match_handler) (struct authorization_s *, void *); /* Bedingung */
  void *ressource;              /* Ressource */
  ressource_type_t ressource_type;
  time_t validity;              /* Timestamp */
  permission_t permission;      /* Erlaubnis */
  struct ws4d_list_node list;   /* used for building db of elements in memory */
};

typedef struct authorization_s AuthStorage;

#endif /* SECURITY_SHELL_TYPES_H_ */
