/*! \page tutorial_page Step by Step Airconditioner Tutorial
 *
 * This tutorial gives an introduction to the development of devices and
 * control points (clients) conforming to the Devices Profile for Web Services
 * using WS4D gSOAP. During this tutorial an easy example was chosen to show
 * the features of WS4D-gSOAP. This example is based on a virtual airconditioner
 * that can be controlled by DPWS. It was designed to include all features of DPWS
 * and thus to explain how this features can be implemented with WS4D-gSOAP.
 *
 * The work flow to develop devices or clients with WS4D-gSOAP in this tutorial
 * can be split up in several phases.
 * <ul>
 * <li>definition of device and service metadata</li>
 * <li>implementation of the device</li>
 * <li>running the device </li>
 * <li>implementation of the client</li>
 * <li>running the client</li>
 * </ul>
 *
 * The airconditioner example is implemented in several times, each time with more
 * features. First the \ref simplAC is described. Then two simple clients
 * \ref tutGetStatusClient and \ref tutSimplSetTargetTemperatureClient are
 * implmented. Additionally a more advanced client is described in
 * \ref tutAdvGetStatusClient .The next implementation of the airconditioner
 * device in \ref tutAdvDevice also includes the eventing feature.
 * Finally \ref tutTemperatureEventClient is the related client that can
 * subscribe for and receive events.
 *
 * \section simplAC Simple air conditioner device
 *
 * \subsection setUpProjEvn Setup project environment
 *
 * The best way to setup project environments for projects using
 * WS4D-gSOAP is to use an environment based on the cmake build system.
 *
 * This project environment will be configured to do a out of source tree build.
 * This means all code and generated files are written to another directory tree to
 * keep the source tree clean. This makes the work with version control systems
 * easier.
 *
 * Create the following directory structure for your new project by copying the
 * relevant parts from the air conditioner example in the installation directory.
 * You can find this example in the installation directory of the ws4d-gSOAP
 * toolkit under share/doc/ws4d-gSOAP/example.
 *
 * \verbatim
[Project Main Directory]
 |-- Build           (directory for out of source build)
 |-- CMakeLists.txt  (copy this file from the air conditioner example)
 |-- buildversion    (a file containing the version of your project)
 |-- cmake-modules   (copy this directory from the Airconditioner example)
 |   |-- FindDPWS-C.cmake
 |   |-- FindDPWS-D.cmake
 |   |-- FindDPWS.cmake
 |   |-- FindGSOAP.cmake
 |   |-- FindGnuPatch.cmake
 |   |-- FindGnuSed.cmake
 |   `-- FindXSLTproc.cmake
 `-- src             (directory for source files)
     `-- CMakeLists.txt (empty file)
\endverbatim
 *
 * Then you have to modify CMakeLists.txt and create an empty src/CMakeLists.txt.
 * In CMakeLists.txt file you can modify the following part:
 * \verbatim
######################################################################
# Edit the parts below to use this cmake build environment for your  #
# own projects                                                       #
######################################################################

# Name of the project, language
PROJECT(AC C)

# Path to gsoap installation
SET(GSOAP_PATH "[path where gSOAP is installed]" CACHE PATH "Path where gSOAP is installed")

# Path to dpws installation
SET(DPWS_PATH "[path where ws4d-gSOAP is installed]" CACHE PATH "Path where WS4D-gSOAP is installed")
\endverbatim
 * Here you can specify the project name, in this case "AC" and where cmake will
 * find gSOAP (GSOAP_PATH) and the ws4d-gSOAP toolkit (DPWS_PATH).
 *
 * Now you can test if your project environment is working:
 * \verbatim
$ cd Build
$ cmake ..
-- The C compiler identification is GNU
-- Check for working C compiler: /usr/bin/gcc
-- Check for working C compiler: /usr/bin/gcc -- works
-- Detecting C compiler ABI info
-- Detecting C compiler ABI info - done
-- Using gnu patch: /usr/bin/patch
-- Found gSOAP in: /home/elmex/Dokumente/eclipse/gsoap-linux-2.7.10
-- Using xsltproc: /usr/bin/xsltproc
-- Found DPWS in /home/elmex/Dokumente/eclipse/dpws-core/Install-Debug
-- Looking for include files CMAKE_HAVE_PTHREAD_H
-- Looking for include files CMAKE_HAVE_PTHREAD_H - found
-- Looking for pthread_create in pthreads
-- Looking for pthread_create in pthreads - not found
-- Looking for pthread_create in pthread
-- Looking for pthread_create in pthread - found
-- Buildversion: 0.1

-- Build type not set, defaulting to DEBUG!
-- Configuring done
-- Generating done
-- Build files have been written to: /home/elmex/test/Build
\endverbatim
 *
 * If you get error messages, you have to specify the correct paths to the gSOAP
 * and the WS4D-gSOAP installation.
 *
 * \verbatim
$ cd Build
$ cmake -DGSOAP_PATH:PATH=[absolute path to gSOAP installation] -DDPWS_PATH:PATH=[absolute path to ws4d-gSOAP installation directory] ..
\endverbatim
 *
 * After calling cmake, there should be a build environment
 * for the platform you are working on. At a Linux or Unix platform
 * cmake generates a build environment based on make files.
 *
 * Now you can start to create the device and service description.
 *
 * \subsection tutDeviceDescription Device Definition
 *
 * DPWS offers device description at runtime. This description can be
 * defined at runtime with API functions of WS4D-gSOAP. A better approach
 * is to define the description at development time and use the
 * WS4D-gSOAP code generator to integrate the description into a device.
 *
 * Here is an example description of the air conditioner device:
 * \verbatim
<?xml version="1.0" encoding="ISO-8859-1"?>
<wsm:Metadata xmlns:wsm="http://schemas.xmlsoap.org/ws/2004/09/mex"
        xmlns:acs1="http://www.ws4d.org/axis2/tutorial/AirConditioner"
        xmlns:aces1="http://www.ws4d.org/axis2/tutorial/AirConditionerEvent"
        xmlns:wdp="http://schemas.xmlsoap.org/ws/2006/02/devprof"
        xmlns:xsi="http://www.w3.org/2001/XMLSchema-instance"
        xmlns:wsd="http://schemas.xmlsoap.org/ws/2005/04/discovery"
        xmlns:wsa="http://schemas.xmlsoap.org/ws/2004/08/addressing"
        xmlns:xsd="http://www.w3.org/2001/XMLSchema">
  <wsm:MetadataSection Dialect="http://schemas.xmlsoap.org/ws/2006/02/devprof/Relationship">
    <wdp:Relationship Type="http://schemas.xmlsoap.org/ws/2006/02/devprof/host">
      <wdp:Host>
        <wdp:Types>acs1:AirConditioner</wdp:Types>
        <wdp:ServiceId>DPWS-HostingService</wdp:ServiceId>
      </wdp:Host>
      <wdp:Hosted>
        <wdp:Types>acs1:AirConditionerInterface</wdp:Types>
        <wdp:ServiceId>AirConditioner</wdp:ServiceId>
        <Name>AirConditioner</Name>
      </wdp:Hosted>
    </wdp:Relationship>
  </wsm:MetadataSection>
  <wsm:MetadataSection Dialect="http://schemas.xmlsoap.org/ws/2006/02/devprof/ThisModel">
    <wdp:ThisModel>
      <wdp:Manufacturer lang="de">Universität Rostock</wdp:Manufacturer>
      <wdp:Manufacturer lang="en">University of Rostock</wdp:Manufacturer>
      <wdp:ManufacturerUrl>http://www.uni-rostock.de</wdp:ManufacturerUrl>
      <wdp:ModelName lang="de">Klimaanlagendienst</wdp:ModelName>
      <wdp:ModelName lang="en">AirConditioner Service</wdp:ModelName>
      <wdp:ModelNumber>1.0</wdp:ModelNumber>
      <wdp:ModelUrl>http://www.uni-rostock.de</wdp:ModelUrl>
    </wdp:ThisModel>
  </wsm:MetadataSection>
  <wsm:MetadataSection Dialect="http://schemas.xmlsoap.org/ws/2006/02/devprof/ThisDevice">
    <wdp:ThisDevice>
      <wdp:FriendlyName lang="de">Klimaanlagendienst</wdp:FriendlyName>
      <wdp:FriendlyName lang="en">AirConditioner Service</wdp:FriendlyName>
      <wdp:FirmwareVersion>Version 0.1</wdp:FirmwareVersion>
      <wdp:SerialNumber>234d5e4f-c23f-4d91-84ba-1f159284b37a</wdp:SerialNumber>
    </wdp:ThisDevice>
  </wsm:MetadataSection>
</wsm:Metadata>
\endverbatim
 *
 * The device description consists of 3 parts: Relationship, ThisModel
 * and ThisDevice.
 *
 * ThisModel and ThisDevice contains characteristics of the device and
 * the model that are device users. The Manufacturer,
 * ModelName and Friendlyname tags are localized strings an can be
 * defined in several languages. ThisRelationship describes the
 * relationship between services hosted on the device. There are Host
 * (hosting) services and Hosted services.
 *
 * Host services take over description of their Hosted services. Host
 * services are classified into types specified in a Types tag. A type
 * should define a specific relationship between a Host service an its
 * Hosted services. Host services can be clearly identified on a device
 * by the id specified in the ServiceId tag.
 *
 * Hosted services are also classified into types with a Types tag. A
 * type should describe which interface a service has. The best way to do
 * this is to use port types from WSDL files as types. Hosted services
 * can also be clearly identified on a device by the id specified in the
 * ServiceId tag.
 *
 * All this information is used by the WS4D-gSOAP meta data code generator
 * to generate code to setup the device and all description
 * information. The code generator defines following functions:
 * <ul>
 * <li>acs_set_Metadata() : Sets the description specified in the device
 * definition document described above.</li>
 * <li>acs_setup_HostingService() : Setup the Host (hosting) service</li>
 * </ul>
 *
 * In addition the code generator generates functions to setup hosted
 * services. As the ServiceId tag in hosted services can't be used as
 * identifier in C, a C identifier must be specified in the Name tag. The
 * code generator then generates functions called [prefix]_setup_[Identifier
 * in Name tag].
 *
 * You should save the device description under src/metadata.xml.
 *
 * \subsection tutInterface Service Definition
 *
 * Having defined the device description, the service description of the
 * Hosted services must be defined. The designated way of defining a
 * service interface is WSDL. The design of a WSDL is cumbersome as there
 * are only a few tools to design WSDL files. Further complication may
 * appear when using WSDL files with Web Services Toolkits that support
 * code generation of stub und skeleton code based on WSDL files.
 *
 * A usable tool for creating WSDL files is the WSDL-Editor component of
 * the Web Tools Platform (WTP) of eclipse.
 *
 * The following points for developing WSDL files should be considered:
 * <ul>
 * <li>the service should use document literal message format.</li>
 * <li>WS-Addressing must be used and wsa:Action attributes must be defined
 * for every operation input and output.</li>
 * <li>No soapAction should be used!</li>
 * <li>the service address tag in the binding should be empty.</li>
 * </ul>
 *
 * To use a WSDL for code generation with gSOAP it must be translated to
 * gSOAPs annotated c header syntax for service description. This can be
 * done with the wsdl2h contained in the gSOAP package.
 *
 * You can use the AirConditioner.wsdl from the installation directory from
 * [path to WS4D-gSOAP installation]/share/doc/ws4d-gsoap/example. You should
 * save the air conditioner WSDL file under src/AirConditioner.wsdl
 *
 * \verbatim
$ [path to wsdl2h]/wsdl2h -c -n [prefix] [WSDL file] -o [output gSOAP file]
\endverbatim
 *
 * To generate the acs.gsoap file for the air conditioner device simply call
 *
 * \verbatim
$ cd src
$ [path to wsdl2h]/wsdl2h -c -n acs AirConditioner.wsdl -o acs.gsoap

**  The gSOAP WSDL parser for C and C++ 1.2.10
**  Copyright (C) 2000-2008 Robert van Engelen, Genivia Inc.
**  All Rights Reserved. This product is provided "as is", without any warranty.
**  The gSOAP WSDL parser is released under one of the following two licenses:
**  GPL or the commercial license by Genivia Inc. Use option -l for more info.

Saving acs.gsoap

Cannot open file 'typemap.dat'
Problem reading type map file 'typemap.dat'.
Using internal type definitions for C instead.

Reading file 'AirConditioner.wsdl'
Error: no wsdl:definitions/portType/operation/input

To complete the process, compile with:
soapcpp2 acs.gsoap

\endverbatim
 *
 * You should ignore the error message. It is related to the TemperatureEvent
 * operation in the Airconditioner.wsdl. This operation has no input element on
 * purpose.
 *
 * As gSOAP does not only generate stub and skeleton code for a web
 * service but also a C data binding for the data types used and defined
 * in a service description, it generates a lot of functions for the
 * data binding. The prefix argument defines the prefix for these data
 * types and functions to avoid name collisions.
 *
 * As gSOAP is not aware of DPWS and the WS4D-gSOAP extension, the
 * translated gSOAP file must be modified by hand. First, the DPWS
 * specific protocols must be imported with @#import "devprof.gsoap"
 *
 * \verbatim
[...]

/********************************************************************\
 *                                                                  *
 * Import                                                           *
 *                                                                  *
\********************************************************************/

#import "devprof.gsoap"

/********************************************************************\
 *                                                                  *
 * Schema Namespaces                                                *
 *                                                                  *
\********************************************************************/

[...]
\endverbatim
 *
 * In addition method-action parameters of all operations must be
 * modified or added. This parameters are used to create the message
 * dispatch functions. As DPWS uses document/literal messages can only
 * dispatched with the WS-Addressing actions. So the method-action
 * parameter must be changed to the WS-Addressing action attribute in the
 * first input or output part of an operation.
 *
 * \verbatim
[...]

//gsoap acs1 service method-style:  GetStatus document
//gsoap acs1 service method-encoding: GetStatus literal
//gsoap acs1 service method-action: GetStatus http://www.ws4d.org/axis2/tutorial/AirConditioner/GetStatusIn
int __acs1__GetStatus(
    struct _acs1__EmptyMessage*         acs1__GetStatus,
    struct acs1__ACStateType*           acs1__ACState ///< Response parameter
);

[...]

//gsoap acs1 service method-style:  SetTargetTemperature document
//gsoap acs1 service method-encoding: SetTargetTemperature literal
//gsoap acs1 service method-action: SetTargetTemperature http://www.ws4d.org/axis2/tutorial/AirConditioner/SetTargetTemperatureIn
int __acs1__SetTargetTemperature(
    int                                 acs1__TargetTemperature,  ///< Request parameter
    struct acs1__ACStateType*           acs1__ACState ///< Response parameter
);

[...]
\endverbatim
 *
 * You have to edit src/acs.gsoap to reflect the required modifications.
 *
 * Now further modifications must be applied to the gsoap file, as the
 * code generated by soapcpp2 with this file won't work. There is a
 * problem with empty input messages (messages that go from the client to
 * the device. We use such an empty message in the GetStatus
 * operation. Normally wsdl2h would generate
 *
 * \code
int __acs1__GetStatus(
    struct acs1__ACStateType*           acs1__ACState ///< Response parameter
);
\endcode

for this operation. To let soapcpp2 generate working code a empty
message must be defined in the schema types part of the gsoap file.

\verbatim
/********************************************************************\
 *                                                                  *
 * Schema Types                                                     *
 *                                                                  *
\********************************************************************/

struct _acs1__EmptyMessage
{
  void *_;
};
\endverbatim

This structure must be used as input message in the GetStatus
operation. So the definition of the GetStatus operation should look
like this:

\verbatim
//gsoap acs1 service method-style:  GetStatus document
//gsoap acs1 service method-encoding: GetStatus literal
//gsoap acs1 service method-action: GetStatus http://www.ws4d.org/axis2/tutorial/AirConditioner/GetStatusIn
int __acs1__GetStatus(
    struct _acs1__EmptyMessage*         acs1__GetStatus,
    struct acs1__ACStateType*           acs1__ACState ///< Response parameter
);
\endverbatim

\subsection tutUpdate1BuildSystem Update build system

Now the directory tree of your project should look like this:

\verbatim
[Project Main Directory]
 |-- Build
 |-- CMakeLists.txt
 |-- buildversion
 |-- cmake-modules
 |   |-- FindDPWS-C.cmake
 |   |-- FindDPWS-D.cmake
 |   |-- FindDPWS.cmake
 |   |-- FindGSOAP.cmake
 |   |-- FindGnuPatch.cmake
 |   |-- FindGnuSed.cmake
 |   `-- FindXSLTproc.cmake
 `-- src
     |-- AirConditioner.wsdl
     |-- CMakeLists.txt
     |-- acs.gsoap
     `-- metadata.xml
\endverbatim

Now you have to update src/CMakeLists.txt to do the code generation
part of the build process with the new created files src/metadata.xml
src/AirConditioner.wsdl and src/acs.gsoap.

First you have to add a section that is common for all projects. The code
generation functions of ws4d-gSOAP expect a project structure that is set up
by this section.
\verbatim
######################################################################
# Common for all ws4d-gsoap projects, don't change this !            #
######################################################################

# directory to put all generated files in
SET(gen_DIR ${CMAKE_CURRENT_BINARY_DIR}/gen)

# If the directory for the generated sources does not
# already exists, create it.
IF(NOT EXISTS ${gen_DIR})
  FILE(MAKE_DIRECTORY ${gen_DIR})
ENDIF(NOT EXISTS ${gen_DIR})

# set include paths
INCLUDE_DIRECTORIES(${GSOAP_INCLUDE_DIR} ${DPWS_INCLUDES} ${gen_DIR})

# set options for gSOAP runtime
GSOAP_SET_RUNTIME_FLAGS("-DWITH_NONAMESPACES -DWITH_UDP")
\endverbatim

The GSOAP_SET_RUNTIME_FLAGS macro sets the gSOAP runtime flags, required
to use gSOAP with the ws4d-gSOAP toolkit.

Then we can add the section that generates the code that is needed by the
simple air conditioner device:
\verbatim
######################################################################
######################################################################
##                                                                  ##
## Simple airconditioner device                                     ##
##                                                                  ##
######################################################################
######################################################################

######################################################################
# Code generation - for custom projects this must be modified        #
######################################################################

# generate code and C data binding operations of service
GSOAP_GENERATE(acs.gsoap acs1 ${gen_DIR})   # normal service

# generate device description and setup code
DPWS_METADATA_GENERATE(metadata.xml acs ${gen_DIR})

# embed wsdl files into c code
SET(AIRCONDITIONER_WSDL ${CMAKE_CURRENT_SOURCE_DIR}/AirConditioner.wsdl)
DPWS_EMBEDD_WSDL(${gen_DIR} acs ${AIRCONDITIONER_WSDL})
\endverbatim

GSOAP_GENERATE is a cmake macro that controls the gSOAP code generator
soapcpp2. It has three parameters. The first parameter is the name of
the gSOAP file. The macro searches in the current source directory
(that would be src/ in this case) for the specified file. The second
parameter is the prefix for the stub and skeleton code to avoid c
function name collisions. The third parameter is the directory where
soapcpp2 should write the generated files.

The GSOAP_GENERATE call generates the stub and skeleton code for
the GetStatus and the SetTargetTemperature operation of the
AirConditioner service.

DPWS_METADATA_GENERATE is a cmake macro that controls the WS4D-gSOAP
code generator. This macro has three parameters. The first parameter
is the name of the device description file (metadata.xml). The macro
searches in the current source directory (that would be src/ in this
case) for the specified file. The second parameter is the prefix of
the files and functions to generate. The third parameter is the
directory where the WS4D-gSOAP code generator should write the
generated files.

DPWS_EMBEDD_WSDL is a cmake macro to embed several WSDL files as C
character strings into a C file. The macro has a variable number of
parameters. The first parameter is the directory where the C file with
the embedded WSDLs should be written. The second parameter is the
prefix of the files and functions to generate. The following parameter
are WSDL files with a full absolute path. This filename with path can
later be used to address a wsdl file in the embedded code. The WSDL
file names should be saved in variables to use them later in the cmake
file. We store the location where the AirConditioner.wsdl is located
on the file system to use this information later in the cmake file.

Next step will be the generation of a header file where all
configuration parameters determined at build time will be passed to
the c implementation. This can be done with a C header file with
placeholder that are replaced by cmake at configuration time.

In this case we use the location of the AirConditioner.wsdl stored int the
AIRCONDITIONER_WSDL variable.

You should create the file src/config.h.in with the following content:
\code
#ifndef @PROJECT_NAME@_CONFIG_H_
#define @PROJECT_NAME@_CONFIG_H_

#define AIRCONDITIONER_WSDL "@AIRCONDITIONER_WSDL@"

#endif /*@PROJECT_NAME@_CONFIG_H_*/
\endcode

Then you have to add the following section to the src/CMakeLists.txt
to add the configuration of config.h at build time.

\verbatim
######################################################################
# create config.h                                                    #
######################################################################

CONFIGURE_FILE(${CMAKE_CURRENT_SOURCE_DIR}/config.h.in
               ${gen_DIR}/config.h
               @ONLY)
\endverbatim

This will pass the WSDL file name of the AirConditioner WSDL to the C
code. In the code we can use AIRCONDITIONER_WSDL where we need the
file name. cmake replaces the @@AIRCONDITIONER_WSDL@ with the content
of the AIRCONDITIONER_WSDL variable. In this way further variables of
the cmake build system can be passed to the C implementation of the
device.

Now we can add the part to build the simple air conditioner device:
\verbatim
######################################################################
# build simple air conditioner - for custom projects this must be    #
# modified                                                           #
######################################################################

# ${simple_arcondtitioner_SRCS} defines the dependencies to build
# acs_simple_device. The device and service logic is implemented in
# simple_airconditioner.c. ${GSOAP_STDSOAP2_SOURCE} is the gSOAP runtime
# stdsoap2.c. ${gen_DIR}/acs1ServerLib.c is the skeleton code generated by
# gSOAP. The device description and setup code in ${gen_DIR}/acs_metadata.c was
# generated by WS4D-gSOAP. The AirConditioner.wsdl is embedded into
# ${gen_DIR}/acs_wsdl.c by WS4D-gSOAP.

SET(simple_arcondtitioner_SRCS
  simple_airconditioner.c
  ${GSOAP_STDSOAP2_SOURCE}
  ${gen_DIR}/acs1ServerLib.c
  ${gen_DIR}/acs_metadata.c
  ${gen_DIR}/acs_wsdl.c)

# compile all files in simple_arcondtitioner_SRCS and link them into
# the acs_simple_device executable
ADD_EXECUTABLE(acs_simple_device ${simple_arcondtitioner_SRCS})

EXTEND_TGT_COMPILE_FLAGS(acs_simple_device FLAGS "-DDPWS_DEVICE")

# link the device specific dpws libraries into the airconditioner executable
TARGET_LINK_LIBRARIES(acs_simple_device ${DPWS_LIBRARIES} ${DPWS-D_LIBRARIES})
\endverbatim

To create executables cmake has a macro called ADD_EXECUTABLE where
first the name of the executable and the the name of the C files, that
should be compiled and linked in the executable, is given. In this
case we have defined these files with the cmake variable
simple_arcondtitioner_SRCS.

In case of the simple air conditioner executable, following source files are
required:
 - simple_airconditioner.c: the airconditioner implementation
 - ${GSOAP_STDSOAP2_SOURCE}: the gSOAP runtime
 - ${gen_DIR}/acs1ServerLib.c: the server side code generated by soapcpp2 for the Airconditioner service
 - ${gen_DIR}/acs_metadata.c: the code generated by the WS4D-gSOAP code generator
 - ${gen_DIR}/acs_wsdl.c: the embedded WSDLs

The WS4D-gSOAP toolkit has several operation modes: client, device, peer,
hosted, threadsafe, etc. . This modes are activated by preprocessor definitions
and specific libraries.

The EXTEND_TGT_COMPILE_FLAGS cmake macro is used to define operation flags for
WS4D-gSOAP per target. For the target acs_simple_device we have to define the
flag "DPWS_DEVICE".

Libraries that must be linked to an executable are specified with a
separate macro in cmake. The macro is called TARGET_LINK_LIBRARIES and
has a target name (in this case air conditioner) and the libraries to
link as parameter.

To use WS4D-gSOAP in device mode the libraries defined in ${DPWS_LIBRARIES}
${DPWS-D_LIBRARIES} must be linked into the binary.

Now we can create an minimal c code for an application and save it under
src/simple_airconditioner.c:
\code
int main()
{
  return 0;
}
\endcode

The project tree should look like this:

\verbatim
[Project Main Directory]
 |-- Build
 |-- CMakeLists.txt
 |-- buildversion
 |-- cmake-modules
 |   |-- FindDPWS-C.cmake
 |   |-- FindDPWS-D.cmake
 |   |-- FindDPWS.cmake
 |   |-- FindGSOAP.cmake
 |   |-- FindGnuPatch.cmake
 |   |-- FindGnuSed.cmake
 |   `-- FindXSLTproc.cmake
 `-- src
     |-- AirConditioner.wsdl
     |-- CMakeLists.txt
     |-- acs.gsoap
     |-- config.h.in
     |-- metadata.xml
     `-- simple_airconditioner.c
\endverbatim

To test our project environment we try to build the simple air conditioner
example.
\verbatim
$ cd Build
$ make
-- Using gnu patch: /usr/bin/patch
-- Found gSOAP in: /home/elmex/Dokumente/eclipse/gsoap-linux-2.7.10
-- Found DPWS in /home/elmex/Dokumente/eclipse/dpws-core/Install-Debug
-- Buildversion: 0.1

-- Build type not set, defaulting to DEBUG!
-- Choosing target is: device - none specified
-- Configuring done
-- Generating done
-- Build files have been written to: /home/elmex/test/Build
[ 12%] Generating gen/acs_wsdl.c, gen/acs_wsdl.h
[ 25%] Generating gen/acs1ServerLib.c, gen/acs1Server.c, gen/acs1ClientLib.c, gen/acs1Client.c, gen/acs1C.c, gen/acs1H.h, gen/acs1Stub.h, gen/acs1.nsmap

**  The gSOAP Stub and Skeleton Compiler for C and C++ 2.7.10
**  Copyright (C) 2000-2008, Robert van Engelen, Genivia Inc.
**  All Rights Reserved. This product is provided "as is", without any warranty.
**  The gSOAP compiler is released under one of the following three licenses:
**  GPL, the gSOAP public license, or the commercial license by Genivia Inc.

Using project directory path: /home/elmex/test/Build/src/gen/
Saving /home/elmex/test/Build/src/gen/acs1Stub.h
Saving /home/elmex/test/Build/src/gen/acs1H.h
Saving /home/elmex/test/Build/src/gen/acs1C.c
Saving /home/elmex/test/Build/src/gen/acs1Client.c
Saving /home/elmex/test/Build/src/gen/acs1ClientLib.c
Saving /home/elmex/test/Build/src/gen/acs1Server.c
Saving /home/elmex/test/Build/src/gen/acs1ServerLib.c
Using acs1 service name: ACServiceSoap12Binding
Using acs1 service style: document
Using acs1 service encoding: literal
Using acs1 service location:
Using acs1 schema namespace: http://www.ws4d.org/axis2/tutorial/AirConditioner
Saving /home/elmex/test/Build/src/gen/acs1.nsmap namespace mapping table

Compilation successful

[ 37%] Generating gen/acs_metadata.c, gen/acs_metadata.h
Scanning dependencies of target acs_simple_device
[ 50%] Building C object src/CMakeFiles/acs_simple_device.dir/simple_airconditioner.o
[ 62%] Building C object src/CMakeFiles/acs_simple_device.dir/home/elmex/Dokumente/eclipse/gsoap-linux-2.7.10/gsoap/stdsoap2.o
[ 75%] Building C object src/CMakeFiles/acs_simple_device.dir/gen/acs1ServerLib.o
[ 87%] Building C object src/CMakeFiles/acs_simple_device.dir/gen/acs_metadata.o
[100%] Building C object src/CMakeFiles/acs_simple_device.dir/gen/acs_wsdl.o
Linking C executable ../bin/acs_simple_device
CMakeFiles/acs_simple_device.dir/gen/acs1ServerLib.o: In function `soap_serve___acs1__GetStatus':
/home/elmex/test/Build/src/gen/acs1Server.c:99: undefined reference to `__acs1__GetStatus'
CMakeFiles/acs_simple_device.dir/gen/acs1ServerLib.o: In function `soap_serve___acs1__SetTargetTemperature':
/home/elmex/test/Build/src/gen/acs1Server.c:140: undefined reference to `__acs1__SetTargetTemperature'
collect2: ld gab 1 als Ende-Status zurück
make[2]: *** [bin/acs_simple_device] Fehler 1
make[1]: *** [src/CMakeFiles/acs_simple_device.dir/all] Fehler 2
make: *** [all] Fehler 2
\endverbatim

This won't work, because gSOAP generates skeleton code that expects the
functions __acs1__GetStatus and __acs1__SetTargetTemperature. How these
functions are implemented is described in the following section.

\subsection tutImplServ Implementing the AirConditioner device

The next step is to implement a device with help of the generated
code.

First open src/simple_airconditioner.c and add the following include files:

\code
#include "acs1.nsmap"
#include "dpws_device.h"
#include "acs_metadata.h"
#include "acs_wsdl.h"
#include "config.h"
#include <signal.h>
\endcode

 - acs1.nsmap is generated by soapcpp2 and includes the namespace
table of the service and the prototypes of all generated functions and
structures. You can only include one nsmap file in every C file. If
you want implement more than one service in a device you have to
implement them in separate C files.

 - dpws_device.h is the include file for functions and structures
offered by the dpws device library.

 - acs_metadata.h contains prototypes for functions to use the code
generated by WS4D-gSOAP code generator. The functions we will use in
this device are acs_setup_HostingService(), acs_setup_AirConditioner,
acs_set_Metadata and acs_update_Metadata.

 - acs_wsdl.h contains prototypes for a function acs_set_wsdl, to add
   the embedded wsdls to the device.

 - config.h contains the configuration variables set by the cmake
build system.

 - signal.h is used later to stop the device with ctrl-c.

Before the main function is implemented, a dpws handle to hold all
dpws specific information and a soap handle for the service should be
declared as global variable that all parts of the code have access to
the handles.

\code
struct soap service;
struct dpws_s device;
\endcode

Now the main function to initialize and run the device can be
implemented. At the moment WS4D-gSOAP has no logic to check it's own
IP address. the best way is to pass the IP address where the device
should operate on. This can be best done by command line.

A dpws device has a unique network independent id in form of a uuid
that should look like this:
\verbatim
urn:uuid:093b817d-d6d5-4d75-854f-c06355b17e31
\endverbatim

To be able to start a device several times with separate ids, the id
is also passed by command line. Because of this the first part of the
main function consists of code to parse the command line.

\code
int
main (int argc, char **argv)
{
#ifndef WIN32
  struct sigaction sa;
#endif
  char *interface = NULL;
  char *uuid = NULL;

  /* parsing command line options */
  while (argc > 1)
    {
      if (argv[1][0] == '-')
        {
          char *option = &argv[1][1];

          switch (option[0])
            {
            case 'i': /* set interface with option -i */
              if (strlen (option) > 2)
                {
                  ++option;
                  interface = option;
                }
              else
                {
                  --argc;
                  ++argv;
                  interface = argv[1];
                }
#ifdef DEBUG
              printf ("\nAirconditioner: Set interface to \"%s\"\n", interface);
#endif
              break;
            case 'u': /* set id with option -u */
              if (strlen (option) > 2)
                {
                  ++option;
                  uuid = option;
                }
              else
                {
                  --argc;
                  ++argv;
                  uuid = argv[1];
                }
#ifdef DEBUG
              printf ("\nAirconditioner: Set uuid to \"%s\"\n", uuid);
#endif
              break;
            default:
              fprintf (stderr, "\nAirconditioner: Bad option %s\n", argv[1]);
              printf ("\n%s -i [interface address] -u urn:uuid[uuid]\n", argv[0]);
              exit(1);
            }
        }
      --argc;
      ++argv;
    }

  if (interface == NULL)
    {
      fprintf (stderr, "\nAirconditioner: No interface addrss was specified!\n");
      exit (1);
    }
\endcode

Now the previous declared soap handle should be initialized and the
namespace table should be set:

\code
  /* initialize soap handle */
  soap_init (&service);
#ifdef DEBUG
  soap_omode (&service, SOAP_XML_INDENT);
#endif
  soap_set_namespaces (&service, acs1_namespaces);
\endcode

The first function should be called for every soap handle that is used
in an device. This function initializes the soap handle for usage with
the gSOAP runtime environment. The second call is for debugging
purpose and can be omitted. soap_omode sets flags of the soap
handle. SOAP_XML_INDENT specifies that the xml code of the generated
messages should be indented which eases the reading of the messages
for debugging purpose. The third function is necessary to initialize
the namespace table of the soap handle. If this step is omitted, gSOAP
won't be able to generate valid SOAP messages.

After the gSOAP specific initialization follows the DPWS-Plugin
initialization that consists of several parts.

\code
  /* initialize device and services */
  if (dpws_init (&device, interface)
      || acs_setup_HostingService (&device, &service, uuid, 100)
      || acs_setup_AirConditioner (&device, &service, AIRCONDITIONER_WSDL,
                                   100))
    {
      fprintf (stderr, "\nAirconditioner: Can't init device and services\n");
      dpws_done (&device);
      exit (1);
    }

  /* Set Metadata */
  acs_set_Metadata (&device);
  acs_set_wsdl (&device);

  /* Update Metadata */
  if (dpws_update_Metadata (&device))
    {
      fprintf (stderr, "\nAirconditioner: Can't init metadata\n");
      dpws_done (&device);
      exit (1);
    }
\endcode

First the dpws handle, the hosting service and the AirConditioner
service are initialized and setup. The function #dpws_init is
responsible for the initialization of dpws structures and binds them
to a network device that was specified at the command line here. Then
the Hosting service is setup. This is done with the function
acs_setup_HostingService generated by the WS4D-gSOAP code
generator. This function internally calls #dpws_add_hosting_service
which binds the hosting service at the specified soap handle, sets the
device id and adds the device types to be advertised.

The WS4D-gSOAP code generator also generates a function for each
hosted service. This functions are called [prefix]_setup_[Name specified
in the device description] which internally initializes a service with
the service id specified in the device description, binds this service
to the passed soap handle, sets the service id as specified in the
device description, registers the wsdl with the passed name and
finally registers the service at the hosting service to be advertised.

Then the acs_set_Metadata function generated by the WS4D-gSOAP code
generator sets the device and model characteristics. Afterwards the
embedded wsdls are included by calling the function acs_set_wsdl which
was generated by embedwsdl.

The last step of the dpws initialization is to generate the static
content for the device an service description. This is done with the
acs_update_Metadata function which internally takes a snapshot of all
device and service data, and generates static messages with this
information.

Every time the device or service description data is changed or
services are added or removed this function must be called to update
the description that is delivered to clients. This function also
triggers a mechanism that informs clients that there is a change in
the description on this device.

After all services to advertise has been registered, the services can
be served. This is done with an message processing loop. This loop
consists of an incoming message listener called #dpws_maccept, a
message dispatcher called dpws_mserve and a function that cleans up
soaps internal memory called soap_end. These three function can be
called in an infinite loop. In the time service example this loop is
only interrupted by an signal handler for SIGINT to shut the device
down by calling the service_exit function.

\code
  /* install signal handler for SIGINT or Ctrl-C */
#ifdef WIN32
  signal (SIGINT, service_exit);
#else
  memset (&sa, 0, sizeof (sa));
  sa.sa_handler = service_exit;
  sigaction (SIGINT, &sa, NULL);
#endif
\endcode

To shut the device down in a controlled manner the service_exit
function should contain the following lines.

\code
void
service_exit ()
{
#ifdef DEBUG
  printf ("\nAirconditioner: shutting down...\n");
#endif

  dpws_deactivate_hosting_service (&device);
  soap_done (&service);
  dpws_done (&device);

  exit (0);
}
\endcode

This function should be implemented before the main function!

Before we start the message processing loop, we should tell the
hosting service to start the advertisement of its services. This is
done with the function #dpws_activate_hosting_service.

\code
  /* Tell hosting service to start advertising its hosted services */
  if (dpws_activate_hosting_service (&device))
    {
      fprintf (stderr, "\nAirconditioner: Can't activate device\n");
      dpws_done (&device);
      exit (1);
    }
\endcode

The message processing loop looks like this:

\code
#ifdef DEBUG
  printf ("\nAirconditioner: ready to serve... (Ctrl-C for shut down)\n");
#endif

  for (;;)
    {
      struct soap *handle = NULL, *soap_set[] = SOAP_HANDLE_SET (&service);
      int (*serve_requests[]) (struct soap * soap) =
        SOAP_SERVE_SET (acs1_serve_request);

#ifdef DEBUG
      printf ("\nAirconditioner: waiting for request\n");
#endif

      /* waiting for new messages */
      handle = dpws_maccept (&device, 100000, 1, soap_set);

      if (handle)
        {

#ifdef DEBUG
          printf ("\nAirconditioner: processing request from %s:%d\n",
                  inet_ntoa (handle->peer.sin_addr),
                  ntohs (handle->peer.sin_port));
#endif

          /* dispatch messages */
          if (dpws_mserve (handle, 1, serve_requests))
            {
              soap_print_fault (handle, stderr);
            }

          /* clean up soaps internaly allocated memory */
          soap_end (handle);
        }
    }

  return -1;
}
\endcode

The function #dpws_maccept takes a set of soap handles and waits for
an incoming message to one of the handles. It then accepts the
message and returns the corresponding handle. Then the message can be
processed with #dpws_mserve. This function takes a set of message
processing functions generated by soapcpp2 and tests which processing
function services the request. If the request could be served, the
internal allocated memory of gSoap used for receiving and processing
messages can be freed with soap_end.

Incoming messages can trigger the service methods to be called. This
happens inside the message processing functions that look like
<tt>[prefix defined in GSOAP_GENERATE cmake
macro]_serve_request</tt>. The function prototypes for the functions
are defined in <tt>[prefix defined in GSOAP_GENERATE cmake
macro]Stub.h</tt>. This file can be found in the build directory under
src/gen. In case of the airconditioner service these two function
prototypes are defined:

\code
SOAP_FMAC5 int SOAP_FMAC6
     __acs1__GetStatus( struct soap*,
               struct _acs1__EmptyMessage *acs1__GetStatus,
            struct acs1__ACStateType *acs1__ACState);

SOAP_FMAC5 int SOAP_FMAC6
     __acs1__SetTargetTemperature( struct soap*,
              int acs1__TargetTemperature,
              struct acs1__ACStateType *acs1__ACState);
\endcode

To implement the service logic these two function must be
implemented. These function are similar to the operations defined in
acs.gsoap. The \c SOAP_FMAC5 and \c SOAP_FMAC6 macros can be ignored, but
the <tt>struct soap *</tt> parameter name must be added.

This is a simple implementation of the operation functionality:
\code
int temp = 23;

int __acs1__GetStatus(struct soap *soap,
                      struct _acs1__EmptyMessage *acs1__GetStatus,
                      struct acs1__ACStateType *acs1__ACState)
{
  /* fill response message */
  acs1__ACState->CurrentTemp = temp;
  acs1__ACState->TargetTemp = temp;

  /* create response header */
  return dpws_header_gen_response (soap, NULL, wsa_header_get_ReplyTo (soap),
                                   "http://www.ws4d.org/axis2/tutorial/AirConditioner/GetStatusOut",
                                   wsa_header_get_MessageId (soap),
                                   sizeof (struct SOAP_ENV__Header));
}


int
__acs1__SetTargetTemperature (struct soap *soap,
                              int acs1__TargetTemperature,
                              struct acs1__ACStateType *acs1__ACState)
{
  /* process request message */
  temp = acs1__TargetTemperature;

  /* fill response message */
  acs1__ACState->CurrentTemp = temp;
  acs1__ACState->TargetTemp = temp;

  /* create response header */
  return dpws_header_gen_response (soap, NULL, wsa_header_get_ReplyTo (soap),
                                   "http://www.ws4d.org/axis2/tutorial/AirConditioner/SetTargetTemperatureOut",
                                   wsa_header_get_MessageId (soap),
                                   sizeof (struct SOAP_ENV__Header));
}
\endcode

There is a global temperature variable that holds the current
temperature. In our simple implementation the target temperature and
the current temperature are always the same as the airconditioner is
very fast when adjusting the temperature.

A operation skeleton function has a typical structure. First the input
message (request) is processed, the the output message (response) and
at last the response header is generated.

It is imported to know that the response message is sent after the
operation skeleton function. This means you should not use local
variables for the response message an the response header. Otherwise
you will get memory access problems as the local variables are not
valid when the message is sent. You should allocate memory for the
output message and header with gSOAPs memory management functions
(soap_malloc, soap_strdup, ...). A more detailed description how a
services logic should be implemented inside the skeleton functions can
be found in the gSOAP documentation.

Now the project is ready for a first build.

\subsection buildDevice Build the AirConditioner  device

In the Build directory you call cmake.

\verbatim
$ cd Build
$ make
[ 12%] Building C object src/CMakeFiles/acs_simple_device.dir/simple_airconditioner.o
Linking C executable ../bin/acs_simple_device
[100%] Built target acs_simple_device
\endverbatim

You will find the airconditioner device under Build/bin/acs_simple_device.

\subsection runDevice Running a device

To start the Airconditioner device we have to pass several arguments.
 - IP Address of the interface to operate on with the -i option
 - Device id in form "urn:uuid:[id as uuid]" with the -u option

\verbatim
$ cd Build/bin
$ ./acs_simple_device -i 192.168.1.100 -u urn:uuid:a2bb3689-62ff-4d13-8da8-82171e0916c3
Airconditioner: Set interface to "192.168.1.100"

Airconditioner: Set uuid to "urn:uuid:a2bb3689-62ff-4d13-8da8-82171e0916c3"

Airconditioner: ready to serve... (Ctrl-C for shut down)

Airconditioner: waiting for request

Airconditioner: processing request from 0.0.0.0:3702

Airconditioner: waiting for request

Airconditioner: processing request from 192.168.1.100:58051

Airconditioner: waiting for request

Airconditioner: processing request from 192.168.1.100:58052

Airconditioner: waiting for request

Airconditioner: shutting down...
\endverbatim

You can generate an uuid with the uuidgen tool. This should be available on all
unix or linux based systems. Under windows it is installed with Visual Studio
an can be used with the visual studio command prompt. If you don't pass an
uuid, the device will generate a random uuid. This is not the recommended
operation mode! A device should have a unique id that is persistent.

To stop the device press ctrl-c (press and hold ctrl and press c).

If the device was configured with debugging option (CMAKE_BUILD_TYPE =
Debug, the default mode) the device will create three log files with
information about the gsoap runtime (TEST.log), received messages (RECV.log)
and sent messages (SENT.log).

\section tutGetStatusClient Simple one way airconditioner client

Now we implement a simple client that can use the one way GetStatus
operation of the airconditioner device.

\subsection tutUpdate2BuildSystem Update build system

We assume the existing project structure of the simple airconditioner example:
\verbatim
[Project Main Directory]
 |-- Build
 |-- CMakeLists.txt
 |-- buildversion
 |-- cmake-modules
 |   |-- FindDPWS-C.cmake
 |   |-- FindDPWS-D.cmake
 |   |-- FindDPWS.cmake
 |   |-- FindGSOAP.cmake
 |   |-- FindGnuPatch.cmake
 |   |-- FindGnuSed.cmake
 |   `-- FindXSLTproc.cmake
 `-- src
     |-- AirConditioner.wsdl
     |-- CMakeLists.txt
     |-- acs.gsoap
     |-- config.h.in
     |-- metadata.xml
     `-- simple_airconditioner.c
\endverbatim

Add the following section to the src/CMakeLists.txt to build the simple client:
\verbatim
######################################################################
######################################################################
##                                                                  ##
## GetStatus client                                                 ##
##                                                                  ##
######################################################################
######################################################################

######################################################################
# build simple client - for custom projects this must be modified    #
######################################################################

SET(GetStatus_client_SRCS
  GetStatus_client.c
  ${GSOAP_STDSOAP2_SOURCE}
  ${gen_DIR}/acs1ClientLib.c)

ADD_EXECUTABLE(acs_GetStatus_client ${GetStatus_client_SRCS})

EXTEND_TGT_COMPILE_FLAGS(acs_GetStatus_client FLAGS "-DDPWS_CLIENT")

TARGET_LINK_LIBRARIES(acs_GetStatus_client ${DPWS_LIBRARIES} ${DPWS-C_LIBRARIES})
\endverbatim
The acs_GetStatus_client executable is created with the cmake macro ADD_EXECUTABLE.
All source dependencies are defined with the cmake variable
GetStatus_client_SRCS.

In case of the GetStatus client executable, following source files are
required:
 - GetStatus_client.c: the GetStatus client implementation
 - ${GSOAP_STDSOAP2_SOURCE}: the gSOAP runtime
 - ${gen_DIR}/acs1ClientLib.c: the client side code generated by soapcpp2 for the Airconditioner service

The EXTEND_TGT_COMPILE_FLAGS cmake macro is used to define the operation mode
"DPWS_CLIENT" for the target acs_GetStatus_client.

To use WS4D-gSOAP in client mode the libraries defined in ${DPWS_LIBRARIES}
${DPWS-C_LIBRARIES} must be linked into the binary with the
TARGET_LINK_LIBRARIES cmake macro.

Now we can create an minimal c code for an application and save it under
src/GetStatus_client.c:
\code
int main()
{
  return 0;
}
\endcode

The project tree should look like this:

\verbatim
[Project Main Directory]
 |-- Build
 |-- CMakeLists.txt
 |-- buildversion
 |-- cmake-modules
 |   |-- FindDPWS-C.cmake
 |   |-- FindDPWS-D.cmake
 |   |-- FindDPWS.cmake
 |   |-- FindGSOAP.cmake
 |   |-- FindGnuPatch.cmake
 |   |-- FindGnuSed.cmake
 |   `-- FindXSLTproc.cmake
 `-- src
     |-- AirConditioner.wsdl
     |-- CMakeLists.txt
     |-- acs.gsoap
     |-- config.h.in
     |-- GetStatus_client.c
     |-- metadata.xml
     `-- simple_airconditioner.c
\endverbatim

To test our project environment we try to build the simple GetStatus client.
\verbatim
$ cd Build
$ make
-- Found gSOAP in: /home/elmex/Dokumente/eclipse/gsoap-linux-2.7.10
-- Found DPWS in /home/elmex/Dokumente/eclipse/dpws-core/Install-Debug
-- Buildversion: 0.1

-- Build type not set, defaulting to DEBUG!
-- Choosing target is: device - none specified
-- Configuring done
-- Generating done
-- Build files have been written to: /home/elmex/test/Build
[  8%] Building C object src/CMakeFiles/acs_GetStatus_client.dir/GetStatus_client.o
[ 16%] Building C object src/CMakeFiles/acs_GetStatus_client.dir/home/elmex/Dokumente/eclipse/gsoap-linux-2.7.10/gsoap/stdsoap2.o
[ 25%] Building C object src/CMakeFiles/acs_GetStatus_client.dir/gen/acs1ClientLib.o
Linking C executable ../bin/acs_GetStatus_client
[ 33%] Built target acs_GetStatus_client
[ 41%] Generating gen/acs_wsdl.c, gen/acs_wsdl.h
[ 50%] Generating gen/acs1ServerLib.c, gen/acs1Server.c, gen/acs1ClientLib.c, gen/acs1Client.c, gen/acs1C.c, gen/acs1H.h, gen/acs1Stub.h, gen/acs1.nsmap

**  The gSOAP Stub and Skeleton Compiler for C and C++ 2.7.10
**  Copyright (C) 2000-2008, Robert van Engelen, Genivia Inc.
**  All Rights Reserved. This product is provided "as is", without any warranty.
**  The gSOAP compiler is released under one of the following three licenses:
**  GPL, the gSOAP public license, or the commercial license by Genivia Inc.

Using project directory path: /home/elmex/test/Build/src/gen/
Saving /home/elmex/test/Build/src/gen/acs1Stub.h
Saving /home/elmex/test/Build/src/gen/acs1H.h
Saving /home/elmex/test/Build/src/gen/acs1C.c
Saving /home/elmex/test/Build/src/gen/acs1Client.c
Saving /home/elmex/test/Build/src/gen/acs1ClientLib.c
Saving /home/elmex/test/Build/src/gen/acs1Server.c
Saving /home/elmex/test/Build/src/gen/acs1ServerLib.c
Using acs1 service name: ACServiceSoap12Binding
Using acs1 service style: document
Using acs1 service encoding: literal
Using acs1 service location:
Using acs1 schema namespace: http://www.ws4d.org/axis2/tutorial/AirConditioner
Saving /home/elmex/test/Build/src/gen/acs1.nsmap namespace mapping table

Compilation successful

[ 58%] Generating gen/acs_metadata.c, gen/acs_metadata.h
Scanning dependencies of target acs_simple_device
[ 66%] Building C object src/CMakeFiles/acs_simple_device.dir/simple_airconditioner.o
[ 75%] Building C object src/CMakeFiles/acs_simple_device.dir/home/elmex/Dokumente/eclipse/gsoap-linux-2.7.10/gsoap/stdsoap2.o
[ 83%] Building C object src/CMakeFiles/acs_simple_device.dir/gen/acs1ServerLib.o
[ 91%] Building C object src/CMakeFiles/acs_simple_device.dir/gen/acs_metadata.o
[100%] Building C object src/CMakeFiles/acs_simple_device.dir/gen/acs_wsdl.o
Linking C executable ../bin/acs_simple_device
[100%] Built target acs_simple_device
\endverbatim

This will work but the simple_client won't do something useful. First we
have to implement the client logic that uses the GetSatus operation.

\subsection tutImplGetStatusClient Implementing the simple client using the GetSatus operation

The next step is to implement a client using the GetStatus operation of
the AirConditioner service on the AirConditioner device. Therefor the following
code must be added to src/GetStatus_client.c:

First we have to specify the include files:

\code
#include "acs1.nsmap"
#include "dpws_client.h"

#include "ws4d_eprllist.h"
\endcode

 - acs1.nsmap is generated by soapcpp2 and includes the namespace
   table of the service and the prototypes of all generated functions
   and structures. You can only include one nsmap file in every C
   file. If you want implement more than one service in a device you
   have to implement them in separate C files.

 - dpws_device.h is the include file for functions and structures
   offered by the dpws device library.

 - ws4d_eprllist.h is an implementation of the abstract endpoint reference list
   interface, that is used in the WS4D-gSOAP toolkit to pass lists of endpoint
   references.

We declare the soap and dpws handle globally.

\code
struct soap client;
struct dpws_s dpws;
\endcode

Now we can start the main function. The first part of the main
function is the declaration of variables.

\code
int
main (int argc, char **argv)
{
  ws4d_alloc_list alist;

  struct acs1__ACStateType ACState;

  struct ws4d_epr device, *service = NULL;
  struct ws4d_abs_eprlist services;
  ws4d_qnamelist service_type_list;

  char *deviceaddr = NULL, *interface = NULL, *XAddrs = NULL;
\endcode

 - alist: allocation list to manage memory allocations needed in the simple client

 - ACState: structure for storing the results of an operation call.

 - device, service: end point references for the device and the service to use

 - services: endpoint reference list to be uses for service discovery later on

 - service_type_list: a qname list used for finding the wanted service on the device

 - deviceaddr, interface, XAddrs: miscellaneous variables

The next part is code for parsing the command line. We expect one
parameter -h to specify the host device, one parameter with the
logical device address (uuid) of the device we want to use.

\code
  /* parsing command line options */
  while (argc > 1)
    {
      if (argv[1][0] == '-')
        {
          char *option = &argv[1][1];
          switch (option[0])
            {
            case 'i': /* set interface with option -i */
              if (strlen (option) > 2)
                {
                  ++option;
                  interface = option;
                }
              else
                {
                  --argc;
                  ++argv;
                  interface = argv[1];
                }
#ifdef DEBUG
              printf ("\nsimple_client: Set interface to \"%s\"\n", interface);
#endif
              break;
            case 'd': /* set device address with option -d */
              if (strlen (option) > 2)
                {
                  ++option;
                  deviceaddr = option;
                }
              else
                {
                  --argc;
                  ++argv;
                  deviceaddr = argv[1];
                }
#ifdef DEBUG
              printf ("\nsimple_client: Set device address to \"%s\"\n",
                      deviceaddr);
#endif
              break;
            default:
              fprintf (stderr, "\nsimple_client: Bad option %s\n", argv[1]);
              printf ("\n%s -i [interface address] -d [device address]\n",
                      argv[0]);
              exit(1);
            }
        }
      --argc;
      ++argv;
    }

  if (interface == NULL)
    {
      fprintf (stderr, "\nsimple_client: No interface address was specified!\n");
      exit (1);
    }

  if (deviceaddr == NULL)
    {
      fprintf (stderr, "\nsimple_client: No device address was specified!\n");
      exit (1);
    }
\endcode

After parsing the command line, the allocation list is initialized.

\code
  WS4D_ALLOCLIST_INIT (&alist);
\endcode

Now the soap handle can be initialized.

\code
  /* initialize soap handle */
  soap_init (&client);
#ifdef DEBUG
  soap_omode (&client, SOAP_XML_INDENT);
#endif
\endcode

After initialization of the soap handle, the dpws handle must be
initialized.

\code
  /* initialize WS4D-gSOAP */
  if (dpws_init (&dpws, interface) != SOAP_OK)
    {
      fprintf (stderr, "\nsimple_client: could not initialize dpws handle\n");
      dpws_done (&dpws);
      exit (1);
    }
\endcode

In the WS4D-gSOAP API we use end point references (EPR) (in this case
of type struct wsa_eprlist_elem) to address endpoints. So to resolve
the logical adress of a device passed as a command line parameter, we
have to allocate an epr structure and assigned the logical address.

\code
  /* prepare device epr to resolve */
  ws4d_epr_init (&device);
  ws4d_epr_set_Addrs (&device, deviceaddr);
\endcode

Then we can pass this epr to the dpws_resolve_addr() function. The
function resolves a logical device address (uuid form) into a http address.

\code
  /* resolve address */
  XAddrs = (char *) dpws_resolve_addr(&dpws, &device, NULL, 10000);
  if (XAddrs != NULL)
    {
#ifdef DEBUG
      printf("\nsimple_client: device %s found at addr %s\n",
             ws4d_epr_get_Addrs(&device), XAddrs);
#endif
    }
  else
    {
      fprintf(stderr, "\nsimple_client: device %s cannot be found\n",
              ws4d_epr_get_Addrs(&device));
      exit(1);
    }
\endcode

After resolving the logical address we have to search for the address of
the service on the device we want to use. The services can be filtered
by service type or by service id. In this case we initialize a
service type list with the service type of the AirConditioner service
AirConditionerInterface.

\code
  /* prepare service type list */
  ws4d_qnamelist_init (&service_type_list);
  ws4d_qnamelist_addstring
    ("\"http://www.ws4d.org/axis2/tutorial/AirConditioner\":AirConditionerInterface",
     &service_type_list, &alist);
\endcode

Then we pass the service type list and the resolved device epr to the
dpws_find_services() function to get the epr of the service we want to
use.

\code
  /* look up service with matching service types on device */
  ws4d_eprlist_init(&services, ws4d_eprllist_init, NULL);
  if (dpws_find_services(&dpws, &device, &service_type_list, 10000, &services)
      == SOAP_OK)
    {
      /* use first service */
      service = ws4d_eprlist_get_first(&services);
#ifdef DEBUG
      printf("\nsimple_client: device offers AirConditionerService at %s\n",
             ws4d_epr_get_Addrs(service));
#endif
    }
  else
    {
      fprintf(stderr,
              "\nsimple_client: AirConditionerService not found on %s\n",
              ws4d_epr_get_Addrs(&device));
      exit(1);
    }
\endcode

If we have the service epr and the thus its address, we can set up the
soap handle to call the GetStatus operation. First we have to set up
namespaces of the soap handle. Then the soap header of the request
message must be generated with dpws_header_gen_request(). To generate
the header we have to pass the soap handle, we want to use later, the
service address with ws4d_epr_get_Addrs() and the action of
the input message as specified in the WSDL.

\code
  /* prepare soap handel to use service */
  soap_set_namespaces(&client, acs1_namespaces);
  dpws_header_gen_request(&client, NULL, ws4d_epr_get_Addrs(service),
      "http://www.ws4d.org/axis2/tutorial/AirConditioner/GetStatusIn", NULL,
      NULL, sizeof(struct SOAP_ENV__Header));
\endcode

Now we can call the service with the stub function for the GetSatus
operation generated by soapcpp2. The function has the following
prototyp, as you can see in acs1Stub.h in [Build directorry]/src/gen/.

\code
SOAP_FMAC5 int SOAP_FMAC6
     soap_call___acs1__GetStatus(struct soap *soap,
              const char *soap_endpoint,
              const char *soap_action,
              struct _acs1__EmptyMessage *acs1__GetStatus,
              struct acs1__ACStateType *acs1__ACState);
\endcode

To call this function correctly we have to pass the client soap
handle, the soap_endpoint (the service address in this case) and the
result structure ACState. If the function returns SOAP_OK the
operation call was successful. The result values can then be found in
the ACState structure.

\code
  /* call GetStatus operation */
  if (soap_call___acs1__GetStatus(&client, ws4d_epr_get_Addrs (service),
                                  NULL, NULL, &ACState) == SOAP_OK)
    {
#ifdef DEBUG
      printf("\nsimple_client: got status from %s\n",
             ws4d_epr_get_Addrs(service));
#endif

      printf ("\nCurrentTemp: %d\nTargetTemp: %d\n", ACState.CurrentTemp,
              ACState.TargetTemp);
    }
  else
    {
      fprintf(stderr, "\nsimple_client: error calling GetStatus on %s\n",
              ws4d_epr_get_Addrs(service));
    }
\endcode

After the operation call we have to clean up the allocated memory to
shutdown the application. First we have to call soap_end() and
soap_done() with the soap handle. Then we have to shutdown the stack
by calling dpws_done with the dpws handle as parameter. After freeing
the device epr we free the remaining allocations in the allocation
list by calling ws4d_alloclist_done().

\code
  /* clean up */
  soap_end(&client);
  soap_done(&client);
  dpws_done(&dpws);

  ws4d_eprlist_done(&services);
  ws4d_epr_reset(&device);

  ws4d_alloclist_done(&alist);

  exit (0);
}
\endcode

\subsection compSimplClient Build the GetStatus client

In the Build directory you call make.

\verbatim
$ cd Build
$ make
Scanning dependencies of target acs_GetStatus_client
[  8%] Building C object src/CMakeFiles/acs_GetStatus_client.dir/GetStatus_client.o
Linking C executable ../bin/acs_GetStatus_client
[ 33%] Built target acs_GetStatus_client
[100%] Built target acs_simple_device
\endverbatim

\subsection runSimplClient Running a client implementation

To start the GetStatus client we have to pass several arguments.

 - IP Address of the interface to operate on with the -i option
 - Device address in form "urn:uuid:[id as uuid]" or "http://" with the -d option

\verbatim
$ cd Build/bin
$ ./acs_GetStatus_client -i 192.168.1.100 -d urn:uuid:a2bb3689-62ff-4d13-8da8-82171e0916c3

simple_client: Set interface to "192.168.1.100"

simple_client: Set device address to "urn:uuid:a2bb3689-62ff-4d13-8da8-82171e0916c3"

simple_client: device urn:uuid:a2bb3689-62ff-4d13-8da8-82171e0916c3 found at addr http://192.168.1.100:36032

simple_client: device offers AirConditionerService at http://192.168.1.100:36032/

simple_client: got status from http://192.168.1.100:36032/

CurrentTemp: 23
TargetTemp: 23
\endverbatim

If the client was configured with debugging option (CMAKE_BUILD_TYPE = Debug,
the default mode) the client will create three log files with information about
the gsoap runtime (TEST.log), received messages (RECV.log) and sent messages
(SENT.log).

\section tutSimplSetTargetTemperatureClient Simple two way airconditioner client

Now we implement a simple client that can use the two way SetTargetTemperature
operation of the airconditioner device. This client will be very similar to
the GetStatus client. So same steps explained in the GetStatus client will
be skipped in the SetTargetTemperature client description.

tutUpdateBuildSimplSetTargetTemperatureClient
\subsection tutUpdateBuildSimplSetTargetTemperatureClient Update build system

We assume the existing project structure of the GetStatuse client example:
\verbatim
[Project Main Directory]
 |-- Build
 |-- CMakeLists.txt
 |-- buildversion
 |-- cmake-modules
 |   |-- FindDPWS-C.cmake
 |   |-- FindDPWS-D.cmake
 |   |-- FindDPWS.cmake
 |   |-- FindGSOAP.cmake
 |   |-- FindGnuPatch.cmake
 |   |-- FindGnuSed.cmake
 |   `-- FindXSLTproc.cmake
 `-- src
     |-- AirConditioner.wsdl
     |-- CMakeLists.txt
     |-- acs.gsoap
     |-- config.h.in
     |-- GetStatus_client.c
     |-- metadata.xml
     `-- simple_airconditioner.c
\endverbatim

Add the following section to the src/CMakeLists.txt to build the
SetTargetTemperature client:
\verbatim
######################################################################
######################################################################
##                                                                  ##
## SetTargetTemperature client                                      ##
##                                                                  ##
######################################################################
######################################################################

######################################################################
# build simple client - for custom projects this must be modified   ##
######################################################################

SET(SetTargetTemperature_client_SRCS
  SetTargetTemperature_client.c
  ${GSOAP_STDSOAP2_SOURCE}
  ${gen_DIR}/acs1ClientLib.c)

ADD_EXECUTABLE(acs_SetTargetTemperature_client
  ${SetTargetTemperature_client_SRCS})

EXTEND_TGT_COMPILE_FLAGS(acs_SetTargetTemperature_client
  FLAGS "-DDPWS_CLIENT")

TARGET_LINK_LIBRARIES(acs_SetTargetTemperature_client
  ${DPWS_LIBRARIES}
  ${DPWS-C_LIBRARIES})
\endverbatim

In case of the SetTargetTemperature client executable, following source files are
required:
 - SetTargetTemperature_client.c: the SetTargetTemperature client implementation
 - ${GSOAP_STDSOAP2_SOURCE}: the gSOAP runtime
 - ${gen_DIR}/acs1ClientLib.c: the client side code generated by soapcpp2 for the Airconditioner service

Now we copy the file src/GetStatus_client.c to src/SetTargetTemperature_client.c
as a basis for the SetTargetTemperature client.

\verbatim
$ cd src
$ cp GetStatus_client.c SetTargetTemperature_client.c
\endverbatim

The project tree should look like this:

\verbatim
[Project Main Directory]
 |-- Build
 |-- CMakeLists.txt
 |-- buildversion
 |-- cmake-modules
 |   |-- FindDPWS-C.cmake
 |   |-- FindDPWS-D.cmake
 |   |-- FindDPWS.cmake
 |   |-- FindGSOAP.cmake
 |   |-- FindGnuPatch.cmake
 |   |-- FindGnuSed.cmake
 |   `-- FindXSLTproc.cmake
 `-- src
     |-- AirConditioner.wsdl
     |-- CMakeLists.txt
     |-- acs.gsoap
     |-- config.h.in
     |-- GetStatus_client.c
     |-- metadata.xml
     |-- SetTargetTemperature_client.c
     `-- simple_airconditioner.c
\endverbatim

To test our project environment we try to build the SetTargetTemperature
client.
\verbatim
$ cd Build
$ make
-- Found gSOAP in: /media/disk/elmex/EclipseWorkspace/gsoap-2.7.11
-- Found DPWS in /media/disk/elmex/EclipseWorkspace/ws4d-gsoap-devel/Install-Debug
-- Buildversion: 0.1

-- Build type not set, defaulting to DEBUG!
-- Choosing target is: device - none specified
-- Configuring done
-- Generating done
-- Build files have been written to: /media/disk/elmex/EclipseWorkspace/ws4d-gsoap-devel/Install-Debug/share/doc/ws4d-gSOAP/example/build
[ 25%] Built target acs_GetStatus_client
Scanning dependencies of target acs_SetTargetTemperature_client
[ 31%] Building C object src/CMakeFiles/acs_SetTargetTemperature_client.dir/SetTargetTemperature_client.o
[ 37%] Building C object src/CMakeFiles/acs_SetTargetTemperature_client.dir/gen/acs1ClientLib.o
Linking C executable ../bin/acs_SetTargetTemperature_client
[ 50%] Built target acs_SetTargetTemperature_client
[100%] Built target acs_simple_device
\endverbatim

This will work but the acs_SetTargetTemperature_client will do the same as the
acs_GetStatus_client. First we have to change the code to call the
SetTargetTemperature operation.

\subsection tutImplSimplSetTargetTemperatureClient Implementing a simple client using the SetTargetTemperature operation

The next step is to implement a client using the SetTargetTemperature operation of
the AirConditioner service on the AirConditioner device. Therefor the following
code must be changed in the src/SetTargetTemperature_client.c:

At line 28 you have to add a preprocessor definition to define the initializer of
the TargetTemperature variable:
\code
[...]
struct soap client;
struct dpws_s dpws;

#define ACS_NO_TEMPERATURE -1

int
main (int argc, char **argv)
{
[...]
\endcode

Then you have to define the TargetTemperature variable itself at line 44:
\code
[...]
  char *deviceaddr = NULL;
  char *interf = NULL;
  char *XAddrs = NULL;
  int TargetTemperature = ACS_NO_TEMPERATURE;

  /* parsing command line options */
  while (argc > 1)
[...]
\endcode

Then you have to change the command line parsing code to add the parameter t to
pass the target temperature that the client sends in the SetTargetTemperature
request message:

\code
[...]
  /* parsing command line options */
  while (argc > 1)
    {
      if (argv[1][0] == '-')
        {
          char *option = &argv[1][1];
          switch (option[0])
            {
            case 'i':          /* set interface with option -i */
              if (strlen (option) > 2)
                {
                  ++option;
                  interf = option;
                }
              else
                {
                  --argc;
                  ++argv;
                  interf = argv[1];
                }
#ifdef DEBUG
              printf ("\nsimple_client: Set interface to \"%s\"\n", interf);
#endif
              break;
            case 'd':          /* set device address with option -d */
              if (strlen (option) > 2)
                {
                  ++option;
                  deviceaddr = option;
                }
              else
                {
                  --argc;
                  ++argv;
                  deviceaddr = argv[1];
                }
#ifdef DEBUG
              printf ("\nsimple_client: Set device address to \"%s\"\n",
                      deviceaddr);
#endif
              break;
            case 't':
              if (strlen (option) > 2)
                {
                  ++option;
                  TargetTemperature = atoi (option);
                }
              else
                {
                  --argc;
                  ++argv;
                  TargetTemperature = atoi (argv[1]);
                }
#ifdef DEBUG
              printf ("\nsimple_client: Set target temperature to \"%d\"\n",
                      TargetTemperature);
#endif
              break;
            default:
              fprintf (stderr, "\nsimple_client: Bad option %s\n", argv[1]);
              printf
                ("\n%s -i [interface address] -d [device address] -t [target temperature]\n",
                 argv[0]);
              exit (1);
            }
        }
      --argc;
      ++argv;
    }

    if (TargetTemperature == ACS_NO_TEMPERATURE)
    {
      fprintf (stderr,
               "\nsimple_client: No target temperature was specified!\n");
      exit (1);
    }
[...]
\endcode

Now you only have to change the part that calls the GetStatus operation to
call the SetTargetTemperature operation. These changes contain a different
values in the soap header and another gSOAP stub function to call starting
from line 199:

\code
[...]
  dpws_header_gen_request (&client, NULL, ws4d_epr_get_Addrs (service),
                           "http://www.ws4d.org/axis2/tutorial/AirConditioner/SetTargetTemperatureIn",
                           NULL, NULL, sizeof (struct SOAP_ENV__Header));

  /* call SetTargetTemperature operation */
  if (soap_call___acs1__SetTargetTemperature
      (&client, ws4d_epr_get_Addrs (service), NULL, TargetTemperature,
       &ACState) == SOAP_OK)
    {
#ifdef DEBUG
      printf ("\nsimple_client: Successfully set TargetTemperature on %s\n",
              ws4d_epr_get_Addrs (service));
#endif

      printf ("\nCurrentTemp: %d\nTargetTemp: %d\n", ACState.CurrentTemp,
              ACState.TargetTemp);
    }
  else
    {
      fprintf (stderr,
               "\nsimple_client: Error calling SetTargetTemperature on %s\n",
               ws4d_epr_get_Addrs (service));
    }
[...]
\endcode

Now the SetTargetTemperature implementation is complete.

\subsection tutCompSimplSetTargetTemperatureClient Build the SetTargetTemperature client

In the Build directory you call make.

\verbatim
$ cd Build
$ make
[ 25%] Built target acs_GetStatus_client
Scanning dependencies of target acs_SetTargetTemperature_client
[ 31%] Building C object src/CMakeFiles/acs_SetTargetTemperature_client.dir/SetTargetTemperature_client.o
Linking C executable ../bin/acs_SetTargetTemperature_client
[ 50%] Built target acs_SetTargetTemperature_client
[100%] Built target acs_simple_device
\endverbatim

\subsection tutRunSimplSetTargetTemperatureClient Running the SetTargetTemperature client implementation

To start the GetStatus client we have to pass several arguments.

 - IP Address of the interface to operate on with the -i option
 - Device address in form "urn:uuid:[id as uuid]" or "http://" with the -d option
 - Target temperature for the SetTargetTemperature request with the -t option

\verbatim
$ cd Build/bin
$  ./acs_SetTargetTemperature_client -i 192.168.1.100 -d urn:uuid:a2bb3689-62ff-4d13-8da8-82171e0916c3 -t 20

simple_client: Set interface to "192.168.1.100"

simple_client: Set device address to "urn:uuid:a2bb3689-62ff-4d13-8da8-82171e0916c3"

simple_client: Set target temperature to "20"

simple_client: device urn:uuid:a2bb3689-62ff-4d13-8da8-82171e0916c3 found at addr http://192.168.1.100:39366/a2bb3689-62ff-4d13-8da8-82171e0916c3

simple_client: device offers AirConditionerService at http://192.168.1.100:39366/

simple_client: Successfully set TargetTemperature on http://192.168.1.100:39366/

CurrentTemp: 20
TargetTemp: 20
\endverbatim

\section tutAdvGetStatusClient Advanced one way airconditioner client

Now we implement an advanced client that can use the GetStatus operation
of several airconditioner devices it discovers at start up. This client will be
similar to the GetStatus client. So again same steps explained in the GetStatus
client will be skipped in the advanced GetStatus client description.

\subsection tutUpdateBuildAdvGetStatusClient Update build system

We assume the existing project structure of the SetTargetTemperature client example:

\verbatim
[Project Main Directory]
 |-- Build
 |-- CMakeLists.txt
 |-- buildversion
 |-- cmake-modules
 |   |-- FindDPWS-C.cmake
 |   |-- FindDPWS-D.cmake
 |   |-- FindDPWS.cmake
 |   |-- FindGSOAP.cmake
 |   |-- FindGnuPatch.cmake
 |   |-- FindGnuSed.cmake
 |   `-- FindXSLTproc.cmake
 `-- src
     |-- AirConditioner.wsdl
     |-- CMakeLists.txt
     |-- acs.gsoap
     |-- config.h.in
     |-- GetStatus_client.c
     |-- metadata.xml
     |-- SetTargetTemperature_client.c
     `-- simple_airconditioner.c
\endverbatim

Add the following section to the src/CMakeLists.txt to build the
advanced GetStatus client:
\verbatim
######################################################################
######################################################################
##                                                                  ##
## advanced GetStatus client                                        ##
##                                                                  ##
######################################################################
######################################################################

######################################################################
# build advanced client - for custom projects this must be modified  #
######################################################################

SET(advanced_GetStatus_client_SRCS
  advanced_GetStatus_client.c
  ${GSOAP_STDSOAP2_SOURCE}
  ${gen_DIR}/acs1ClientLib.c)

ADD_EXECUTABLE(acs_advanced_GetStatus_client
  ${advanced_GetStatus_client_SRCS})

EXTEND_TGT_COMPILE_FLAGS(acs_advanced_GetStatus_client
  FLAGS "-DDPWS_CLIENT")

TARGET_LINK_LIBRARIES(acs_advanced_GetStatus_client
  ${DPWS_LIBRARIES}
  ${DPWS-C_LIBRARIES})
\endverbatim

In case of the advanced GetStatus client executable, following source files are
required:
 - advanced_GetStatus_client.c: the advanced GetStatus client implementation
 - ${GSOAP_STDSOAP2_SOURCE}: the gSOAP runtime
 - ${gen_DIR}/acs1ClientLib.c: the client side code generated by soapcpp2 for the Airconditioner service

Now we can create an minimal c code for an application and save it under
src/advanced_GetStatus_client.c:
\code
int main()
{
  return 0;
}
\endcode

The project tree should look like this:

\verbatim
[Project Main Directory]
 |-- Build
 |-- CMakeLists.txt
 |-- buildversion
 |-- cmake-modules
 |   |-- FindDPWS-C.cmake
 |   |-- FindDPWS-D.cmake
 |   |-- FindDPWS.cmake
 |   |-- FindGSOAP.cmake
 |   |-- FindGnuPatch.cmake
 |   |-- FindGnuSed.cmake
 |   `-- FindXSLTproc.cmake
 `-- src
     |-- advanced_GetStatus_client.c
     |-- AirConditioner.wsdl
     |-- CMakeLists.txt
     |-- acs.gsoap
     |-- config.h.in
     |-- GetStatus_client.c
     |-- metadata.xml
     |-- SetTargetTemperature_client.c
     `-- simple_airconditioner.c
\endverbatim

To test our project environment we try to build the advanced GetStatus client.
\verbatim
$ cd Build
$ make
-- Found gSOAP in: /media/disk/elmex/EclipseWorkspace/gsoap-2.7.11
-- Found DPWS in /media/disk/elmex/EclipseWorkspace/ws4d-gsoap-devel/Install-Debug
-- Buildversion: 0.1

-- Build type not set, defaulting to DEBUG!
-- Choosing target is: device - none specified
-- Configuring done
-- Generating done
-- Build files have been written to: /media/disk/elmex/EclipseWorkspace/ws4d-gsoap-devel/Install-Debug/share/doc/ws4d-gSOAP/example/build
[ 20%] Built target acs_GetStatus_client
[ 40%] Built target acs_SetTargetTemperature_client
[ 45%] Building C object src/CMakeFiles/acs_advanced_GetStatus_client.dir/advanced_GetStatus_client.o
[ 50%] Building C object src/CMakeFiles/acs_advanced_GetStatus_client.dir/media/disk/elmex/EclipseWorkspace/gsoap-2.7.11/gsoap/stdsoap2.o
[ 55%] Building C object src/CMakeFiles/acs_advanced_GetStatus_client.dir/gen/acs1ClientLib.o
Linking C executable ../bin/acs_advanced_GetStatus_client
[ 60%] Built target acs_advanced_GetStatus_client
[100%] Built target acs_simple_device
\endverbatim

This will work but the simple_client won't do something useful. First we
have to implement the client logic that uses the GetSatus operation.

\subsection tutImplAdvGetStatusClient Implementing the advanced client using the GetSatus operation

The next step is to implement the advanced client using the GetStatus operation of
the AirConditioner service on the AirConditioner device. Therefor the following
code must be added to src/advanced_GetStatus_client.c:

First we have to specify the include files:
\code
#include "acs1.nsmap"
#include "dpws_client.h"

#include "ws4d_eprllist.h"
\endcode

We declare the soap and dpws handle globally.

\code
struct soap client;
struct dpws_s dpws;
\endcode

And we define the maximum number of supported devices. If more devices are
connected to the network they are discarded.

\code
#define ACS_MAX_DEVICES 20
\endcode

Now we can start the main function. The first part of the main
function is the declaration of variables.

\code
int
main (int argc, char **argv)
{
  ws4d_alloc_list alist;

  struct acs1__ACStateType ACState;

  struct ws4d_epr *device = NULL, *iter = NULL;
  struct ws4d_abs_eprlist devices, services;
  int ret = 0;

  char *deviceaddr = NULL, *host = NULL, *XAddrs = NULL;
\endcode

 - alist: allocation list to manage memory allocations needed in the simple client

 - ACState: structure for storing the results of an operation call.

 - device, itter: end point references for the devices used later to iterate the device list

 - devices, services: endpoint reference list to be uses for device and service discovery

 - deviceaddr, interface, XAddrs: miscellaneous variables

The next part is code for parsing the command line. We expect one
parameter -h to specify the host device.

\code
  /* parsing command line options */
  while (argc > 1)
    {
      if (argv[1][0] == '-')
        {
          char *option = &argv[1][1];
          switch (option[0])
            {
            case 'i': /* set interface with option -i */
              if (strlen (option) > 2)
                {
                  ++option;
                  host = option;
                }
              else
                {
                  --argc;
                  ++argv;
                  host = argv[1];
                }
              break;
            default:
              fprintf (stderr, "\nsimple_client: Bad option %s\n", argv[1]);
              printf ("\n%s -i [interface address]\n", argv[0]);
              exit (1);
            }
        }
      --argc;
      ++argv;
    }

  if (host == NULL)
    {
      printf ("\nNo host was specified!\n");
      fflush (NULL);
      exit (1);
    }
\endcode

After parsing the command line, the allocation list is initialized,

\code
  WS4D_ALLOCLIST_INIT (&alist);
\endcode

Now the soap handle can be initialized.

\code
  /* initialize client soap handle */
  soap_init (&client);
#ifdef DEBUG
  soap_omode (&client, SOAP_XML_INDENT);
#endif
\endcode

After initialization of the soap handle, the dpws handle must be
initialized.

\code
  /* initialize WS4D-gSOAP */
  if (dpws_init (&dpws, host) != SOAP_OK)
    {
      fprintf (stderr, "\n%s: Could not initialize dpws handle\n", argv[0]);
      dpws_done (&dpws);
      exit (1);
    }
\endcode

As the simple GetStatus client could talk to only one airconditioner device
we used a single EPR (in this case of type struct wsa_eprlist_elem) to address
the device. As we want to address all devices we find at start up we need a
list of EPRs (in this case of type struct ws4d_abs_eprlist).

\code
  /* initialize device list */
  ws4d_eprlist_init (&devices, ws4d_eprllist_init, NULL);
\endcode

The advanced GetStatus client doesn't talk to every device but only those
that host an AirconditionerService. This is signaled with the airconditioner
device type we can use when we search for devices in the network. As we could
also search for a list of device types that devices must match all, we have to
pass a list of device types when searching for devices. Device types are
actually QNames and can be passed to the WS4D-gSOAP-API with the structure
ws4d_qnamelist. Now we initialize such a list and add the airconditioner device
type QName to this list.

\code
  /* probe for all airconditioner devices */
  ws4d_qnamelist type_list;

  /* init type list */
  ws4d_qnamelist_init (&type_list);
  ws4d_qnamelist_addstring
  ("\"http://www.ws4d.org/axis2/tutorial/AirConditioner\":AirConditioner",
   &type_list, &alist);
\endcode

Now we have prepared all parameters for the WS4D-gSOAP API function dpws_probe()
that can be use to search for devices in the network that match the specified
device types.

\code
#ifdef DEBUG
  printf ("\nProbing for acs devices for 10 seconds ... ");
  fflush (NULL);
#endif
  ret = dpws_probe (&dpws, &type_list, NULL, 10000, 100, NULL,
          NULL, &devices);
  if (ret != WS4D_OK)
  {
    if (ret == WS4D_TO)
    {
      fprintf (stderr, "No devices found!\n");
    }
    else
    {
      fprintf (stderr, "Could not probe for devices!\n");
    }
    dpws_done (&dpws);
    exit (1);
  }
#ifdef DEBUG
  printf ("done\n");
  fflush (NULL);
#endif
\endcode

As shown in this example, you should always check the return value of
dpws_probe(). If the function returns WS4D_OK, the probe was successful and the
found devices were added to the devices list. If the function returns WS4D_TO
no device was found in the given timeout (in this case 10000 milli seconds).
When the returns WS4D_ERR then there was an internal error.

After a successful probe the devices list contains the found airconditioner
devices. Now we have to search for the address of the services on the devices
we want to use. We do this by iterating all devices in the devices list.

\code
  /* look for airconditioner service at every device found before */
  ws4d_eprlist_foreach (device, iter, &devices)
  {
    if (ws4d_epr_isvalid (device))
      {
\endcode

First we resolve the logical addresses of the devices to addressable
addresses with the WS4D-gSOAP API function dpws_resolve_addr()
\code
        struct ws4d_epr *service = NULL;
        ws4d_qnamelist service_type_list;

        /* resolve address */
        XAddrs = (char *) dpws_resolve_addr (&dpws, device, NULL, 10000);
        if (XAddrs != NULL)
          {
            fprintf (stderr, "\n\nDevice %s found at addr %s\n",
                     ws4d_epr_get_Addrs (device), XAddrs);
          }
        else
          {
            fprintf (stderr, "\nDevice %s cannot be found\n",
                     ws4d_epr_get_Addrs (device));
            fflush (NULL);
            continue;
          }
\endcode

After resolving the logical address we have to search for the address of
the service on this device. The services can be filtered by service type or
by service id. In this case we initialize a service type list with the service
type of the AirConditioner service AirConditionerInterface.

\code
        /* prepare service type list */
        ws4d_qnamelist_init (&service_type_list);
        ws4d_qnamelist_addstring
          ("\"http://www.ws4d.org/axis2/tutorial/AirConditioner\":AirConditionerInterface",
           &service_type_list, &alist);
\endcode

Then we pass the service type list and the resolved device epr to the
dpws_find_services() function to get the epr of the service we want to
use.

\code
        /* look up service with matching service types on device */
        ws4d_eprlist_init (&services, ws4d_eprllist_init, NULL);
        if (dpws_find_services (&dpws, device, &service_type_list,
                                10000, &services) == SOAP_OK)
          {
            service = ws4d_eprlist_get_first (&services);
            printf ("\nDevice offers AirConditionerService at %s\n",
                    ws4d_epr_get_Addrs (service));
          }
        else
          {
#ifdef DEBUG
            fprintf (stderr,
                     "\nsimple_client: AirConditionerService not found on %s\n",
                     ws4d_epr_get_Addrs (device));
#endif
            continue;
          }
\endcode

If we have the service epr and the thus its address, we can set up the
soap handle to call the GetStatus operation. Otherwise we continue with the
next device.

The following steps to call the GetStatus operation are similar to the simple
GetStatus client example:

\code
    /* prepare soap handel to use service */
    soap_set_namespaces (&client, acs1_namespaces);
    dpws_header_gen_request (&client, NULL,
                 ws4d_epr_get_Addrs (service),
                 "http://www.ws4d.org/axis2/tutorial/AirConditioner/GetStatusIn",
                 NULL, NULL,
                 sizeof (struct SOAP_ENV__Header));

    /* call GetStatus operation */
    if (soap_call___acs1__GetStatus (&client,
                     ws4d_epr_get_Addrs
                     (service), NULL, NULL,
                     &ACState) == SOAP_OK)
      {
      printf ("\n%s - %s\n\tCurrentTemp: %d\n\tTargetTemp: %d\n",
          ws4d_epr_get_Addrs (device), ws4d_epr_get_Addrs (service),
          ACState.CurrentTemp, ACState.TargetTemp);
      }
    else
      {
        fprintf (stderr, "\nsimple_client: error calling GetStatus on %s\n",
                 ws4d_epr_get_Addrs (service));
      }
\endcode

After each operation call we have to clean up the allocated memory. This is
important as wo get memory leaks in gSOAP and WS4D-gSOAP otherwise.

\code
        /* clean up */
        ws4d_eprlist_done (&services);
        soap_end (&client);

      }
  }
\endcode

And this will be repeated for every device in the devices list.

Now we cann shutdown the application. First we have to call
soap_done() with the soap handle. Then we have to shutdown the stack
by calling dpws_done with the dpws handle as parameter. After freeing
the devices list we free the remaining allocations in the allocation
list by calling ws4d_alloclist_done().

\code
  /* clean up */
  soap_done (&client);
  dpws_done (&dpws);

  ws4d_eprlist_done (&devices);

  ws4d_alloclist_done (&alist);

  exit (0);
}
\endcode

\subsection tutCompAdvGetStatusClient Build the advanced GetStatus client

In the Build directory you call make.

\verbatim
$ cd Build
$ make
[ 20%] Built target acs_GetStatus_client
[ 40%] Built target acs_SetTargetTemperature_client
Scanning dependencies of target acs_advanced_GetStatus_client
[ 45%] Building C object src/CMakeFiles/acs_advanced_GetStatus_client.dir/advanced_GetStatus_client.o
Linking C executable ../bin/acs_advanced_GetStatus_client
[ 60%] Built target acs_advanced_GetStatus_client
[100%] Built target acs_simple_device
\endverbatim

\subsection tutRunAdvGetStatusClient Running the advanced GetStatus client implementation

To start the advanced GetStatus client we have to pass only one argument.

 - IP Address of the interface to operate on with the -i option

\verbatim
$ cd Build/bin
$ ./acs_advanced_GetStatus_client -i 192.168.1.100

Probing for acs devices for 10 seconds ... done


Device urn:uuid:31af0499-560d-4af5-8c04-56d1a4faf75d found at addr http://192.168.1.100:43522/31af0499-560d-4af5-8c04-56d1a4faf75d

Device offers AirConditionerService at http://192.168.1.100:43522/

urn:uuid:31af0499-560d-4af5-8c04-56d1a4faf75d - http://192.168.1.100:43522/
  CurrentTemp: 23
  TargetTemp: 23


Device urn:uuid:a2bb3689-62ff-4d13-8da8-82171e0916c3 found at addr http://192.168.1.100:52909/a2bb3689-62ff-4d13-8da8-82171e0916c3

Device offers AirConditionerService at http://192.168.1.100:52909/

urn:uuid:a2bb3689-62ff-4d13-8da8-82171e0916c3 - http://192.168.1.100:52909/
  CurrentTemp: 23
  TargetTemp: 23
\endverbatim

\section tutAdvDevice Advanced AirConditioner device with Eventing

The WSDL of the AirConditioner service contains a third operation
called TemperatureEvent that is not implemented in the device
before. This message has a special message exchange pattern. There is
only a TemperatureEventMessageOut that is sent by the device to the
client. To allow this message exchange pattern, that is used for
events and notifications, a special mechanism is needed. WS-Eventing,
that is used by DPWS defines a subscription mechanism for events and
notifications.

Implementing Events with WS4D-gSOAP is a litte bit mir
complicated. Several steps are needed:

 - Activate event source and subscription management (subscription
   manager)
 - Implement an event triggering function

As gSOAPs code generator soapcpp2 is designed for the typical request
response and oneway message exchange patterns, we have to generate
separate stub and skeleton for the event operations. This is done on
the bases of an inverted WSDL. There is a tool based on XSLT to invert
a WSDL included in the toolkit. This tool simply reverses the order of
input and output tags in operations. So we generate again a gSOAP file
based on this inverted WSDL, generate stub and skeleton code but
include the client are on the device and the server side on the
client this time.

So we have to do the following steps:

 - invert WSDL
 - generate inverted gSOAP file acs_inv.gsoap
 - adapt acs_inv.gsoap
 - generate stub and skeleton code
 - activate event source and subscription management
 - Implement an event triggering function with the generated stub code of the inverted service

The WSDL can be inverted with the following command line:

\verbatim
$ cd src
$ xsltproc -o AirConditioner.wsdl_inv [path to WS4D-gSOAP installation]/share/ws4d-gSOAP/invert.xslt AirConditioner.wsdl
\endverbatim

Now we generate the inverted gSOAP file acs_inv.gsoap

\verbatim
$ cd src
$ wsdl2h -c -n acsinv AirConditioner.wsdl_inv -o acs_inv.gsoap
\endverbatim

Now you should open acs_inv.gsoap in an editor. Before you apply the
modifications as described in \ref tutInterface you should remove all
operations that are not meant to be events. In this case you must
remove all parts that belong to the GetStatus and the
SetTargetTemperature operations.

\subsection tutUpdate3BuildSystem Update build system

We assume the existing project structure of the simple client example with the
new files AirConditioner.wsdl_inv and acs_inv.gsoap:

\verbatim
[Project Main Directory]
 |-- Build
 |-- CMakeLists.txt
 |-- buildversion
 |-- cmake-modules
 |   |-- FindDPWS-C.cmake
 |   |-- FindDPWS-D.cmake
 |   |-- FindDPWS.cmake
 |   |-- FindGSOAP.cmake
 |   |-- FindGnuPatch.cmake
 |   |-- FindGnuSed.cmake
 |   `-- FindXSLTproc.cmake
 `-- src
     |-- advanced_GetStatus_client.c
     |-- AirConditioner.wsdl
     |-- AirConditioner.wsdl_inv
     |-- CMakeLists.txt
     |-- acs.gsoap
     |-- acs_inv.gsoap
     |-- config.h.in
     |-- GetStatus_client.c
     |-- metadata.xml
     |-- SetTargetTemperature_client.c
     `-- simple_airconditioner.c
\endverbatim

Add the following section to the src/CMakeLists.txt to build the
aircondtitioner device:
\verbatim
######################################################################
######################################################################
##                                                                  ##
## air conditioner device with eventing                             ##
##                                                                  ##
######################################################################
######################################################################

# generate code and C data binding for events of service
GSOAP_GENERATE(acs_inv.gsoap acs_inv1 ${gen_DIR}) # event service

SET (advanced_arcondtitioner_SRCS
  advanced_airconditioner.c
  event_worker.c
  ${GSOAP_STDSOAP2_SOURCE}
  ${gen_DIR}/acs1ServerLib.c
  ${gen_DIR}/acs_inv1ClientLib.c
  ${gen_DIR}/acs_metadata.c
  ${gen_DIR}/acs_wsdl.c)

# compile all files in simple_arcondtitioner_SRCS and link them into
# the acs_simple_device executable
ADD_EXECUTABLE(acs_advanced_device ${advanced_arcondtitioner_SRCS})

EXTEND_TGT_COMPILE_FLAGS(acs_advanced_device
  FLAGS "-DDPWS_DEVICE -DWITH_MUTEXES")

# link the device specific dpws libraries into the airconditioner executable
TARGET_LINK_LIBRARIES(acs_advanced_device
  ${DPWS_LIBRARIES} ${DPWS-DMT_LIBRARIES} ${CMAKE_THREAD_LIBS_INIT})
\endverbatim

The airconditioner device executable is created with the cmake macro
ADD_EXECUTABLE. All source dependencies are defined with the cmake variable
advanced_arcondtitioner_SRCS.

In case of the airconditioner device executable, following source files are
required:
 - advanced_airconditioner.c: the airconditioner service implementation
 - event_worker.c: a airconditionder state machine and event source
 - ${GSOAP_STDSOAP2_SOURCE}: the gSOAP runtime
 - ${gen_DIR}/acs1ServerLib.c: the server side code generated by soapcpp2
 - ${gen_DIR}/acs_inv1ClientLib.c: the client side code for the event source generated by soapcpp2
 - ${gen_DIR}/acs_metadata.c:
 - ${gen_DIR}/acs_wsdl.c:

The EXTEND_TGT_COMPILE_FLAGS cmake macro is used to define the operation mode
"DPWS_DEVICE" for the target acs_advanced_device. The airconditioner device uses
threads for simultaneously running tasks. Thus the airconditioner device uses
the multi threading mode of the WS4D-gSOAP toolkit activated with "WITH_MUTEXES".

To use WS4D-gSOAP in device mode with multi threading the libraries defined
in ${DPWS_LIBRARIES} ${DPWS-DMT_LIBRARIES} and the pthread libraries must be
linked into the binary with the TARGET_LINK_LIBRARIES cmake macro.

Now we can create an empty event_worker.c, event_worker.h and minimal c code for an application
and save it under src/advanced_airconditioner.c:
\code
int main()
{
  return 0;
}
\endcode

The project tree should look like this:

\verbatim
[Project Main Directory]
 |-- Build
 |-- CMakeLists.txt
 |-- buildversion
 |-- cmake-modules
 |   |-- FindDPWS-C.cmake
 |   |-- FindDPWS-D.cmake
 |   |-- FindDPWS.cmake
 |   |-- FindGSOAP.cmake
 |   |-- FindGnuPatch.cmake
 |   |-- FindGnuSed.cmake
 |   `-- FindXSLTproc.cmake
 `-- src
     |-- advanced_GetStatus_client.c
     |-- AirConditioner.wsdl
     |-- AirConditioner.wsdl_inv
     |-- CMakeLists.txt
     |-- acs.gsoap
     |-- acs_inv.gsoap
     |-- advanced_airconditioner.c
     |-- config.h.in
     |-- event_worker.c
     |-- event_worker.h
     |-- GetStatus_client.c
     |-- metadata.xml
     |-- SetTargetTemperature_client.c
     `-- simple_airconditioner.c
\endverbatim

To test our project environment we try to build the project.
\verbatim
$ cd Build
$ make
-- Found gSOAP in: /media/disk/elmex/EclipseWorkspace/gsoap-2.7.11
-- Found DPWS in /media/disk/elmex/EclipseWorkspace/ws4d-gsoap-devel/Install-Debug
-- Buildversion: 0.1

-- Build type not set, defaulting to DEBUG!
-- Choosing target is: device - none specified
-- Configuring done
-- Generating done
-- Build files have been written to: /media/disk/elmex/EclipseWorkspace/ws4d-gsoap-devel/Install-Debug/share/doc/ws4d-gSOAP/example/build
[ 12%] Built target acs_GetStatus_client
[ 25%] Built target acs_SetTargetTemperature_client
[ 38%] Built target acs_advanced_GetStatus_client
[ 41%] Generating gen/acs_inv1ServerLib.c, gen/acs_inv1Server.c, gen/acs_inv1ClientLib.c, gen/acs_inv1Client.c, gen/acs_inv1C.c, gen/acs_inv1H.h, gen/acs_inv1Stub.h, gen/acs_inv1.nsmap

**  The gSOAP Stub and Skeleton Compiler for C and C++ 2.7.11
**  Copyright (C) 2000-2008, Robert van Engelen, Genivia Inc.
**  All Rights Reserved. This product is provided "as is", without any warranty.
**  The gSOAP compiler is released under one of the following three licenses:
**  GPL, the gSOAP public license, or the commercial license by Genivia Inc.


**WARNING**: option -1 or -2 overrides SOAP-ENV namespace (detected at line 44 in /media/disk/elmex/EclipseWorkspace/ws4d-gsoap-devel/Install-Debug/share/doc/ws4d-gSOAP/example/src/acs_inv.gsoap)


**WARNING**: option -1 or -2 overrides SOAP-ENC namespace (detected at line 45 in /media/disk/elmex/EclipseWorkspace/ws4d-gsoap-devel/Install-Debug/share/doc/ws4d-gSOAP/example/src/acs_inv.gsoap)

Using project directory path: /media/disk/elmex/EclipseWorkspace/ws4d-gsoap-devel/Install-Debug/share/doc/ws4d-gSOAP/example/build/src/gen/
Saving /media/disk/elmex/EclipseWorkspace/ws4d-gsoap-devel/Install-Debug/share/doc/ws4d-gSOAP/example/build/src/gen/acs_inv1Stub.h
Saving /media/disk/elmex/EclipseWorkspace/ws4d-gsoap-devel/Install-Debug/share/doc/ws4d-gSOAP/example/build/src/gen/acs_inv1H.h
Saving /media/disk/elmex/EclipseWorkspace/ws4d-gsoap-devel/Install-Debug/share/doc/ws4d-gSOAP/example/build/src/gen/acs_inv1C.c
Saving /media/disk/elmex/EclipseWorkspace/ws4d-gsoap-devel/Install-Debug/share/doc/ws4d-gSOAP/example/build/src/gen/acs_inv1Client.c
Saving /media/disk/elmex/EclipseWorkspace/ws4d-gsoap-devel/Install-Debug/share/doc/ws4d-gSOAP/example/build/src/gen/acs_inv1ClientLib.c
Saving /media/disk/elmex/EclipseWorkspace/ws4d-gsoap-devel/Install-Debug/share/doc/ws4d-gSOAP/example/build/src/gen/acs_inv1Server.c
Saving /media/disk/elmex/EclipseWorkspace/ws4d-gsoap-devel/Install-Debug/share/doc/ws4d-gSOAP/example/build/src/gen/acs_inv1ServerLib.c
Using acsinv1 service name: ACServiceSoap12Binding
Using acsinv1 service style: document
Using acsinv1 service encoding: literal
Using acsinv1 service location:
Using acsinv1 schema namespace: http://www.ws4d.org/axis2/tutorial/AirConditioner
Saving /media/disk/elmex/EclipseWorkspace/ws4d-gsoap-devel/Install-Debug/share/doc/ws4d-gSOAP/example/build/src/gen/acs_inv1.nsmap namespace mapping table

Compilation successful (2 warnings)

[ 45%] Building C object src/CMakeFiles/acs_advanced_device.dir/advanced_airconditioner.o
[ 48%] Building C object src/CMakeFiles/acs_advanced_device.dir/event_worker.o
[ 51%] Building C object src/CMakeFiles/acs_advanced_device.dir/media/disk/elmex/EclipseWorkspace/gsoap-2.7.11/gsoap/stdsoap2.o
[ 54%] Building C object src/CMakeFiles/acs_advanced_device.dir/gen/acs1ServerLib.o
[ 58%] Building C object src/CMakeFiles/acs_advanced_device.dir/gen/acs_inv1ClientLib.o
[ 61%] Building C object src/CMakeFiles/acs_advanced_device.dir/gen/acs_metadata.o
[ 64%] Building C object src/CMakeFiles/acs_advanced_device.dir/gen/acs_wsdl.o
Linking C executable ../bin/acs_advanced_device
CMakeFiles/acs_advanced_device.dir/gen/acs1ServerLib.o: In function `soap_serve___acs1__GetStatus':
/media/disk/elmex/EclipseWorkspace/ws4d-gsoap-devel/Install-Debug/share/doc/ws4d-gSOAP/example/build/src/gen/acs1Server.c:99: undefined reference to `__acs1__GetStatus'
CMakeFiles/acs_advanced_device.dir/gen/acs1ServerLib.o: In function `soap_serve___acs1__SetTargetTemperature':
/media/disk/elmex/EclipseWorkspace/ws4d-gsoap-devel/Install-Debug/share/doc/ws4d-gSOAP/example/build/src/gen/acs1Server.c:140: undefined reference to `__acs1__SetTargetTemperature'
collect2: ld gab 1 als Ende-Status zurück
make[2]: *** [bin/acs_advanced_device] Fehler 1
make[1]: *** [src/CMakeFiles/acs_advanced_device.dir/all] Fehler 2
make: *** [all] Fehler 2
\endverbatim

This won't work, because the functions __acs1__GetStatus and
__acs1__SetTargetTemperature must be implemented.

\subsection tutImplEventWorker Implementing the AirConditioner device with eventing

At the moment it is not possible to implement several services for
dpws in one c file because of namespace conflicts. So we implement
the eventing mechanism in a separate c file called
event_worker.c. Besides the event delivery we also implement a state
machine with the states of the AirConditioner service in this file.

First we have to specify the include files:

\code
#include "acs_inv1.nsmap"
#include "dpws_device.h"
#ifndef WIN32
#include <pthread.h>
#endif
\endcode

 - acs_inv1.nsmap is generated by soapcpp2 and includes the namespace
table of the stubs for the event source.

 - dpws_device.h is the include file for functions and structures
offered by the dpws device library.

 - pthread.h for multi threading support

The following code is not specific for DPWS or the usage of the
WS4D-gSOAP toolkit, but an example how to implement a state machine
for the airconditioner. You can skip this section if you want to
implement your own state machine.

First we need some global variables, for keeping a dpws handle
reference, the airconditioner state and handles for multi threading.

\code
/* reference to device handle */
struct dpws_s *_device = NULL;

/* Supported states of the air conditioner. */
enum ac_states { OFF, IDLE, COOLING };
int currentState = OFF;         /* Initial state is turned off. */
int currentTemp = 24;           /* Initial temperature in degrees celsius. */
int targetTemp = 24;            /* Initial target temperature in degrees celsius. */
int shutdownFlag = 0;           /* Flag indicates systemshutdown. */

/* handle for multithreading */
#ifdef WIN32
DWORD workerid;
HANDLE worker;
#else
pthread_t worker;
#endif
\endcode

Then we need an initialization and a shutdown function.

\code
void *event_worker_loop ();

int
event_worker_init (struct dpws_s *device)
{
  if (_device)
    {
      return SOAP_ERR;
    }
  else
    {
      if (!device)
  {
    return SOAP_ERR;
  }
      _device = device;
    }

#ifdef WIN32
  worker = CreateThread (NULL, 0, (DWORD WINAPI) event_worker_loop, NULL, 0, &workerid);
  if (worker != NULL)
#else
  if (pthread_create (&worker, NULL, event_worker_loop, NULL) == 0)
#endif
    {
      return SOAP_OK;
    }
  else
    {
      return SOAP_ERR;
    }
}
\endcode

This function initializes the device handle reference with the passed
device handle and creates a thread for the state machine that is
implemented in the event_worker_loop function.

\code
int
event_worker_shutdown ()
{
  shutdownFlag = 1;
#ifdef WIN32
  WaitForSingleObject(worker, INFINITE);
#else
  pthread_join (worker, NULL);
#endif
}
\endcode

The event_worker_shutdown function sets the shutdownFlag which is
monitored in the event_worker_loop function. Then it waits for the
event_worker_loop function to finish.

\code
int
event_worker_get_current ()
{
  return currentTemp;
}
\endcode

This function returns the current temperature of the state machine.

\code
int
event_worker_set_target (int target)
{
  targetTemp = target;
  if (currentState == OFF)
    {
      currentState = IDLE;
    }
}


int
event_worker_get_target ()
{
  return targetTemp;
}
\endcode

This functions set and return the target temperature of the state
machine.

\code
void deliver_event (int CurrentTemp, int TargetTemp);

void *
event_worker_loop ()
{
  /* Wait for initial start. */
  while (OFF == currentState && !shutdownFlag)
    SLEEP (1);

  printf ("Airconditioner: Entering working loop...\n");
  while (!shutdownFlag)
    {
      if (targetTemp < currentTemp)
        {
          printf
            ("Airconditioner: Current Temperature %d; Switching to state COOLING.\n",
             currentTemp);
          currentState = COOLING;

          while (targetTemp <= currentTemp && !shutdownFlag)
            {
              currentTemp--;
              SLEEP (5);
            }
        }
      else
        {
          printf
            ("Airconditioner: Reached target temperature: %d; Switching to state IDLE.\n",
             targetTemp);
          currentState = IDLE;

          deliver_event (currentTemp, targetTemp);

          SLEEP (5);
          currentTemp += 2;
        }
    }
  printf ("Airconditioner: Leaving working loop...\n");
}
\endcode

The function event_worker_loop() implements the state machine of the
AirConditioner service.

The functionality of the event_worker module is exported with the
following header file you should save as event_worker.h. Later on
the event_worker module must be integrated into the airconditioner
device.

\code
#ifndef EVENT_WORKER_H_
#define EVENT_WORKER_H_

int
event_worker_init (struct dpws_s *device);

int
event_worker_shutdown ();

int event_worker_get_current ();

int event_worker_set_target (int target);

int event_worker_get_target ();

#endif /*EVENT_WORKER_H_*/
\endcode

At the moment many parts of the event delivery have to be implemented
by the device developer. In principle an event delivery of a specific
event must be triggered by the device implementation.

\code
void
deliver_event (int CurrentTemp, int TargetTemp)
{
  struct soap soap;
  struct ws4d_subscription *subs, *next;
  struct acsinv1__ACStateType event;

  event.CurrentTemp = CurrentTemp;
  event.TargetTemp = TargetTemp;

  soap_init (&soap);
  soap_set_namespaces (&soap, acs_inv1_namespaces);

  dpws_for_each_subs (subs, next, _device,
                      "http://www.ws4d.org/axis2/tutorial/AirConditioner/TemperatureEventOut")
  {
    struct __acsinv1__TemperatureEvent response;
    char *deliverto = dpws_subsm_get_deliveryPush_address (_device, subs);

    if (!deliverto)
      continue;

    dpws_header_gen_oneway (&soap, NULL, deliverto,
                            "http://www.ws4d.org/axis2/tutorial/AirConditioner/TemperatureEventOut",
                            NULL, sizeof (struct SOAP_ENV__Header));

    printf ("Sending Event to %s\n", deliverto);

    if (soap_send___acsinv1__TemperatureEvent
        (&soap, deliverto, NULL, &event))
      {
        soap_print_fault (&soap, stderr);
      }
    else
      {
        soap_recv___acsinv1__TemperatureEvent (&soap, &response);
      }

    soap_end (&soap);
  }

  soap_done (&soap);
}
\endcode

An event delivery function must iterate over the subscriptions known
by the subscription manager and filter out those that are
subscriptions for the event to be delivered. There are two functions
dpws_subsm_get_first_by_action and dpws_subsm_get_next_by_action to
iterate over all known subscriptions and filter them by the specified
action filter.

Then for each matching function the delivery address must be acquired
with the function dpws_subsm_get_deliveryPush_address().

The event message itself is delivered like a typical message send by a
client. First we have to allocate and initialize a soap handle (this
is done outside of the subscription filter loop), generate a dpws
header. Depending on the message exchange pattern used by the
operation used for event delivery, soapcpp has created a
soap_call_[...] function (for solicit response) or a pair of
soap_send_[...] and soap_recv_[...] function (for notification). In
the letter case the recv function must be used to receive the http
response, if omitted the device is not dpws compliant and may cause
problems on some clients.

After sending the event, all memory allocated for the soap handle must
be freed with soap_end() and soap_done().

The differences of simple_airconditioner.c and advanced_airconditioner.c are
minimal. So the best was to create advanced_airconditioner.c is to copy
simple_airconditioner.c to advanced_airconditioner.c and integrate the
following changes.

First we have to add the include file of event worker:
\code
#include "event_worker.h"
\endcode

Then we have to add the event_worker_shoutdown() function to
service_exit() function to properly shut the event_worker thread down.

\code
service_exit ()
{
  printf ("\nAirconditioner: shutting down...\n");

  event_worker_shutdown();

  dpws_deactivate_hosting_service (&device);
  soap_done (&service);
  dpws_done (&device);

  exit (0);
}
\endcode

The next step is important, as if we omit this, the device won't
accept subscriptions. To accept subscriptions we have to activate the
event source on the AirConditioner service. This can be done with the
dpws_activate_eventsource function, where we have to pass the device
handle and the soap handle where the AirConditioner service is bound
to. This should be done after the service was set up but before we
start the message processing loop.

\code
[...]

  /* activate eventing. */
  if (dpws_activate_eventsource (&device, &service))
    {
      printf ("\nAirconditioner: Can't activate eventing\n");
      dpws_done (&device);
      exit (0);
    }

  /* install signal handler for SIGINT or Ctrl-C */
#ifdef WIN32
  signal (SIGINT, service_exit);
#else
  memset (&sa, 0, sizeof (sa));
  sa.sa_handler = service_exit;
  sigaction (SIGINT, &sa, NULL);
#endif
  if (event_worker_init(&device))
  {
      printf ("\nAirconditioner: Can't init event worker\n");
      dpws_done (&device);
      exit (0);
  }

[...]
\endcode

After activating the event source, we have to initialize the event
worker with the function event_worker_init implemented in
event_worker.c

To automatically expire subscriptions the function dpws_check_subscriptions()
must be integrated into the main loop.

\code
#ifdef DEBUG
  printf ("\nAirconditioner: ready to serve... (Ctrl-C for shut down)\n");
#endif

  for (;;)
    {
      struct soap *handle = NULL, *soap_set[] = SOAP_HANDLE_SET (&service);
      int (*serve_requests[]) (struct soap * soap) =
        SOAP_SERVE_SET (acs1_serve_request);

#ifdef DEBUG
      printf ("\nAirconditioner: waiting for request\n");
#endif

      /* waiting for new messages */
      handle = dpws_maccept (&device, 100000, 1, soap_set);

      if (handle)
        {

#ifdef DEBUG
          printf ("\nAirconditioner: processing request from %s:%d\n",
                  inet_ntoa (handle->peer.sin_addr),
                  ntohs (handle->peer.sin_port));
#endif

          /* dispatch messages */
          if (dpws_mserve (handle, 1, serve_requests))
            {
              soap_print_fault (handle, stderr);
            }

          /* clean up soaps internaly allocated memory */
          soap_end (handle);
        }

      dpws_check_subscriptions (&device);

    }

  return -1;
\endcode

At least we have to integrate the function to get and set target and
current temperature. The operation skeletons of GetStatus and
SetTemperature then looks like.

\code
int __acs1__GetStatus(struct soap *soap,
                      void *_,
                      struct acs1__ACStateType *acs1__ACState)
{
  /* fill response message */
  acs1__ACState->CurrentTemp = event_worker_get_current();
  acs1__ACState->TargetTemp = event_worker_get_target();

  /* create response header */
  return dpws_header_gen_response (soap, NULL, wsa_header_get_ReplyTo (soap),
                                   "http://www.ws4d.org/axis2/tutorial/AirConditioner/GetStatusOut",
                                   wsa_header_get_MessageId (soap),
                                   sizeof (struct SOAP_ENV__Header));
}


int
__acs1__SetTargetTemperature (struct soap *soap,
                              int acs1__TargetTemperature,
                              struct acs1__ACStateType *acs1__ACState)
{
  /* process request message */
  event_worker_set_target ( acs1__TargetTemperature);

  /* fill response message */
  acs1__ACState->CurrentTemp = event_worker_get_current();
  acs1__ACState->TargetTemp = event_worker_get_target();

  /* create response header */
  return dpws_header_gen_response (soap, NULL, wsa_header_get_ReplyTo (soap),
                                   "http://www.ws4d.org/axis2/tutorial/AirConditioner/SetTargetTemperatureOut",
                                   wsa_header_get_MessageId (soap),
                                   sizeof (struct SOAP_ENV__Header));
}
\endcode

\subsection tutbuildAdvDevice Build the advanced airconditioner device

In the Build directory you call make.

\verbatim
$ cd Build
$ make
[ 12%] Built target acs_GetStatus_client
[ 25%] Built target acs_SetTargetTemperature_client
[ 38%] Built target acs_advanced_GetStatus_client
Scanning dependencies of target acs_advanced_device
[ 41%] Building C object src/CMakeFiles/acs_advanced_device.dir/advanced_airconditioner.o
[ 45%] Building C object src/CMakeFiles/acs_advanced_device.dir/event_worker.o
Linking C executable ../bin/acs_advanced_device
[ 74%] Built target acs_advanced_device
[100%] Built target acs_simple_device
\endverbatim

\subsection tutRunAdvDevice Run the advanced airconditioner device

To start the advanced Airconditioner device we have to pass several arguments.
 - IP Address of the interface to operate on with the -i option
 - Device id in form "urn:uuid:[id as uuid]" with the -u option

\verbatim
./acs_advanced_device -i 192.168.1.100 -u urn:uuid:31af0499-560d-4af5-8c04-56d1a4faf75d

Airconditioner: Set interface to "192.168.1.100"

Airconditioner: Set uuid to "urn:uuid:31af0499-560d-4af5-8c04-56d1a4faf75d"

Airconditioner: ready to serve... (Ctrl-C for shut down)

Airconditioner: waiting for request

Airconditioner: shutting down...
Airconditioner: Entering working loop...
Airconditioner: Leaving working loop...
\endverbatim

To stop the device press ctrl-c (press and hold ctrl and press c).

\section tutTemperatureEventClient Advanced airconditioner client with eventing

Now we implement a client that can receive the TemperatureEvent event of the
advanced device.

\subsection tutUpdateBuildTemperatureEventClient Update build system

We assume the existing project structure of the advanced device example:

\verbatim
[Project Main Directory]
 |-- Build
 |-- CMakeLists.txt
 |-- buildversion
 |-- cmake-modules
 |   |-- FindDPWS-C.cmake
 |   |-- FindDPWS-D.cmake
 |   |-- FindDPWS.cmake
 |   |-- FindGSOAP.cmake
 |   |-- FindGnuPatch.cmake
 |   |-- FindGnuSed.cmake
 |   `-- FindXSLTproc.cmake
 `-- src
     |-- advanced_GetStatus_client.c
     |-- AirConditioner.wsdl
     |-- AirConditioner.wsdl_inv
     |-- CMakeLists.txt
     |-- acs.gsoap
     |-- acs_inv.gsoap
     |-- advanced_airconditioner.c
     |-- config.h.in
     |-- event_worker.c
     |-- event_worker.h
     |-- GetStatus_client.c
     |-- metadata.xml
     |-- SetTargetTemperature_client.c
     `-- simple_airconditioner.c
\endverbatim

Add the following section to the src/CMakeLists.txt to build the
TemperatureEvent client:
\verbatim
######################################################################
######################################################################
##                                                                  ##
## client with eventing                                             ##
##                                                                  ##
######################################################################
######################################################################

######################################################################
# build eventing client - for custom projects this must be modified  #
######################################################################
SET(TemperatureEvent_client_SRCS
  TemperatureEvent_client.c
  ${GSOAP_STDSOAP2_SOURCE}
  ${gen_DIR}/acs_inv1ServerLib.c)

ADD_EXECUTABLE(acs_TemperatureEvent_client
  ${TemperatureEvent_client_SRCS})

EXTEND_TGT_COMPILE_FLAGS(acs_TemperatureEvent_client
  FLAGS "-DDPWS_CLIENT")

TARGET_LINK_LIBRARIES(acs_TemperatureEvent_client
  ${DPWS_LIBRARIES}
  ${DPWS-C_LIBRARIES})
\endverbatim

In case of the advanced GetStatus client executable, following source files are
required:
 - TemperatureEvent_client.c: the TemperatureEvent client implementation
 - ${GSOAP_STDSOAP2_SOURCE}: the gSOAP runtime
 - ${gen_DIR}/acs_inv1ServerLib.c: the server side code generated by soapcpp2 for the Airconditioner service event sink

Now we can create an minimal c code for an application and save it under
src/TemperatureEvent_client.c:
\code
int main()
{
  return 0;
}
\endcode

The project tree should look like this:

\verbatim
[Project Main Directory]
 |-- Build
 |-- CMakeLists.txt
 |-- buildversion
 |-- cmake-modules
 |   |-- FindDPWS-C.cmake
 |   |-- FindDPWS-D.cmake
 |   |-- FindDPWS.cmake
 |   |-- FindGSOAP.cmake
 |   |-- FindGnuPatch.cmake
 |   |-- FindGnuSed.cmake
 |   `-- FindXSLTproc.cmake
 `-- src
     |-- advanced_GetStatus_client.c
     |-- AirConditioner.wsdl
     |-- AirConditioner.wsdl_inv
     |-- CMakeLists.txt
     |-- acs.gsoap
     |-- acs_inv.gsoap
     |-- advanced_airconditioner.c
     |-- config.h.in
     |-- event_worker.c
     |-- event_worker.h
     |-- GetStatus_client.c
     |-- metadata.xml
     |-- SetTargetTemperature_client.c
     |-- simple_airconditioner.c
     `-- TemperatureEvent_client.c
\endverbatim

To test our project environment we try to build the TemperatureEvent client.
\verbatim
$ cd Build
$ make
-- Found gSOAP in: /media/disk/elmex/EclipseWorkspace/gsoap-2.7.11
-- Found DPWS in /media/disk/elmex/EclipseWorkspace/ws4d-gsoap-devel/Install-Debug
-- Buildversion: 0.1

-- Build type not set, defaulting to DEBUG!
-- Choosing target is: device - none specified
-- Configuring done
-- Generating done
-- Build files have been written to: /media/disk/elmex/EclipseWorkspace/ws4d-gsoap-devel/Install-Debug/share/doc/ws4d-gSOAP/example/build
[ 11%] Built target acs_GetStatus_client
[ 22%] Built target acs_SetTargetTemperature_client
Scanning dependencies of target acs_TemperatureEvent_client
[ 25%] Building C object src/CMakeFiles/acs_TemperatureEvent_client.dir/TemperatureEvent_client.o
Linking C executable ../bin/acs_TemperatureEvent_client
CMakeFiles/acs_TemperatureEvent_client.dir/gen/acs_inv1ServerLib.o: In function `soap_serve___acsinv1__TemperatureEvent':
/media/disk/elmex/EclipseWorkspace/ws4d-gsoap-devel/Install-Debug/share/doc/ws4d-gSOAP/example/build/src/gen/acs_inv1Server.c:95: undefined reference to `__acsinv1__TemperatureEvent'
collect2: ld gab 1 als Ende-Status zurück
make[2]: *** [bin/acs_TemperatureEvent_client] Fehler 1
make[1]: *** [src/CMakeFiles/acs_TemperatureEvent_client.dir/all] Fehler 2
make: *** [all] Fehler 2
\endverbatim

This wont work because we have to implement the function
__acsinv1__TemperatureEvent() to receive TemperatureEvent events.

\subsection tutImplTemperatureEventClient Implementing a client using the TemperatureEvent event

The next step is to implement the client using the TemperatureEvent event of
the AirConditioner service on the AirConditioner device. Therefor the following
code must be added to src/TemperatureEvent_client.c:

First we have to specify the include files:

\code
#include "acs_inv1.nsmap"
#include "dpws_client.h"
#include <signal.h>

#include "ws4d_eprllist.h"
\endcode

We declare the soap and dpws handles globally.


\code
struct soap client, handler;
struct dpws_s dpws;
int shutdownFlag = 0;
\endcode

We need more soap clients this time. One soap handle to subscribe for the event
at the device and one for the handler that receives the TemperatureEvent events
from the device. We define the shutdownFlag variable global as we use a signal
to terminate the client.

This is the termination function client_exit() to initate a proper shut down of
the client. It unsubscribes from the service and frees all allocated memory.

\code
void
client_exit ()
{
  printf ("\nTemperatureEventListener: waiting for mainloop to shut down...\n");
  fflush(NULL);
  shutdownFlag = 1;
}
\endcode

Now we can start the main function. The first part of the main
function is the declaration of variables.

\code
int
main (int argc, char **argv)
{
#ifndef WIN32
  struct sigaction sa;
#endif

  ws4d_alloc_list alist;

  struct ws4d_epr device, *service = NULL;
  struct ws4d_abs_eprlist services;
  ws4d_qnamelist service_type_list;

  char *deviceaddr = NULL, *host = NULL, *XAddrs = NULL;
  const char *id = NULL;

  char handler_uri[DPWS_URI_MAX_LEN + 1] = "http://host:0/";

  struct ws4d_delivery_type *delivery = NULL;
  struct ws4d_filter_type *filter = NULL;

  ws4d_time duration = 3600;
\endcode

 - alist: allocation list to manage memory allocations

 - device, service: end point references for the device and the service to use

 - services: endpoint reference list to be uses for service discovery later on

 - service_type_list: a qname list used for finding the wanted service on the device

 - deviceaddr, host, XAddrs: miscellaneous variables

 - id: variable to store the subscription id

 - handler_uri: variable to specify and store the uri of the event handler

 - delivery: variable to specify the delivery mode

 - filter: variable to specify the event filter

 - duration: variable to specify and store the duration of the subscription

The next part is code for parsing the command line. We expect one
parameter -h to specify the host device, one parameter with the
logical device address (uuid) of the device we want to use.

\code
  /* parsing command line options */
  while (argc > 1)
    {
      if (argv[1][0] == '-')
        {
          char *option = &argv[1][1];
          switch (option[0])
            {
            case 'i': /* set interface with option -i */
              if (strlen (option) > 2)
                {
                  ++option;
                  host = option;
                }
              else
                {
                  --argc;
                  ++argv;
                  host = argv[1];
                }
#ifdef DEBUG
              printf ("\nTemperatureEvent_client: Set interface to \"%s\"\n", host);
#endif
              break;
            case 'd': /* set device address with option -d */
              if (strlen (option) > 2)
                {
                  ++option;
                  deviceaddr = option;
                }
              else
                {
                  --argc;
                  ++argv;
                  deviceaddr = argv[1];
                }
#ifdef DEBUG
              printf ("\nTemperatureEvent_client: Set device address to \"%s\"\n",
                      deviceaddr);
#endif
              break;
            default:
              fprintf (stderr, "\nTemperatureEvent_client: Bad option %s", argv[1]);
              printf ("\n%s -i [interface address] -d [device address]\n",
                      argv[0]);
              exit (1);
            }
        }
      --argc;
      ++argv;
    }

  if (deviceaddr == NULL)
    {
      fprintf (stderr,
               "\nTemperatureEvent_client: No device id or address was specified!\n");
      exit (1);
    }

  if (host == NULL)
    {
      fprintf (stderr,
               "\nTemperatureEvent_client: No interface address was specified!\n");
      exit (1);
    }
\endcode

After parsing the command line, the allocation list is initialized.

\code
  WS4D_ALLOCLIST_INIT (&alist);
\endcode

Now the soap handles can be initialized.

\code
  /* initialize client soap handle */
  soap_init (&client);
#ifdef DEBUG
  soap_omode (&client, SOAP_XML_INDENT);
#endif
  soap_set_namespaces (&client, acs_inv1_namespaces);

  soap_init (&handler);
#ifdef DEBUG
  soap_omode (&handler, SOAP_XML_INDENT);
#endif
  soap_set_namespaces (&handler, acs_inv1_namespaces);
\endcode

After initialization of the soap handle, the dpws handle must be
initialized.

\code
  /* initialize WS4D-gSOAP */
  if (dpws_init (&dpws, host) != SOAP_OK)
    {
      fprintf (stderr,
               "\nTemperatureEventListener: Could not initialize dpws handle\n");
      dpws_done (&dpws);
      exit (1);
    }
\endcode

Now we have to bind the handler soap handel to an address. Address
"http://host:0/" means: same host as dpws handle and random port.

\code
  /* bind listener handle to an address */
  dpws_handle_init (&dpws, &handler);
  if (dpws_handle_bind (&dpws, &handler, handler_uri, DPWS_URI_MAX_LEN, 100)
      == SOAP_INVALID_SOCKET)
    {
      fprintf (stderr,
               "\nTemperatureEventListener: error calling dpws_handle_bind");
      exit (1);
    }
\endcode

When the function dpws_handle_bind successfully returns the actual URI of the
soap handle is written to handler_uri.

To resolve the logical adress of a device passed as a command line parameter, we
have to allocate an epr structure and assigned the logical address.
\code
  /* allocate and prepare device to resolve */
  ws4d_epr_init (&device);
  ws4d_epr_set_Addrs (&device, deviceaddr);
\endcode

Then we can pass this epr to the dpws_resolve_addr() function. The
function resolves a logical device address (uuid form) into a http address.

\code
  /* resolve address */
  XAddrs = (char *) dpws_resolve_addr (&dpws, &device, NULL, 10000);
  if (XAddrs != NULL)
    {
#ifdef DEBUG
      printf ("\nTemperatureEventListener: Device %s found at addr %s\n",
              ws4d_epr_get_Addrs (&device), XAddrs);
#endif
    }
  else
    {
      fprintf (stderr,
               "\nTemperatureEventListener: Device %s cannot be found\n",
               ws4d_epr_get_Addrs (&device));
      exit (1);
    }
\endcode

After resolving the logical address we have to search for the address of
the service on the device we want to use. The services can be filtered
by service type or by service id. In this case we initialize a
service type list with the service type of the AirConditioner service
AirConditionerInterface.

\code
  /* prepare service type list */
  ws4d_qnamelist_init (&service_type_list);
  ws4d_qnamelist_addstring
    ("\"http://www.ws4d.org/axis2/tutorial/AirConditioner\":AirConditionerInterface",
     &service_type_list, &alist);
\endcode

Then we pass the service type list and the resolved device epr to the
dpws_find_services() function to get the epr of the service we want to
use.

\code
  /* look up service with matching service types on device */
  ws4d_eprlist_init (&services, ws4d_eprllist_init, NULL);
  if (dpws_find_services (&dpws, &device, &service_type_list,
                          10000, &services) == SOAP_OK)
    {
      /* use first service */
      service = ws4d_eprlist_get_first (&services);
#ifdef DEBUG
      printf
        ("\nTemperatureEventListener: Device offers AirConditionerService at %s\n",
         ws4d_epr_get_Addrs (service));
#endif
    }
  else
    {
      fprintf (stderr,
               "\nTemperatureEventListener: AirConditionerService not found on %s\n",
               ws4d_epr_get_Addrs (&device));
      exit (1);
    }
\endcode

If we have the service epr and the thus its address, we can prepare the
subscription for the TemperatureEvent event.

First we have to define a delivery mode. The push delivery mode can be
initialized with the function dpws_gen_delivery_push(). With this delivery mode
the device will send the event to our event handler inizialized before.

\code
  /* prepare filter and delivery mode */
  delivery =
    dpws_gen_delivery_push (&client, dpws_handle_get_paddr (&handler));
\endcode

Then we have to define an event filter. DPWS defines the action based filter
that can be used to filter the events based on the actions defined in the WSDL.
In this case we want to receive all events with the action "http://www.ws4d.org/axis2/tutorial/AirConditioner/TemperatureEventOut".

\code
  filter =
    dpws_gen_filter_action (&client,
                            "http://www.ws4d.org/axis2/tutorial/AirConditioner/TemperatureEventOut");
\endcode

And finally we call the subcribe operation of the airconditioner service with
the WS4D-gSOAP API function dpws_subscribe().

\code
  /* Subscribing to Airconditioner Service */
#ifdef DEBUG
  printf
    ("\nTemperatureEventListener: Subscribe to Airconditioner Service ... ");
#endif
  id = dpws_subscribe (&client, service, NULL, &duration, delivery, filter);
  if (!id)
    {
      soap_print_fault (&client, stderr);
      exit (1);
    }

#ifdef DEBUG
  printf ("OK (%s %s)\n", ws4d_subsproxy_getsubsman (service, id), id);
#endif
\endcode

If the subscription was successful its id is stored in the variable id. We need
this id as parameter for operations at the subscription manager service on the
device. There we can get the status of the subscription, renew it or unsubscribe
it as shown later.

Now we begin with the initialization of a typlical message processing loop. In
this case not for a device or service but for the event message handler.

First we register the signal handler client_exit() for the signal SIGINT.

\code
  /* install signal handler for SIGINT or Ctrl-C */
#ifdef WIN32
  signal (SIGINT, client_exit);
#else
  memset (&sa, 0, sizeof (sa));
  sa.sa_handler = client_exit;
  sigaction (SIGINT, &sa, NULL);
#endif
\endcode

Now we define a timeout for the soap_accept() function that it will not block
forever.

\code
  /* set accept timeout to one second */
  handler.accept_timeout = 1;

#ifdef DEBUG
  printf
    ("\nTemperatureEventListener: ready to receive events ... (Ctrl-C for shut down)\n");
#endif
\endcode

The message processing loop of the event handler looks like this:

\code
  while(!shutdownFlag)
    {

#ifdef DEBUG
      printf ("\nTemperatureEventListener: waiting for events\n");
#endif

      if (soap_accept (&handler) != SOAP_INVALID_SOCKET)
        {

#ifdef DEBUG
          printf ("\nTemperatureEventListener: processing event from %s:%d\n",
                  inet_ntoa (handler.peer.sin_addr),
                  ntohs (handler.peer.sin_port));
#endif

          if (acs_inv1_serve (&handler))
            {
              soap_print_fault (&handler, stderr);
            }

          /* clean up soaps internaly allocated memory */
          soap_end (&handler);
        }
    }
\endcode

This is a typical gsoap message processing loop with a timeout for soap_accept.
The timeout is one second. So this loop checks the shutdownFlag. If it is set to
1 by the signal handler client_exit() the code exits thhe processing loop and
continue with a shut down. As long as the shutdownFlag is equal to 0 the
processing waits for incoming events. If a event is received the
__acsinv1__TemperatureEvent() function to process the event data is called
inside of acs_inv1_serve(). As the event processing is done in a separate
function we first have to write the code for the shut down:

\code
  printf ("\nTemperatureEventListener: shutting down...");
\endcode

We first unsubscribe the subscription at the service with the subscription id.

\code
  if (dpws_subs_unsubscribe (&client, service, id) != SOAP_OK)
    {
      soap_print_fault (&client, stderr);
      exit (1);
    }
\endcode

Now we terminate all handles and free all memory.

\code
  soap_end (&client);
  soap_done (&client);
  soap_end (&handler);
  soap_done (&handler);
  dpws_done (&dpws);

  ws4d_eprlist_done (&services);
  ws4d_epr_reset (&device);

  ws4d_alloclist_done (&alist);

  printf ("done\n");

  exit (0);
}
\endcode

Now we have to implement the __acsinv1__TemperatureEvent() function to process
received events. This is similar to an implementation of a operation of a
service.

\code
int
__acsinv1__TemperatureEvent (struct soap *soap,
                             struct acsinv1__ACStateType *acsinv1__ACState)
{
  printf ("\nGot TemperatureEvent: ");
  if (acsinv1__ACState)
    {
      printf ("\n\tCurrentTemp: %d\n\tTargetTemp: %d\n",
              acsinv1__ACState->CurrentTemp, acsinv1__ACState->TargetTemp);
    }
  else
    {
      printf ("no ACState transmitted!\n");
    }

  return SOAP_OK;
}
\endcode

\subsection tutCompTemperatureEventClient Build the TemperatureEvent client

In the Build directory you call make.

\verbatim
$ cd Build
$ make
[ 11%] Built target acs_GetStatus_client
[ 22%] Built target acs_SetTargetTemperature_client
Scanning dependencies of target acs_TemperatureEvent_client
[ 25%] Building C object src/CMakeFiles/acs_TemperatureEvent_client.dir/TemperatureEvent_client.o
Linking C executable ../bin/acs_TemperatureEvent_client
[ 34%] Built target acs_TemperatureEvent_client
[ 45%] Built target acs_advanced_GetStatus_client
[ 77%] Built target acs_advanced_device
[100%] Built target acs_simple_device
\endverbatim

\subsection tutRunTemperatureEventClient Running the TemperatureEvent client

To start the TemperatureEvent client we have to pass two arguments.

 - IP Address of the interface to operate on with the -i option
 - Device address in form "urn:uuid:[id as uuid]" or "http://" with the -d option

\verbatim
$ cd Build/bin
$ ./acs_TemperatureEvent_client -i 192.168.1.100 -d urn:uuid:31af0499-560d-4af5-8c04-56d1a4faf75d

TemperatureEvent_client: Set interface to "192.168.1.100"

TemperatureEvent_client: Set device address to "urn:uuid:31af0499-560d-4af5-8c04-56d1a4faf75d"

TemperatureEventListener: Device urn:uuid:31af0499-560d-4af5-8c04-56d1a4faf75d found at addr http://192.168.1.100:34928/31af0499-560d-4af5-8c04-56d1a4faf75d

TemperatureEventListener: Device offers AirConditionerService at http://192.168.1.100:34928/

TemperatureEventListener: Subscribe to Airconditioner Service ... OK (http://192.168.1.100:34928/31af0499-560d-4af5-8c04-56d1a4faf75d urn:uuid:971371ba-6cd7-47ce-8bfc-e017d893904e)

TemperatureEventListener: ready to receive events ... (Ctrl-C for shut down)

TemperatureEventListener: waiting for events

TemperatureEventListener: waiting for events

TemperatureEventListener: waiting for events

[...]
\endverbatim

The client now waits for an event. The best way to trigger an event is to change
the target temperature of the airconditioner service. The service will send an
event if the current temperature has reached the target temperature. We do this
with the SetTargetTemperature client:

\verbatim
$ cd Build/bin
$ ./acs_SetTargetTemperature_client -i 192.168.1.100 -d urn:uuid:31af0499-560d-4af5-8c04-56d1a4faf75d -t 20
simple_client: Set interface to "192.168.1.100"

simple_client: Set device address to "urn:uuid:31af0499-560d-4af5-8c04-56d1a4faf75d"

simple_client: Set target temperature to "20"

simple_client: device urn:uuid:31af0499-560d-4af5-8c04-56d1a4faf75d found at addr http://192.168.1.100:34928/31af0499-560d-4af5-8c04-56d1a4faf75d

simple_client: device offers AirConditionerService at http://192.168.1.100:34928/

simple_client: Successfully set TargetTemperature on http://192.168.1.100:34928/

CurrentTemp: 24
TargetTemp: 20
\endverbatim

After some time when the current temperature of the airconditioner service has
reached the target temperature you will receive a event and the output of the
TemperatureEvent client will look like this:

\verbatim
[...]

TemperatureEventListener: waiting for events

TemperatureEventListener: processing event from 192.168.1.100:46193

Got TemperatureEvent:
  CurrentTemp: 19
  TargetTemp: 20

TemperatureEventListener: waiting for events

TemperatureEventListener: waiting for events

TemperatureEvent_client: waiting for mainloop to shut down...

TemperatureEventListener: shutting down...done
\endverbatim

If you press ctrl-c the client will shut down.

\section compile_tutorial_page Compile the included Airconditioner Tutorial

A ready to compile version of the Airconditioner Tutorial is included
in the WS4D-gSOAP installation. You can use this sample project as guideline
for your own projects. The sample project is called airconditioner example and
further described in the text section . You can find the project files in
your installation directory under share/doc/ws4d-gsoap/example.

The airconditioner project also uses CMake. The configuration of the
airconditioner project is similar to the configuration of the
WS4D-gSOAP Toolkit.

A typical out-of-source configuration for the airconditioner project
might look like this:

\verbatim
$ mkdir airconditioner-build
$ cd airconditioner-build
$ cmake [path to WS4D-gSOAP installation]/share/doc/ws4d-gsoap/example
\endverbatim

The project has a built in default configuration to use the WS4D-gSOAP
installation it is part of and the gSOAP installation that was used to
compile WS4D-gSOAP. You can also specifiy other installations of gSOAP
and WS4D-gSOAP with the GSOAP_PATH and DPWS_PATH variables. These
varaibles can either be specified as environment variables or as
arguments to cmake.

\verbatim
$ cmake \
  -DGSOAP_PATH:PATH=[absolute path to gsoap installation] \
  -DDPWS_PATH:PATH=[absolute path to WS4D-gSOAP installation] \
  [path to WS4D-gSOAP installation]/share/doc/ws4d-gsoap/example
\endverbatim

You should always use the same gSOAP installation that was also used
to compile WS4D-gSOAP!!! WS4D-gSOAP relies on code generated by gSOAP
and won't work if you use another gSOAP version than the version you
used to compile WS4D-gSOAP!!!

A typical out-of-source build for the airconditioner project might
look like this:

\verbatim
$ cd airconditioner-build
$ make
\endverbatim

You can find the executables of the project under airconditioner-build/bin.

*/
